<?php declare(strict_types = 1);

namespace Mall\Model\Services;

use Core\Model\Helpers\BaseEntityService;
use EshopCatalog\Model\Entities\CategoryProduct;
use EshopCatalog\Model\Entities\ProductInSite;
use Mall\Model\Entities\ProductMap;

/**
 * @method ProductMap get($id)
 * @method ProductMap[] getAll()
 * @method ProductMap getReference($id)
 */
class ProductMapService extends BaseEntityService
{
	protected $entityClass = ProductMap::class;

	protected ?array $cMap            = null;
	protected ?array $cMapByMallId    = null;
	protected ?array $cMapByIdForMall = null;

	public function addToMap(string $country, int $productId, int $mallId, string $idForMall): bool
	{
		try {
			$this->em->getConnection()->executeQuery("INSERT IGNORE INTO mall__product_map (product_id, country, mall_id, id_for_mall, is_active) VALUES (?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE mall_id = ?, id_for_mall = ?",
				[$productId, $country, $mallId, $idForMall, 1, $mallId, $idForMall]);

			if ($this->cMap !== null)
				$this->cMap[$country][$productId] = $mallId;

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	public function getByMallId(string $country, $mallId): ?ProductMap
	{
		return $this->getEr()->createQueryBuilder('mp')
			->where('mp.mallId = :id')
			->andWhere('mp.country = :country')
			->setParameters([
				'id'      => $mallId,
				'country' => $country,
			])
			->getQuery()->getOneOrNullResult();
	}

	public function setMallId(string $country, int $productId, int $mallId, string $idForMall)
	{
		$this->em->getConnection()->executeStatement("UPDATE mall__product_map SET mall_id = ?, id_for_mall = ? WHERE country = ? AND product_id = ?",
			[$mallId, $idForMall, $country, $productId]);

		if ($this->cMap !== null)
			$this->cMap[$country][$productId] = $mallId;
	}

	public function getMap(): array
	{
		if ($this->cMap === null) {
			$this->cMap = [];

			foreach ($this->getEr()->createQueryBuilder('pm')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cMap[$row['country']][$row['product_id']] = $row;
			}
		}

		return $this->cMap;
	}

	public function getMapByMallId(): array
	{
		if ($this->cMapByMallId === null) {
			$this->cMapByMallId = [];
			foreach ($this->getMap() as $country => $rows) {
				foreach ($rows as $productId => $row) {
					$this->cMapByMallId[$country][$row['mallId']] = $row;
				}
			}
		}

		return $this->cMapByMallId;
	}

	public function getMapByIdForMall(): array
	{
		if ($this->cMapByIdForMall === null) {
			$this->cMapByIdForMall = [];
			foreach ($this->getMap() as $country => $rows) {
				foreach ($rows as $productId => $row) {
					$this->cMapByIdForMall[$country][$row['idForMall']] = $row;
				}
			}
		}

		return $this->cMapByIdForMall;
	}

	/**
	 * @param int  $id
	 * @param bool $asEntity
	 *
	 * @return array|ProductMap[]
	 */
	public function getByEshopProduct(int $id, bool $asEntity = false): array
	{
		$data = [];

		$qb = $this->getEr()->createQueryBuilder('pm')
			->where('pm.eshopProduct = :id')
			->setParameter('id', $id);

		if ($asEntity) {
			foreach ($qb->getQuery()->getResult() as $row)
				$data[$row->country] = $row;
		} else {
			foreach ($qb->getQuery()->getArrayResult() as $row)
				$data[$row['country']] = $row;
		}

		return $data;
	}

	public function changeActive(array $prodIds, string $country, int $isActive): bool
	{
		try {
			$this->getEr()->createQueryBuilder('pm')->update()
				->set('pm.isActive', $isActive)
				->where('pm.eshopProduct IN (:ids)')
				->andWhere('pm.country = :country')
				->setParameters([
					'ids'     => $prodIds,
					'country' => $country,
				])->getQuery()->execute();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	public function findBaseCategories(array $productIds): array
	{
		$arr = [];

		foreach ($this->em->getRepository(ProductInSite::class)->createQueryBuilder('pis')
			         ->select('IDENTITY(pis.product) as product, IDENTITY(pis.category) as category')
			         ->where('pis.product IN (:ids)')
			         ->andWhere('pis.isActive = 1')
			         ->setParameters([
				         'ids' => $productIds,
			         ])
			         ->getQuery()->getArrayResult() as $row) {
			$arr[$row['product']][] = $row['category'];
		}

		foreach ($this->em->getRepository(CategoryProduct::class)->createQueryBuilder('cp')
			         ->select('IDENTITY(cp.category) as category, IDENTITY(cp.product) as product')
			         ->where('cp.product IN (:ids)')
			         ->setParameters([
				         'ids' => $productIds,
			         ])
			         ->getQuery()->getArrayResult() as $row) {
			$arr[$row['product']][] = $row['category'];
		}

		return $arr;
	}

	public function clearClassCache(): void
	{
		$this->cMap         = null;
		$this->cMapByMallId = null;
	}
}
