<?php declare(strict_types = 1);

namespace Mall\Model\Subscribers;

use Core\Model\Event\Event;
use EshopOrders\AdminModule\Model\Dao\OrderExpedition;
use EshopOrders\AdminModule\Model\Orders;
use Mall\Model\Entities\MallOrder;
use Mall\Model\Services\MallApiOrders;
use Mall\Model\Services\OrdersService;
use Nette\Utils\FileSystem;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Mall\Model\OrdersExported;

class ExpeditionSubscriber implements EventSubscriberInterface
{
	protected OrdersExported $ordersExported;

	protected MallApiOrders $mallApiOrders;

	protected OrdersService $ordersService;

	public function __construct(OrdersExported $ordersExported, MallApiOrders $mallApiOrders, OrdersService $ordersService)
	{
		$this->ordersExported = $ordersExported;
		$this->mallApiOrders  = $mallApiOrders;
		$this->ordersService  = $ordersService;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			Orders::class . '::processOrdersForExpedition' => 'processOrdersForExpedition',
			Orders::class . '::sendExpeditionData'         => 'sendExpeditionData',
			Orders::class . '::generateLabel'              => 'generateLabel',
			Orders::class . '::afterGenerateLabel'         => 'afterGenerateLabel',
		];
	}

	public function processOrdersForExpedition(Event $event): void
	{
		$orderExpeditions = [];
		$ids              = $this->filterOrders($event->data['data']);

		if ($ids) {
			foreach ($this->ordersExported->getOrdersNotExported($ids) as $id => $mo) {
				$orderExpeditions[$id] = new OrderExpedition($mo->getOrder(), 'mdp', $mo);
			}

			$event->data['expedition'] += $orderExpeditions;
		}
	}

	public function sendExpeditionData(Event $event): void
	{
		$ids = $this->filterOrders($event->data['data']);

		if ($ids) {
			$orders = $this->ordersExported->getOrdersNotExported($ids);

			foreach ($orders as $mallOrder) {
				$result = $this->mallApiOrders->setShipping($mallOrder->mallClient, $mallOrder->getMallId());

				if ($result === true) {
					$event->data['okCount']++;
					$this->ordersExported->setLastStatus([$mallOrder->getOrder()->getId()], MallOrder::STATUS_SHIPPING);
				} else
					$event->data['errorCount']--;
			}
		}
	}

	public function generateLabel(Event $event): void
	{
		$ids = $this->filterOrders($event->data['data']);

		if ($ids) {
			$orders = $this->ordersExported->getOrdersNotExported($ids);

			foreach ($orders as $mallOrder) {
				$result = $this->mallApiOrders->generateLabelPdf($mallOrder->mallClient, $mallOrder->getMallId());

				if ($result !== null) {
					FileSystem::createDir(TMP_DIR . '/mallLabels');
					if (file_put_contents(TMP_DIR . '/mallLabels/stitky.pdf', base64_decode($result))) {
						$event->data['files'][] = TMP_DIR . '/mallLabels/stitky.pdf';
						$event->data['okCount']++;
					} else {
						$event->data['errorCount']++;
					}
				} else {
					$event->data['errorCount']++;
				}
			}
		}
	}

	public function afterGenerateLabel(Event $event): void
	{
		$ids = $this->filterOrders($event->data['data']);

		if ($ids) {
			$orders = $this->ordersExported->getOrdersNotExported($ids);

			foreach ($orders as $mallOrder) {
				$spedition      = $mallOrder->getOrder()->getSpedition();
				$trackingNumber = $spedition->trackingNumber ?? null;
				$trackingUrl    = $spedition->trackingUrl ?? null;

				$result = $this->mallApiOrders->setShipped($mallOrder->mallClient, $mallOrder->getMallId(), $trackingNumber, $trackingUrl);

				if ($result === true) {
					$event->data['okCount']++;
					$this->ordersService->export($mallOrder, $trackingNumber, $trackingUrl);
					$this->ordersExported->setLastStatus([$mallOrder->getOrder()->getId()], MallOrder::STATUS_SHIPPED);
				} else
					$event->data['errorCount']--;
			}
		}
	}

	/**
	 * @param array $data
	 *
	 * @return array
	 */
	protected function filterOrders(array $data): array
	{
		$ids  = [];
		$left = [];
		foreach ($data as $v) {
			if ($v['service'] === 'mdp') {
				$ids[] = $v['id'];
			} else {
				$left[] = $v['id'];
			}
		}

		if (!empty($left)) {
			$mallOrders = array_flip($this->ordersService->getExisting());

			foreach ($left as $id) {
				if (isset($mallOrders[$id])) {
					$ids[] = $id;
				}
			}
		}

		return $ids;
	}
}
