<?php declare(strict_types = 1);

namespace Mall\Model\Subscribers;

use Mall\Model\Sync\SyncProduct;
use Core\Model\Event\Event;
use EshopOrders\AdminModule\Model\Statuses;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Event\OrderStatusEvent;
use Mall\Model\Entities\MallOrder;
use Mall\Model\MallClients;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;
use Mall\Model\Services\MallApiOrders;
use Mall\Model\OrdersExported;
use Mall\Model\Services\OrdersService as MallOrdersService;

class OrderSubscriber implements EventSubscriberInterface
{
	protected Statuses $statuses;

	protected OrdersExported $ordersExported;

	protected MallApiOrders $mallApiOrders;

	protected MallClients $mallClients;

	protected MallOrdersService $mallOrdersService;

	protected SyncProduct $syncProducts;

	public function __construct(Statuses    $statuses, OrdersExported $ordersExported, MallApiOrders $mallApiOrders,
	                            MallClients $mallClients, MallOrdersService $mallOrdersService, SyncProduct $syncProducts)
	{
		$this->statuses          = $statuses;
		$this->ordersExported    = $ordersExported;
		$this->mallApiOrders     = $mallApiOrders;
		$this->mallClients       = $mallClients;
		$this->mallOrdersService = $mallOrdersService;
		$this->syncProducts      = $syncProducts;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orders.checkCompleted' => 'checkCompleted',
			Order::class . '::changeStatus'     => 'orderChangeStatus',
			'eshopOrders.admin.orderOnSuccess'  => ['orderOnSuccess', -100],
		];
	}

	public function checkCompleted(Event $event)
	{
		$orders = $this->ordersExported->getOrdersNotCompleted();

		$completed = [];
		$others    = [];
		foreach ($orders as $id => $uo) {
			if (!$uo->country)
				continue;
			$detail = $this->mallApiOrders->getDetail($this->mallClients->getClient($uo->country), (int) $uo->getMallId());

			if ($detail) {
				switch ($detail['status']) {
					case MallOrder::STATUS_CANCELED:
					case MallOrder::STATUS_RETURNED:
					case MallOrder::STATUS_LOST:
					case MallOrder::STATUS_STORNO:
						$others[$detail['status']][] = $id;
						break;
					case MallOrder::STATUS_DELIVERED:
						$completed[$id] = $detail;
						break;
				}
			}
		}

		if (!empty($others)) {
			foreach ($others as $status => $ids) {
				try {
					$result = $this->statuses->changeStatus($ids, OrderStatus::STATUS_CANCELED);

					if ($result)
						$this->ordersExported->setLastStatus($ids, $status);
				} catch (\Exception $e) {
					Debugger::log('Mall error - ' . $status . ' - ' . implode(', ', $ids), 'ordersCheckCanceled');
					Debugger::log('Mall error - ' . $e->getMessage(), 'ordersCheckCanceled');
					Debugger::log($e);
				}
			}
		}

		if (!empty($completed)) {
			try {
				$completedIds = array_keys($completed);
				$result       = $this->statuses->changeStatus($completedIds, OrderStatus::STATUS_FINISHED);

				if ($result) {
					$this->ordersExported->setLastStatus($completedIds, MallOrder::STATUS_DELIVERED);

					foreach ($completed as $id => $detail) {
						if ($detail['delivered_at'] ?? null) {
							continue;
						}

						$uo = $orders[$id];
						$this->mallApiOrders->setDelivered($this->mallClients->getClient($uo->country), (string) $uo->getMallId(), (new DateTime()));
					}
				}
			} catch (\Exception $e) {
				Debugger::log('Mall error - ' . $e->getMessage(), 'ordersCheckComplete');
				Debugger::log($e);
			}
		}
	}

	public function orderChangeStatus(OrderStatusEvent $event): void
	{
		if (!$event->order || !$event->order->getId() || !in_array($event->status, [OrderStatus::STATUS_CANCELED,
				OrderStatus::STATUS_FINISHED]))
			return;

		$mallOrder = $this->mallOrdersService->getByEshopId($event->order->getId());

		if (!$mallOrder)
			return;

		if ($event->status === OrderStatus::STATUS_FINISHED) {
			if ($mallOrder->lastStatus !== MallOrder::STATUS_DELIVERED) {
				$this->ordersExported->setLastStatus([$event->order->getId()], MallOrder::STATUS_DELIVERED);
				$this->mallApiOrders->setDelivered($this->mallClients->getClient($mallOrder->country), (string) $mallOrder->getMallId(), (new DateTime()));
			}
		} else if ($event->status === OrderStatus::STATUS_CANCELED) {
			if ($mallOrder->lastStatus !== MallOrder::STATUS_CANCELED) {
				$this->ordersExported->setLastStatus([$event->order->getId()], MallOrder::STATUS_CANCELED);
				$this->mallApiOrders->setCancelled($this->mallClients->getClient($mallOrder->country), (string) $mallOrder->getMallId());

				foreach ($event->order->getOrderItems() as $v) {
					if ($v->getProduct())
						ProductSubscriber::$updateQuantity[] = $v->getProductId();
				}
			}
		}
	}

	public function orderOnSuccess(OrderEvent $event): void
	{
		if (!$event->order || !$event->order->getId())
			return;

		$mallOrder = $this->mallOrdersService->getByEshopId($event->order->getId());
		if (!$mallOrder)
			return;

		foreach ($event->order->getOrderItems() as $v) {
			if ($v->getProduct())
				ProductSubscriber::$updateQuantity[] = $v->getProductId();
		}
	}
}
