<?php declare(strict_types = 1);

namespace Mall\Model\Subscribers;

use Doctrine\Common\EventSubscriber;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Event\OnFlushEventArgs;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use Mall\Model\Services\ProductMapService;
use Mall\Model\Sync\SyncProduct;

class ProductSubscriber implements EventSubscriber
{
	protected SyncProduct $syncProduct;

	protected ProductMapService $productMapService;

	public static array $updateFull     = [];
	public static array $updatePrices   = [];
	public static array $updateQuantity = [];

	public function __construct(SyncProduct $syncProduct, ProductMapService $productMapService)
	{
		$this->syncProduct       = $syncProduct;
		$this->productMapService = $productMapService;
	}

	public function getSubscribedEvents()
	{
		return [
			'onFlush',
			'postUpdate',
			'postRemove',
		];
	}

	public function onFlush(OnFlushEventArgs $args): void
	{
		$uow = $args->getEntityManager()->getUnitOfWork();

		foreach ($uow->getScheduledEntityUpdates() as $object) {
			if ($object instanceof ProductTexts) {
				$changeSet = $uow->getEntityChangeSet($object);

				if (!empty($changeSet))
					self::$updateFull[] = $object->getProduct()->getId();
			} else if ($object instanceof Product) {
				$changeSet = $uow->getEntityChangeSet($object);
				unset($changeSet['modified']);
				unset($changeSet['productTexts']);

				$changeSetCount = count($changeSet);
				if ($changeSetCount === 0)
					continue;

				if ($changeSetCount === 1 && isset($changeSet['price'])) {
					self::$updatePrices[] = $object->getId();
				} else if ($changeSetCount === 1 && isset($changeSet['quantity'])) {
					self::$updateQuantity[] = $object->getId();
				} else {
					self::$updateFull[] = $object->getId();
				}
			}
		}

		foreach ($uow->getScheduledEntityDeletions() as $object) {
			if ($object instanceof Product === false)
				continue;

			if (empty($this->productMapService->getByEshopProduct($object->getId())))
				continue;

			if ($object->isVariant() && !$object->isVariant()->isDefault) {
				$parentId = $args->getEntityManager()->getConnection()->fetchOne("SELECT product_id FROM eshop_catalog__product_variant WHERE is_default = 1 AND variant_id = ?", [$object->isVariant()->getVariantId()]);
				$this->syncProduct->deleteVariant($parentId, $object->getId());
			} else {
				$this->syncProduct->deleteProduct($object->getId());
			}
		}
	}

	public function postUpdate(LifecycleEventArgs $event)
	{
		try {
			$this->checkProductQuantity($event->getObject());
		} catch (\Exception $e) {
		}
	}

	public function postRemove(LifecycleEventArgs $event)
	{
		try {
			$this->checkProductQuantity($event->getObject());
		} catch (\Exception $e) {
		}
	}

	protected function checkProductQuantity(object $entity)
	{
		if ($entity instanceof Product === false)
			return;

		// Pokud je ulozen productForm, tak ignorujeme aktualizaci
		if (isset($_REQUEST['_do']) && $_REQUEST['_do'] === 'productForm-form-submit')
			return;

		if ($entity instanceof Product === false || !$entity->getId() || empty($this->productMapService->getByEshopProduct($entity->getId())))
			return;

		self::$updateQuantity[] = $entity->getId();
	}
}
