<?php declare(strict_types = 1);

namespace Mall\Model\Sync;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use EshopOrders\Model\Entities\OrderItem;
use Mall\Model\Entities\MallOrder;
use Mall\Model\MallClients;
use Mall\Model\OrdersExported;
use Mall\Model\Services\MallApiOrders;
use Mall\Model\Services\OrdersFacade;
use Mall\Model\Services\OrdersService;
use Tracy\Debugger;

class SyncOrders
{
	protected EntityManagerDecorator $em;

	protected MallClients $mallClients;

	protected MallApiOrders $mallApiOrders;

	protected OrdersService $ordersService;

	protected OrdersExported $ordersExported;

	protected OrdersFacade $ordersFacade;

	public function __construct(EntityManagerDecorator $em, MallClients $mallClients, MallApiOrders $mallApiOrders,
	                            OrdersService          $ordersService, OrdersFacade $ordersFacade, OrdersExported $ordersExported)
	{
		$this->em             = $em;
		$this->mallClients    = $mallClients;
		$this->mallApiOrders  = $mallApiOrders;
		$this->ordersService  = $ordersService;
		$this->ordersFacade   = $ordersFacade;
		$this->ordersExported = $ordersExported;
	}

	public function downloadNew(): void
	{
		foreach ($this->mallClients->getClients() as $country => $client) {
			foreach ($this->mallApiOrders->getOpenIds($client) as $mallId) {
				if ($this->ordersService->checkExist($mallId))
					continue;

				$detail = $this->mallApiOrders->getDetail($client, $mallId);

				// Pokud nenalezne detail nebo jde o test tak pokracovat
				if (!$detail)
					continue;

				try {
					$this->ordersFacade->addOrder($client, $detail);
				} catch (\Exception $e) {
					LogNotifier::toDevelopers('Create mall order Error ' . $mallId, 'Create mall order error');
					Debugger::log($mallId . ' - ' . $e->getMessage(), '_mallAddOrderError');
				}
			}
		}
	}

	public function downloadTrackingData(): void
	{
		foreach ($this->mallClients->getClients() as $country => $client) {
			$notTrackedIds = $this->ordersService->getNotTrackedIds();

			foreach ($this->mallApiOrders->getShippedBasic($client) as $row) {
				if (!isset($notTrackedIds[$row['id']]))
					continue;

				$this->ordersService->setTracking((int) $notTrackedIds[$row['id']], $row);
			}
		}
	}

	public function downloadMissingCommissions(): void
	{
		$clients = $this->mallClients->getClients();

		$notFound = [];
		$orders   = [];
		foreach ($this->em->getConnection()->fetchAllAssociative("SELECT order_id, mall_id, country FROM mall__order") as $row) {
			$orders[$row['mall_id']] = $row['order_id'];
			$client                  = $clients[$row['country']];

			$mallOrder = $this->mallApiOrders->getDetail($client, (int) $row['mall_id']);

			foreach ($this->em->getRepository(OrderItem::class)->findBy(['order' => $row['order_id']]) as $item) {
				/** @var OrderItem $item */
				if ($item->getMoreDataValue('commission'))
					continue;

				$updated = false;
				foreach ($mallOrder['items'] as $mallItem) {
					if ($item->getOrderItemText()->getName() == $mallItem['title']) {
						$item->setMoreDataValue('commission', $mallItem['commission']);
						$this->em->persist($item);
						$this->em->flush($item);
						$updated = true;
						break;
					}
				}

				if (!$updated)
					$notFound[$row['mall_id']][] = $item;
			}
		}
	}

	public function fixLastStatus(): void
	{
		$clients = $this->mallClients->getClients();

		foreach ($this->ordersService->getDiffLastStatusFinished() as $row) {
			if (!$row['country'])
				continue;

			$client    = $clients[$row['country']];
			$mallOrder = $this->mallApiOrders->getDetail($client, (int) $row['mallId']);

			if ($mallOrder['status'] == MallOrder::STATUS_DELIVERED) {
				$this->ordersExported->setLastStatus([$row['orderId']], MallOrder::STATUS_DELIVERED);
			}
		}
	}
}
