<?php declare(strict_types = 1);

namespace Mall\Model\Sync;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Strings;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Entities\ProductVariant;
use Mall\Model\Dao\MallClient;
use Mall\Model\Dao\Product\Availability;
use Mall\Model\MallClients;
use Mall\Model\ProductBuilder;
use Mall\Model\Services\CategoriesService;
use Mall\Model\Services\MallApiProducts;
use Mall\Model\Services\ProductMapService;
use Mall\Model\Services\ProductService;
use Mall\Model\Settings;

class SyncProduct
{
	protected MallClients $mallClients;

	protected ProductBuilder $productBuilder;

	protected ProductsFacade $productsFacade;

	protected CategoriesService $categoriesService;

	protected MallApiProducts $mallApiProducts;

	protected ProductMapService $productMapService;

	protected ProductService $productService;

	protected Settings $settings;

	protected EntityManagerDecorator $em;

	protected EventDispatcher $eventDispatcher;

	protected array $cProductsSent = [];

	protected array $cAvailabilitiesSent = [];

	public function __construct(MallClients       $mallClients, ProductBuilder $productBuilder, ProductsFacade $productsFacade,
	                            CategoriesService $categoriesService, MallApiProducts $mallApiProducts, ProductMapService $productMapService,
	                            Settings          $settings, ProductService $productService, EntityManagerDecorator $em,
	                            EventDispatcher   $eventDispatcher)
	{
		$this->mallClients       = $mallClients;
		$this->productBuilder    = $productBuilder;
		$this->productsFacade    = $productsFacade;
		$this->categoriesService = $categoriesService;
		$this->mallApiProducts   = $mallApiProducts;
		$this->productMapService = $productMapService;
		$this->settings          = $settings;
		$this->productService    = $productService;
		$this->em                = $em;
		$this->eventDispatcher   = $eventDispatcher;
	}

	public function processProducts(array $ids)
	{
		ProductQuery::$ignoreProductPublish = true;
		ProductsFacade::$cacheEnabled       = false;
		foreach ($this->mallClients->getClients() as $client) {
			if (!$this->settings->syncEnabled($client->getCountry()))
				continue;

			if (!isset($this->cProductsSent[$client->getCountry()]))
				$this->cProductsSent[$client->getCountry()] = [];

			$this->productsFacade->clearTemp();
			ProductsFacade::$forceLocale     = $client->getLocale();
			Currencies::$currentCodeOverride = $client->currency;
			Currencies::clearConvertedProducts();
			$productBaseCategories = $this->productMapService->findBaseCategories($ids);

			foreach ($this->productsFacade->getProducts($ids) as $product) {
				$tmpProductId = $product->getId();
				if ($product->variantOf && $product->id != $product->variantOf)
					$product = $this->productsFacade->getProduct($product->variantOf);

				if (in_array($product->id, $this->cProductsSent[$client->getCountry()])
					|| !in_array($client->getEshop(), (array) $product->inSites))
					continue;

				$this->cProductsSent[$client->getCountry()][] = $product->id;

				$categoryId = $this->findCategory($client, $product->getId(), $productBaseCategories);
				if (!$categoryId) {
					$categoryId = $this->findCategory($client, $tmpProductId, $productBaseCategories);
				}

				if (!$categoryId)
					continue;

				$this->sendProduct($client, $product, $categoryId);
			}
		}

		ProductsFacade::$forceLocale        = null;
		Currencies::$currentCodeOverride    = null;
		ProductQuery::$ignoreProductPublish = false;
		ProductsFacade::$cacheEnabled       = true;
		Currencies::clearConvertedProducts();
	}

	public function sendForceToken(string $country, $productId, string $forceToken): bool
	{
		$client = $this->mallClients->getClient($country);

		ProductsFacade::$forceLocale     = $client->getLocale();
		Currencies::$currentCodeOverride = $client->currency;
		Currencies::clearConvertedProducts();

		$product = $this->productsFacade->getProduct((int) $productId);

		if (!$product)
			return false;

		$productBaseCategories = $this->productMapService->findBaseCategories([$productId]);

		$categoryId = $this->findCategory($client, $product->getId(), $productBaseCategories);

		if (!$categoryId)
			return false;

		$result = $this->sendProduct($client, $product, $categoryId, $forceToken);

		return $result ? true : false;
	}

	public function sendInactive($productId)
	{
		$mallProductsMap = $this->productMapService->getMap();

		foreach ($this->mallClients->getClients() as $client) {
			$productMap = $mallProductsMap[$client->getCountry()][$productId] ?? null;

			if ($productMap && $productMap['mallId']) {
				$this->sendAvailabilities($client, [
					'id'       => (string) $productId,
					'status'   => Availability::INACTIVE,
					'in_stock' => 0,
				]);
			}
		}
	}

	public function processProductAvailabilities(array $ids)
	{
		if (empty($ids))
			return;

		$mallProductsMap = $this->productMapService->getMap();
		$clients         = $this->mallClients->getClients();
		$productsData    = [];

		foreach ($this->productsFacade->productsService->getEr()->createQueryBuilder('p')
			         ->select('p.id, p.isPublished, p.quantity, av.canAddToCart')
			         ->innerJoin('p.availability', 'av')
			         ->where('p.id IN (' . implode(',', $ids) . ')')
			         ->groupBy('p.id')
			         ->getQuery()->getArrayResult() as $row) {
			$productsData[$row['id']] = $row;
		}

		foreach ($clients as $client) {
			if (!$this->settings->syncEnabled($client->getCountry()))
				continue;

			$event = new Event(['products' => &$productsData]);
			$this->eventDispatcher->dispatch($event, __CLASS__ . '::beforeSendAvailabilitiesProcess');

			$forSend = [];
			foreach ($ids as $id) {
				$productMap = $mallProductsMap[$client->getCountry()][$id] ?? null;
				if (!$productMap)
					continue;

				$productId   = $productMap['idForMall'] ?: $id;
				$productData = $productsData[$id];
				$forSend[]   = [
					'id'       => (string) $productId,
					'status'   => $productData['isPublished'] && $productData['quantity'] > 0 ? Availability::ACTIVE : Availability::INACTIVE,
					'in_stock' => (int) $productData['quantity'],
				];
			}

			if (!empty($forSend)) {
				foreach (array_chunk($forSend, 900) as $chunk) {
					$this->mallApiProducts->sendAvailabilities($client, $chunk);
				}
			}
		}
	}

	public function processProductsPrices(array $ids)
	{
		if (empty($ids))
			return;

		$mallProductsMap = $this->productMapService->getMap();
		$clients         = $this->mallClients->getClients();
		$productsData    = [];
		$variantParents  = [];

		foreach ($this->productsFacade->productsService->getEr()->createQueryBuilder('p')
			         ->select('p.id, p.price, p.retailPrice, pv.variantId')
			         ->leftJoin('p.isVariant', 'pv')
			         ->where('p.id IN (' . implode(',', $ids) . ')')
			         ->getQuery()->getArrayResult() as $row) {
			$productsData[$row['id']] = $row;

			if ($row['variantId'])
				$variantParents[$row['variantId']] = [];
		}

		if (!empty($variantParents)) {
			foreach ($this->em->getRepository(ProductVariant::class)->createQueryBuilder('pv')
				         ->select('IDENTITY(pv.product) as productId, pv.variantId')
				         ->where('pv.isDefault = 1')
				         ->andWhere('pv.variantId IN (' . implode(',', array_keys($variantParents)) . ')')
				         ->getQuery()->getArrayResult() as $row)
				$variantParents[$row['variantId']] = (int) $row['productId'];
		}

		foreach ($clients as $client) {
			if (!$this->settings->syncEnabled($client->getCountry()))
				continue;

			$event = new Event(['products' => &$productsData]);
			$this->eventDispatcher->dispatch($event, __CLASS__ . '::beforeSendPricesProcess');

			foreach ($ids as $id) {
				$productMap = $mallProductsMap[$client->getCountry()][$id] ?? null;
				if (!$productMap)
					continue;

				$productData = $productsData[$id];

				$price       = (float) ($productMap['price'] ?: $productData['price']);
				$retailPrice = $productData['retailPrice'] && $productData['retailPrice'] > $price ? (float) ($productData['retailPrice']) : null;
				$productId   = $productMap['idForMall'] ?: $id;

				if ($productData['variantId']) {
					$variantParent    = $variantParents[$productData['variantId']] ?? null;
					$variantParentMap = $mallProductsMap[$variantParent] ?? null;
					if (!$variantParentMap)
						continue;

					$variantParentId = $variantParentMap['idForMall'] ?: $variantParent;

					$this->mallApiProducts->sendVariantPrice($client, (string) $variantParentId, (string) $productId, $price, $retailPrice);
				} else {
					try {
						$this->mallApiProducts->sendPrice($client, (string) $productId, $price, $retailPrice);
					} catch (\Exception $e) {
					}
				}
			}
		}
	}

	protected function sendAvailabilities(MallClient $client, array $data)
	{
		$this->mallApiProducts->sendAvailabilities($client, $data);
	}

	protected function findCategory(MallClient $client, $productId, array $productBaseCategories): ?array
	{
		$categoryMap = $this->categoriesService->getAllCategoriesByEshopCategory();

		foreach ($productBaseCategories[$productId] ?? [] as $cat) {
			if (isset($categoryMap[$cat][$client->getCountry()])) {
				return $categoryMap[$cat][$client->getCountry()];
			}
		}

		return null;
	}

	protected function sendProduct(MallClient $client, Product $product, $categoryId, ?string $forceToken = null)
	{
		$mallProductsMap         = $this->productMapService->getMap();
		$mallProductsMapByMallId = $this->productMapService->getMapByMallId();
		$productMap              = $mallProductsMap[$client->getCountry()][$product->getId()] ?? null;
		$mallProduct             = $this->productBuilder->build($client, $product, $categoryId['mallId'], $productMap);
		$isUpdate                = $productMap && $productMap['mallId'] ? true : false;

		if ((!$productMap || !$productMap['mallId']) && isset($this->productMapService->getMapByIdForMall()[$client->getCountry()][$mallProduct->id]))
			$mallProduct->id = 'p' . $mallProduct->id;

		if (!$isUpdate && $mallProduct->variants) {
			foreach ($mallProduct->variants as $k => $variant) {
				if ($variant->availability->status == Availability::INACTIVE)
					unset($mallProduct->variants[$k]);
			}
		}

		// Disabled manufacturer
		$disabledManufacturers = $this->settings->getDisabledManufacturers($client->getCountry()) ?? [];
		if (in_array($product->getManufacturer()->id, $disabledManufacturers) || !$product->isActive) {
			if ($isUpdate) {
				$avs = [
					'id'       => (string) $mallProduct->id,
					'status'   => Availability::INACTIVE,
					'in_stock' => 0,
				];

				foreach ($mallProduct->variants as $variant) {
					$avs[] = [
						'id'       => (string) $variant->id,
						'status'   => Availability::INACTIVE,
						'in_stock' => 0,
					];
				}

				$this->mallApiProducts->sendAvailabilities($client, $avs);
			}

			return null;
		}

		// Kontrola jestli se může odeslat pokud je nový produkt
		if (!$isUpdate && $mallProduct->availability->status !== Availability::ACTIVE)
			return null;

		$result = $this->mallApiProducts->sendProduct($client, $mallProduct, $isUpdate, $forceToken);

		if ($result) {
			if (!$productMap) {
				$this->productMapService->addToMap($client->getCountry(), $product->getId(), (int) $result['article_id'], (string) $result['id']);
			} else if (!$productMap['mallId']) {
				$this->productMapService->setMallId($client->getCountry(), $product->getId(), (int) $result['article_id'], (string) $result['id']);
			}

			if (isset($result['variants'])) {
				$baseId  = null;
				$prodIds = [];
				foreach ($mallProduct->variants as $variant) {
					$prodIds[$variant->id] = $variant->eshopId;
					if ($variant->isBase)
						$baseId = $variant->id;
				}

				foreach ($result['variants'] as $variant) {
					if ($variant['isBase'] === true || $variant['id'] == $baseId)
						continue;

					$variantDetail = $this->mallApiProducts->getVariantData($client, (string) $result['id'], (string) $variant['id']);
					$inMap         = $mallProductsMapByMallId[$client->getCountry()][$variantDetail['article_id']] ?? null;
					$prodId        = $prodIds[$variant['id']];
					if (!$inMap) {
						$this->productMapService->addToMap($client->getCountry(), (int) $prodId, (int) $variantDetail['article_id'], (string) $variant['id']);
					} else if (!$inMap['mallId'])
						$this->productMapService->setMallId($client->getCountry(), (int) $prodId, (int) $variantDetail['article_id'], (string) $variant['id']);
				}
			}
		}
	}

	public function processAllAvailabilities()
	{
		$currentMaps      = $this->productMapService->getMap();
		$eshopCategoryIds = array_keys($this->categoriesService->getAllCategoriesByEshopCategory());
		$prodIds          = $this->em->getConnection()->fetchFirstColumn("SELECT product_id FROM eshop_catalog__product_in_site WHERE category_id IN (" . implode(',', $eshopCategoryIds) . ") AND is_active = 1 GROUP BY product_id");
		$sendUpdate       = [];
		$sendSoldOut      = [];

		foreach ($this->mallClients->getClients() as $client) {
			$currentMap = $currentMaps[$client->getCountry()];

			foreach ($prodIds as $prodId) {
				$isInMall = $currentMap[$prodId] ?? null;
				if (!$isInMall)
					continue;

				$sendUpdate[]                             = $prodId;
				$mallIdsFromEshop[$isInMall['idForMall']] = $isInMall['idForMall'];
			}

			foreach ($this->mallApiProducts->getProductsList($client) as $mallProduct) {
				if (!isset($mallIdsFromEshop[$mallProduct['id']]))
					$sendSoldOut[] = $mallProduct['id'];
			}

			if (!empty($sendSoldOut)) {
				$this->sendSoldOut($client, $sendSoldOut);
			}

			if (!empty($sendUpdate)) {
				$this->processProductAvailabilities($sendUpdate);
			}
		}
	}

	public function processAll()
	{
		$currentMaps = $this->productMapService->getMap();
		$sendUpdate  = [];

		$eshopCategoryIds = array_keys($this->categoriesService->getAllCategoriesByEshopCategory());
		$prodIds          = $this->em->getConnection()->fetchFirstColumn("SELECT product_id FROM eshop_catalog__product_in_site WHERE category_id IN (" . implode(',', $eshopCategoryIds) . ") AND is_active = 1 GROUP BY product_id");

		foreach ($this->mallClients->getClients() as $client) {
			$currentMap            = $currentMaps[$client->getCountry()];
			$currentMapByIdForMall = [];
			$sendSoldOut           = [];
			$mallIdsFromEshop      = [];

			foreach ($currentMap as $row)
				$currentMapByIdForMall[$row['idForMall']] = $row;

			foreach ($prodIds as $prodId) {
				$sendUpdate[] = $prodId;
				$isInMall     = $currentMap[$prodId] ?? null;

				if ($isInMall)
					$mallIdsFromEshop[$isInMall['idForMall']] = $isInMall['idForMall'];
			}

			foreach ($this->mallApiProducts->getProductsList($client) as $mallProduct) {
				if (!isset($mallIdsFromEshop[$mallProduct['id']]))
					$sendSoldOut[] = $mallProduct['id'];
			}

			if ($sendSoldOut) {
				$this->sendSoldOut($client, $sendSoldOut);
			}
		}

		if (!empty($sendUpdate)) {
			foreach (array_chunk(array_unique($sendUpdate), 250) as $batch)
				$this->processProducts($batch);
		}
	}

	public function createMap(): array
	{
		$currentMaps   = $this->productMapService->getMap();
		$allIds        = $this->productService->getAllIds();
		$allOldIds     = $this->productService->getAllOldIdsMap();
		$addedProducts = 0;
		$addedVariants = 0;

		foreach ($this->mallClients->getClients() as $client) {
			$this->em->beginTransaction();
			try {
				$currentMap            = $currentMaps[$client->getCountry()] ?: [];
				$currentMapByIdForMall = [];

				foreach ($currentMap as $row)
					$currentMapByIdForMall[$row['idForMall']] = $row;

				foreach ($this->mallApiProducts->getProductsList($client) as $mallProduct) {
					if (isset($currentMapByIdForMall[$mallProduct['id']]))
						continue;

					$eshopProductId = $allOldIds[$mallProduct['id']] ?? null;
					if (!$eshopProductId || !isset($allIds[$eshopProductId]))
						continue;

					$this->productMapService->addToMap($client->getCountry(), (int) $eshopProductId, $mallProduct['product_id'], $mallProduct['id']);
					$addedProducts++;

					if ($mallProduct['has_variants']) {
						foreach ($this->mallApiProducts->getProductVariantsList($client, (string) $mallProduct['id']) as $mallVariant) {
							$mallVariantId = $mallVariant['id'];
							if (Strings::startsWith($mallVariantId, 'V'))
								$mallVariantId = substr($mallVariantId, 1);

							$eshopVariantId = $allIds[$mallVariantId] ?? null;
							if ($eshopVariantId) {
								$this->productMapService->addToMap($client->getCountry(), (int) $eshopVariantId, $mallVariant['variant_id'], $mallVariant['id']);
								$addedVariants++;
							}
						}
					}
				}

				$this->em->commit();
			} catch (\Exception $e) {
				if ($this->em->getConnection()->isTransactionActive())
					$this->em->rollback();

				return [
					'message' => $e->getMessage(),
				];
			}
		}

		return [
			'products' => $addedProducts,
			'variants' => $addedVariants,
		];
	}

	public function checkVariants()
	{

		$currentMaps      = $this->productMapService->getMap();
		$eshopCategoryIds = array_keys($this->categoriesService->getAllCategoriesByEshopCategory());
		$prodIds          = $this->em->getConnection()->fetchFirstColumn("SELECT product_id FROM eshop_catalog__product_in_site WHERE category_id IN (" . implode(',', $eshopCategoryIds) . ") AND is_active = 1 GROUP BY product_id");

		foreach ($this->mallClients->getClients() as $client) {
			$currentMap            = $currentMaps[$client->getCountry()];
			$currentMapByIdForMall = [];
			$mallIdsFromEshop      = [];

			foreach ($currentMap as $row)
				$currentMapByIdForMall[$row['idForMall']] = $row;

			foreach ($prodIds as $prodId) {
				$isInMall = $currentMap[$prodId] ?? null;

				if ($isInMall)
					$mallIdsFromEshop[$isInMall['idForMall']] = (int) $prodId;
			}

			foreach ($this->mallApiProducts->getProductsList($client) as $mallProduct) {
				if ($mallProduct['has_variants'] && isset($mallIdsFromEshop[$mallProduct['id']])) {
					$mallProductDetail  = $this->mallApiProducts->getProduct($client, (string) $mallProduct['id']);
					$eshopProductDetail = $this->productsFacade->getProduct($mallIdsFromEshop[$mallProduct['id']]);

					if (!$mallProductDetail || !$eshopProductDetail)
						continue;

					foreach ($mallProductDetail['variants'] as $mallVariant) {
						$variantId = (string) $mallVariant['id'];
						if (Strings::startsWith($variantId, 'V'))
							$variantId = substr($variantId, 1);

						if (!isset($eshopProductDetail->variants[$variantId])) {
							$this->mallApiProducts->deleteVariant($client, (string) $mallProduct['id'], (string) $mallVariant['id']);
						}
					}
				}
			}
		}
	}

	public function deleteProduct(int $productId): void
	{
		$productsMap = $this->productMapService->getMap();

		foreach ($this->mallClients->getClients() as $client) {
			$mallId = $productsMap[$client->getCountry()][$productId]['idForMall'] ?? null;

			if (!$mallId)
				continue;

			$this->mallApiProducts->deleteProduct($client, (string) $mallId);
		}
	}

	public function deleteVariant(int $productId, int $variantId): void
	{
		$productsMap = $this->productMapService->getMap();

		foreach ($this->mallClients->getClients() as $client) {
			$variantMallId = $productsMap[$client->getCountry()][$variantId]['idForMall'] ?? null;
			$productMallId = $productsMap[$client->getCountry()][$productId]['idForMall'] ?? null;

			if (!$variantMallId || !$productMallId)
				continue;

			$this->mallApiProducts->deleteVariant($client, (string) $productMallId, (string) $variantMallId);
		}
	}

	public function sendSoldOut(MallClient $client, array $ids)
	{
		if (empty($ids))
			return;

		$data = [];
		foreach ($ids as $id) {
			$data[] = [
				'id'       => (string) $id,
				'status'   => Availability::INACTIVE,
				'in_stock' => 0,
			];
		}

		foreach (array_chunk($data, 900) as $chunk)
			$this->mallApiProducts->sendAvailabilities($client, $chunk);
	}
}
