<?php declare(strict_types = 1);

namespace MapPoints\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use MapPoints\Model\Entities\Map;
use MapPoints\Model\Entities\Point;
use MapPoints\Model\Maps;
use MapPoints\Model\Points;
use Nette\Http\FileUpload;
use Nette\Http\IResponse;
use Nette\Http\Request;
use Nette\Utils\ArrayHash;
use Nette\Utils\Strings;

class MapForm extends BaseControl
{
	/** @var Map */
	public $map;

	/** @var float @persistent */
	public $x;

	/** @var float @persistent */
	public $y;

	/** @var Points */
	protected $pointsService;

	/** @var Maps */
	protected $mapsService;

	/** @var Request */
	protected $httpRequest;

	/** @var IPointFormFactory */
	protected $pointFormFactory;

	/** @var Point[]|array */
	protected $mapPoints = [];

	/** @var callable[] */
	public $pointFormCallback = [];

	public function __construct(Points $points, Maps $maps, Request $request, IPointFormFactory $pointFormFactory)
	{
		$this->pointsService    = $points;
		$this->mapsService      = $maps;
		$this->httpRequest      = $request;
		$this->pointFormFactory = $pointFormFactory;
	}

	public function render()
	{
		$this->template->map       = $this->map;
		$this->template->mapPoints = $this->mapPoints;
		$this->template->allPoints = $this->pointsService->getAll();
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	public function handlePointUpdatePos()
	{
		$pointId = intval($this->httpRequest->getPost('point'));
		$x       = floatval($this->httpRequest->getPost('x'));
		$y       = floatval($this->httpRequest->getPost('y'));

		if ($pointId && $x && $y) {
			$this->mapsService->setPoint($this->map->getId(), $pointId, $x, $y);
		}
		$this->redrawControl('points');
	}

	public function handlePointCreate()
	{
		$this->x               = floatval($this->httpRequest->getPost('x'));
		$this->y               = floatval($this->httpRequest->getPost('y'));
		$this->template->modal = 'mapFormMapPoint';
		$this->redrawControl('modal');
	}

	public function handlePointEdit()
	{
		$pointId = intval($this->httpRequest->getPost('point'));

		if ($pointId) {
			$this['pointForm']->setPoint($pointId);
			$this->template->modal = 'mapFormMapPoint';
			$this->redrawControl('modal');
		}
	}

	public function handlePointRemove()
	{
		$pointId = intval($this->httpRequest->getPost('point'));

		if ($pointId) {
			$this->mapsService->removePoint($this->map->getId(), $pointId);
		}
		$this->redrawControl('points');
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$form->addText('title', 'default.title')->setRequired()->setMaxLength(255);
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);

		if ($this->map) {
			$form->addFilesManager('image', 'mapPoints.mapForm.map');
		}
		$form->addSubmit('submitAndEdit', 'mapPoints.mapForm.saveAndSetPoints');
		$form->addSubmit('submit', 'default.save');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->map) {
				$map          = $this->map;
				$flashMessage = 'mapPoints.mapForm.edited';
			} else {
				$map          = new Map($values->title);
				$flashMessage = 'mapPoints.mapForm.added';
			}

			$map->title       = $values->title;
			$map->isPublished = $values->isPublished;
			$map->image       = $values->image;

			$this->em->persist($map)->flush();
			$this->getPresenter()->flashMessageSuccess($flashMessage);
			$this->map = $map;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	protected function createComponentPointForm()
	{
		$control = $this->pointFormFactory->create();

		$control['form']->setAjax();
		$control['form']->onSuccess['redirect'] = function() use ($control) {
			$this->getPresenter()->payload->hideModal = true;

			if ($this->getX() && $this->getY()) {
				$this->mapsService->setPoint($this->map->getId(), $control->getPoint()->getId(), $this->getX(), $this->getY());
			}

			$this->getPresenter()->redrawControl('flashes');
			$this->redrawControl('points');
		};

		return $control;
	}

	/*******************************************************************************************************************
	 * =================  Get / Set
	 */

	public function getMap() { return $this->map; }

	public function setMap($id)
	{
		$this->map = $this->mapsService->get($id);

		if ($this->map) {
			$this['form']->setDefaults([
				'title'       => $this->map->title,
				'isPublished' => $this->map->isPublished,
				'image'       => $this->map->image,
			]);

			$this->mapPoints = $this->map->points;
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}

	/**
	 * @return float|null
	 */
	public function getX() { return $this->x ? floatval(str_replace(',', '.', $this->x)) : null; }

	/**
	 * @return float|null
	 */
	public function getY() { return $this->y ? floatval(str_replace(',', '.', $this->y)) : null; }
}
