<?php declare(strict_types = 1);

namespace MapPoints\Model;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TPublish;
use Gedmo\Tree\Entity\Repository\NestedTreeRepository;
use MapPoints\Model\Entities\Group;
use MapPoints\Model\Entities\Map;
use MapPoints\Model\Entities\MapPoint;
use MapPoints\Model\Entities\Point;
use Nette\Caching\Cache;

/**
 * Class Maps
 * @package MapPoints\Model
 *
 * @method Map|object|null getReference($id)
 * @method Map[]|null getAll()
 * @method Map|null get($id)
 */
class Maps extends BaseEntityService
{
	use TPublish;

	const CACHE_NAMESPACE = 'mapPointsMap';

	protected $entityClass = Map::class;

	/** @var array */
	protected $cacheDep = [
		Cache::TAGS    => [self::CACHE_NAMESPACE],
		Cache::EXPIRE  => '1 week',
		Cache::SLIDING => true,
	];

	public function getCache()
	{
		if ($this->cache === null)
			$this->cache = new Cache($this->cacheStorage, self::CACHE_NAMESPACE);

		return $this->cache;
	}

	/**
	 * Set point on map and create point if not exits
	 *
	 * @param int   $mapId
	 * @param int   $pointId
	 * @param float $x
	 * @param float $y
	 *
	 * @throws \Doctrine\ORM\ORMException
	 */
	public function setPoint($mapId, $pointId, $x, $y)
	{
		$mapPoint = $this->em->getRepository(MapPoint::class)->findOneBy(['point' => $pointId, 'map' => $mapId]);

		if (!$mapPoint) {
			$point    = $this->em->getReference(Point::class, $pointId);
			$mapPoint = new MapPoint($this->getReference($mapId), $point);
		}

		$mapPoint->setPosition($x, $y);
		$this->em->persist($mapPoint)->flush();
	}

	/**
	 * Remove point on map
	 *
	 * @param int $mapId
	 * @param int $pointId
	 *
	 * @throws \Exception
	 */
	public function removePoint($mapId, $pointId)
	{
		$mapPoint = $this->em->getRepository(MapPoint::class)->findOneBy(['point' => $pointId, 'map' => $mapId]);

		if ($mapPoint) {
			$this->em->remove($mapPoint)->flush();
		}
	}

	public function getMapGroupsTree($mapId)
	{
		$key = self::CACHE_NAMESPACE . '/mapGroupsTree/' . $mapId;

		return $this->getCache()->load($key, function(&$dep) use ($mapId) {
			$dep = $this->cacheDep;
			$map = $this->getEr()->createQueryBuilder('m')->addSelect('mp')->addSelect('pp')
				->andWhere('m.id = :mapId')->andWhere('m.isPublished = 1')->setParameter('mapId', $mapId)
				->join('m.points', 'mp')
				->join('mp.point', 'pp')
				->getQuery()->getOneOrNullResult();

			if (!$map || !$map->points) {
				return null;
			}

			$points = [];
			foreach ($map->points as $p) {
				$points[] = $p->point;
			}

			/** @var Group[]|null $groups */
			$groups    = [];
			$gIds      = [];
			$groupsRaw = [];
			foreach ($points as $p) {
				$gIds[$p->group->getId()] = $p->group->getId();
			}

			foreach ($this->em->getRepository(Group::class)->createQueryBuilder('g')->getQuery()->useResultCache(true, 120)->getResult() as $g) {
				if (in_array($g->getId(), $gIds))
					$groupsRaw[] = $g;
			}

			foreach ($groupsRaw as $g) {
				$groups[$g->getId()] = [
					'id'     => $g->getId(),
					'parent' => $g->getParent() ? $g->getParent()->getId() : 0,
					'title'  => $g->title,
					'points' => [],
				];

				$p = $g->getParent();
				while ($p) {
					$groups[$p->getId()] = [
						'id'     => $p->getId(),
						'parent' => $p->getParent() ? $p->getParent()->getId() : 0,
						'title'  => $p->title,
						'points' => [],
					];

					$p = $p->getParent();
				}
			}

			foreach ($map->points->toArray() as $point) {
				$groups[$point->point->group->getId()]['points'][] = $point;
			}

			foreach ($groups as &$g) {
				$g['pointsCount'] = count($g['points']);
			}

			$tree = Arrays::buildTree(array_values($groups), 'parent', 'id');

			return ['groups' => $tree, 'map' => $map];
		});
	}

	public function getMapWithPoints($mapId)
	{
		$map = $this->get($mapId);

		if ($map) {
			$ids = array_map(function($v) { return $v->point->getId(); }, $map->points->toArray());

			$points = [];
			if ($ids) {
				$points = $this->em->getRepository(Point::class)->findBy(['id' => $ids]);
			}

			if ($points) {
				$groups = $this->em->getRepository(Group::class)->findBy(['id' => array_map(function($v) { return $v->group->getId(); }, $points)]);

				if ($groups) {
					$map->groups = $groups;
					$this->em->getRepository(Group::class)->findBy(['id' => array_map(function($v) { return $v->parent->getId(); }, $groups)]);
				}
			}
		}

		return $map;
	}
}
