<?php declare(strict_types = 1);

namespace MapPoints\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use MapPoints\Model\Entities\Group;
use MapPoints\Model\Groups;
use MapPoints\Model\Maps;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;

class GroupForm extends BaseControl
{
	/** @var Group */
	public $group;

	/** @var Groups */
	protected $groupsService;

	/** @var Maps */
	protected $mapsService;

	public function __construct(Groups $groups, Maps $maps)
	{
		$this->groupsService = $groups;
		$this->mapsService   = $maps;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$groups = [null => ''];
		foreach ($this->groupsService->getEr()->createQueryBuilder('g')->orderBy('g.title')->getQuery()->getResult() as $g) {
			/** @var Group $g */
			if ($this->group && $this->group->getId() == $g->getId())
				continue;

			$arr  = [];
			$t    = $g;
			$skip = false;
			while ($t->getParent()) {
				$t = $t->getParent();
				if ($this->group && $this->group->getId() == $t->getId()) {
					$skip = true;
				}

				$arr[] = $t->title;
			}

			if ($skip)
				continue;

			$groups[$g->getId()] = ($arr ? implode(' -> ', $arr) . ' -> ' : '') . $g->title;
		}

		$form->addText('title', 'default.title')->setRequired()->setMaxLength(255);
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);
		$form->addSelect('parent', 'default.parent', $groups);
		$form->addSubmit('submit', 'default.save');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			if ($this->group) {
				$group        = $this->group;
				$flashMessage = 'mapPoints.groupForm.edited';
			} else {
				$group        = new Group($values->title);
				$flashMessage = 'mapPoints.groupForm.added';
			}

			$group->title       = $values->title;
			$group->isPublished = $values->isPublished;

			if ($values->parent == '') {
				$group->setParent(null);
			} else {
				$group->setParent($this->groupsService->get($values->parent));
			}

			$this->em->persist($group)->flush();
			$this->mapsService->clearCache();
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function setGroup($id)
	{
		$this->group = $this->groupsService->get($id);

		if ($this->group) {
			$this['form']->setDefaults([
				'title'       => $this->group->title,
				'isPublished' => $this->group->isPublished,
			]);

			if ($this->group->parent && array_key_exists($this->group->parent->getId(), $this['form']['parent']->getItems())) {
				$this['form']['parent']->setDefaultValue($this->group->parent->getId());
			}
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
