<?php declare(strict_types = 1);

namespace MapPoints\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use MapPoints\Model\Entities\Point;
use MapPoints\Model\Groups;
use MapPoints\Model\Maps;
use MapPoints\Model\Points;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;

class PointForm extends BaseControl
{
	/** @var Point */
	public $point;

	/** @var int @persistent */
	public $pointId;

	/** @var Points */
	protected $pointsService;

	/** @var Groups */
	protected $groupsService;

	/** @var Maps */
	protected $mapsService;

	public function __construct(Points $points, Groups $groups, Maps $maps)
	{
		$this->pointsService = $points;
		$this->groupsService = $groups;
		$this->mapsService = $maps;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm()
	{
		$form = $this->createForm();

		$groups = [];
		foreach ($this->groupsService->getEr()->findBy([], ['title' => 'ASC']) as $g) {
			$arr = [];
			$t   = $g;
			while ($t->parent) {
				$t     = $t->parent;
				$arr[] = $t->title;
			}
			$groups[$g->getId()] = ($arr ? implode(' -> ', $arr) . ' -> ' : '') . $g->title;
		}
		asort($groups);

		$form->addText('title', 'default.title')->setMaxLength(255)->setRequired();
		$form->addText('street', 'default.street')->setMaxLength(255);
		$form->addText('city', 'default.city')->setMaxLength(255);
		$form->addText('zip', 'default.zipNumber')->setMaxLength(8);
		$form->addSelect('group', 'default.group', $groups)->setRequired();
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);
		$form->addEditor('text', 'default.text');
		$form->addSubmit('submit', 'default.save');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash &$values)
	{
		try {
			$group = $values->group ? $this->groupsService->get($values->group) : null;

			if ($this->pointId) {
				$point        = $this->pointsService->get($this->pointId);
				$flashMessage = 'mapPoints.pointForm.edited';
			} else {
				$point        = new Point($values->title);
				$flashMessage = 'mapPoints.pointForm.added';
			}

			$point->title       = $values->title;
			$point->street      = $values->street;
			$point->city        = $values->city;
			$point->zip         = $values->zip;
			$point->isPublished = $values->isPublished;
			$point->group       = $group;
			$point->setText($values->text);

			$this->em->persist($point)->flush();
			$this->mapsService->clearCache();
			$this->getPresenter()->flashMessageSuccess($flashMessage);
			$this->point = $point;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	public function getPoint()
	{
		if (!$this->point && $this->pointId) {
			$this->point = $this->pointsService->get($this->pointId);
		}

		return $this->point;
	}

	public function setPoint($id)
	{
		$this->pointId = $id;
		$this->point   = $this->pointsService->get($id);

		if ($this->point) {
			$p = $this->point;
			$f = $this['form'];

			$f->setDefaults([
				'title'       => $p->title,
				'street'      => $p->street,
				'city'        => $p->city,
				'zip'         => $p->zip,
				'isPublished' => $p->isPublished,
				'text'        => $p->getText(),
			]);

			if ($p->group && array_key_exists($p->group->getId(), $f['group']->getItems()))
				$f['group']->setDefaultValue($p->group->getId());
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
