<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Core\FrontModule\Presenters\BasePresenter;
use Core\Model\Event\ControlEvent;
use Core\Model\Http\Session;
use Core\Model\UI\FrontPresenter;
use Currency\Model\Currencies;
use EshopCatalog\DI\EshopCatalogExtension;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopOrders\FrontModule\Model\CartFacade;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\RemovedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\FrontModule\Model\Subscribers\CartSubscriber;
use EshopOrders\FrontModule\Presenters\DefaultPresenter;
use EshopOrders\Model\Entities\CartItem;
use MeasuringCodes\FrontModule\Components\GTagEventControl;
use MeasuringCodes\FrontModule\Components\IClarityEventControlFactory;
use MeasuringCodes\FrontModule\Components\IDataLayerControlFactory;
use MeasuringCodes\FrontModule\Components\IEcoTrackControlFactory;
use MeasuringCodes\FrontModule\Components\IFBPixelEventControlFactory;
use MeasuringCodes\FrontModule\Components\IGTagEventControlFactory;
use MeasuringCodes\FrontModule\Model\Dao\GoogleAdsType;
use MeasuringCodes\FrontModule\Model\Helpers\DataLayerEventHelper;
use MeasuringCodes\FrontModule\Model\Helpers\DataLayerHelper;
use MeasuringCodes\FrontModule\Model\Helpers\Ga4Helper;
use MeasuringCodes\FrontModule\Model\Helpers\GTagEventHelper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesConfig;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;
use Nette\DI\Container;
use Nette\Http\SessionSection;

class EshopCartSubscriber extends MeasuringCodesEventSubscriber
{
	protected array $cartChanges = [
		'added'   => [],
		'updated' => [],
		'removed' => [],
	];

	protected array $orderChanges = [];

	public function __construct(
		public Container                      $container,
		protected TypesList                   $typesList,
		protected IGTagEventControlFactory    $gTagEventControlFactory,
		protected IFBPixelEventControlFactory $fbPixelEventControlFactory,
		protected IDataLayerControlFactory    $dataLayerControlFactory,
		protected IClarityEventControlFactory $clarityEventControlFactory,
		protected Session                     $session,
		protected Ga4Helper                   $ga4Helper,
		protected DataLayerHelper             $dataLayerHelper,
		protected IEcoTrackControlFactory     $ecoTrackControlFactory,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FrontPresenter::class . '::beforeRender'  => 'beforeRender',
			'eshopOrders.cartAddItem'                 => 'onAddItem',
			'eshopOrders.cartUpdateItem'              => 'onUpdateItem',
			DefaultPresenter::class . '::renderOrder' => 'renderOrder',
			'eshopOrders.saveOrderFormDataStep2'      => 'orderSaveStep2',
		];
	}

	public function beforeRender(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed() || !$this->container->hasService('eshopCatalog.front.productsFacade') || !$this->container->hasService('eshopOrders.front.cartFacade')) {
			return;
		}

		if (!MeasuringCodesConfig::load('allowCartEvents')) {
			return;
		}

		/** @var BasePresenter $presenter */
		$presenter = $event->control;

		/** @var ProductsFacade $productsFacade */
		$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');

		/** @var Currencies $currencies */
		$currencies = $this->container->getService('currency.currencies');
		$currency   = $currencies->getCurrent()->getCode();

		$clarityType = $this->typesList->getType('clarity');
		if ($clarityType && $clarityType->isActive()) {
			$clarityCartComponent = $this->clarityEventControlFactory->create($clarityType);
			$presenter->addBodyEndComponent($clarityCartComponent, $clarityType->getKey() . '_cart');
		} else {
			$clarityCartComponent = null;
		}

		/** @var CartFacade $cartFacade */
		$cartFacade = $this->container->getService('eshopOrders.front.cartFacade');

		$items = [
			'add_to_cart'       => [], // Pocet ks
			'remove_from_cart'  => [], // Pocet ks
			'new_in_cart'       => [], // Nove polozky
			'deleted_from_cart' => [], // Odebrane polozky
		];

		$currentQuantities = [];
		foreach ($cartFacade->getCart()->getCartItems() as $item) {
			$currentQuantities[$item->getId()] = $item->quantity;
		}

		foreach (CartSubscriber::$cartChanges['isNew'] as $item) {
			/** @var CartItem $item */
			$items['new_in_cart'][$item->getId()] = $item;
		}

		foreach (CartSubscriber::$cartChanges['removed'] as $itemId) {
			$items['remove_from_cart'][$itemId]  = $currentQuantities[$itemId] ?? 0;
			$items['deleted_from_cart'][$itemId] = $itemId;
		}

		foreach ($this->cartChanges['added'] as $id => $change) {
			$items['add_to_cart'][$id] = $change;
		}

		foreach ($this->cartChanges['updated'] as $id => $change) {
			if ($change > 0) {
				$items['add_to_cart'][$id] = $change;
			} else {
				$items['remove_from_cart'][$id] = abs($change);
			}
		}

		if ($clarityCartComponent) {
			if (!empty($items['add_to_cart'])) {
				$clarityCartComponent->eventName = 'add_to_cart';
			} else if (!empty($items['remove_from_cart'])) {
				$clarityCartComponent->eventName = 'remove_from_cart';
			}

			if ($presenter->isAjax()) {
				$clarityCartComponent->redrawControl('wrap');
			}
		}

		foreach ($items as $eventName => $v) {
			$ga4Component = null;
			$dlComponent  = null;

			if ($this->ga4Helper->isAllowed() || $this->dataLayerHelper->isAllowed()) {
				$gaItems = [];
				$value   = 0;

				if (!empty($v)) {
					foreach ($v as $id => $quantity) {
						$product = $productsFacade->getProduct((int) $id);

						if ($product) {
							$item             = GTagEventHelper::getGa4ItemFromProduct($product);
							$item['quantity'] = $quantity;
							$value            += $item['price'] * $item['quantity'];
							$gaItems[]        = $item;
						}
					}
				}

				$evName = $eventName === 'add_to_cart' ? 'add_to_cart' : 'remove_from_cart';

				if ($this->ga4Helper->isAllowed()) {
					$ga4Component = $this->getGa4Component($presenter, $evName, 'ga4');

					if ($ga4Component) {
						$sendTo = [$ga4Component->getType()->getFieldValue('ga4_id')];

						/** @var GoogleAdsType|null $gAdsType */
						$gAdsType = $this->typesList->getType('googleAds');
						if ($gAdsType && $gAdsType->isActive()) {
							$sendTo[] = $gAdsType->getFullId();
						}

						if (!empty($gaItems)) {
							$ga4Component->setData([
								'currency' => $currency,
								'value'    => GTagEventHelper::formatNumber($value),
								'items'    => $gaItems,
								'send_to'  => $sendTo,
							]);
						}
					}
				}

				// DL
				$dlType = DataLayerEventHelper::getTypeIfAllowed($this->typesList);
				if ($dlType) {
					$dlComponent = DataLayerEventHelper::getComponent($presenter, $this->dataLayerControlFactory, $evName, $dlType, 'dl');

					if ($dlComponent && !empty($gaItems)) {
						$dlComponent->setGTagData([
							$evName, [
								'currency' => $currency,
								'value'    => GTagEventHelper::formatNumber($value),
								'items'    => $gaItems,
							],
						]);
					}
				}
			}

			// Ajax
			if ($presenter->isAjax()) {
				if ($ga4Component) {
					$ga4Component->redrawControl('wrap');
				}
				if ($dlComponent) {
					$dlComponent->redrawControl('wrap');
				}
			}
		}

		// FB pixel
		$fbpPixel = $this->typesList->getType('facebookPixel');
		if ($fbpPixel->isActive() && $fbpPixel->getFieldValue('enableEcommerce')) {
			$fbpComponent = $this->fbPixelEventControlFactory->create($fbpPixel, 'AddToCart');
			$fbpComponent->showOnlyIfCode3();
			$presenter->addBodyEndComponent($fbpComponent, $fbpPixel->getKey() . 'AddToCart');

			if (!empty($items['add_to_cart'])) {
				$totalValue = 0;
				$data       = [
					'currency'     => $currency,
					'content_type' => 'product',
					'contents'     => [],
				];

				foreach ($items['add_to_cart'] as $id => $quantity) {
					if (!$id) {
						continue;
					}

					$product    = $productsFacade->getProduct((int) $id);
					$price      = round($product->getPrice() * $quantity, 2);
					$totalValue += $price;

					$data['contents'][] = [
						'id'       => (string) $id,
						'quantity' => $quantity,
					];
				}

				$data['value'] = $totalValue;

				$fbpComponent->setCode3($data);

				if ($presenter->isAjax()) {
					$fbpComponent->redrawControl('wrap');
				}
			}
		}

		$eshopOrdersSession = new SessionSection($this->session, OrderSubscriber::SESSION_SECTION);
		$repeatOrder        = $eshopOrdersSession->get(OrderSubscriber::ORDER_REPEAT);
		if ($repeatOrder) {
			$this->ga4Helper->sendEvent(
				'repeat_order',
				[
					'order_id' => $repeatOrder['orderId'],
				],
				$presenter,
			);

			$eshopOrdersSession->remove(OrderSubscriber::ORDER_REPEAT);
		}

		// Ecomail cart tracking
		if (MeasuringCodesConfig::load('typesList.ecoMail.enableCartTracking')) {
			$ecoMailType = $this->typesList->getType('ecoMail');

			if ($ecoMailType && $ecoMailType->isActive()) {
				$ecoMailCartComponent             = $this->ecoTrackControlFactory->create($ecoMailType);
				$ecoMailCartComponent->useSnippet = true;

				if (!empty($items['new_in_cart']) || !empty($items['deleted_from_cart'])) {
					$data = [
						'schema' => '',
						'data'   => [
							'action'   => 'Basket',
							'products' => [],
						],
					];

					$i = 0;
					foreach ($cartFacade->getCart()->getCartItems() as $item) {
						$row = [
							'productId' => $item->id,
							'url'       => $item->getLink(),
							'name'      => $item->title,
							'price'     => $item->getPrice(),
						];

						if ($item->getImage()) {
							$row['img_url'] = $presenter->baseUrl . ltrim($item->getImage()->getFilePath(), '/');
						}

						if ($item->getProduct()) {
							$row['description'] = $item->getProduct()->getDescription();
						}

						$data['data']['products'][] = $row;
						$i++;

						if ($i > 10) {
							break;
						}
					}

					$ecoMailCartComponent->eventName = 'trackUnstructEvent';
					$ecoMailCartComponent->data      = $data;
				}


				$presenter->addBodyEndComponent($ecoMailCartComponent, $ecoMailType->getKey() . 'Cart');

				if ($presenter->isAjax()) {
					$ecoMailCartComponent->redrawControl('wrap');
				}
			}
		}
	}

	public function onAddItem(AddedCartItemEvent $item): void
	{
		$this->cartChanges['added'][$item->item->productId] = $item->item->quantity;
	}

	public function onUpdateItem(UpdatedCartItemEvent $item): void
	{
		$this->cartChanges['updated'][$item->productId] = $item->quantity - $item->beforeQuantity;
	}

	public function onRemoveItem(RemovedCartItemEvent $item): void
	{
		$this->cartChanges['removed'][$item->itemId] = $item->itemId;
	}

	public function renderOrder(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var DefaultPresenter|BasePresenter $presenter */
		$presenter = $event->control;
		/** @var ProductsFacade $productsFacade */
		$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');

		/** @var Currencies $currencies */
		$currencies = $this->container->getService('currency.currencies');
		$currency   = $currencies->getCurrent()->getCode();

		$cartsService = property_exists($presenter, 'cartsService') ? $presenter->cartsService : null;

		if (!$cartsService) {
			return;
		}

		$cart           = $cartsService->getCurrentCart();
		$cartItemsPrice = $cart->getCartItemsPrice();

		$discount     = array_values($cart->discounts)[0] ?? null;
		$discountCode = $discount ? $discount->id : null;

		$cartItems       = $cart->getCartItems();
		$cartDaoProducts = $productsFacade->getProducts(array_map(fn($v) => $v->getProductId(), $cartItems));

		$clarityType = $this->typesList->getType('clarity');
		if ($clarityType && $clarityType->isActive()) {
			$clarityShippingComponent = $this->clarityEventControlFactory->create($clarityType);
			$presenter->addBodyEndComponent($clarityShippingComponent, $clarityType->getKey() . '_cart_shipping');

			$clarityPaymentComponent = $this->clarityEventControlFactory->create($clarityType);
			$presenter->addBodyEndComponent($clarityPaymentComponent, $clarityType->getKey() . '_cart_payment');
		} else {
			$clarityShippingComponent = null;
			$clarityPaymentComponent  = null;
		}

		// InitiateCheckout
		$fbpType = $this->typesList->getType('facebookPixel');
		if ($presenter->getParameter('do') === null && empty($presenter->getHttpRequest()->getPost())) {
			if ($fbpType && $fbpType->isActive() && $fbpType->getFieldValue('enableEcommerce')) {
				$fbpComponent = $this->fbPixelEventControlFactory->create($fbpType, 'InitiateCheckout', 'track');

				$code3 = [
					'currency'  => $currency,
					'num_items' => count($cartItems),
					'value'     => $cartItemsPrice,
				];

				$fbpComponent->setCode3($code3);

				$presenter->addBodyEndComponent($fbpComponent, $fbpType->getKey() . 'InitiateCheckout');
			}
		}

		// GA
		$ga4PaymentComponent   = $this->getGa4Component($presenter, 'add_payment_info');
		$ga4SpeditionComponent = $this->getGa4Component($presenter, 'add_shipping_info');

		// DL
		$dlType = DataLayerEventHelper::getTypeIfAllowed($this->typesList);
		if ($dlType && $dlType->isActive()) {
			$dlPaymentComponent   = DataLayerEventHelper::getComponent($presenter, $this->dataLayerControlFactory, 'add_payment_info', $dlType, 'dl');
			$dlSpeditionComponent = DataLayerEventHelper::getComponent($presenter, $this->dataLayerControlFactory, 'add_shipping_info', $dlType, 'dl');
		} else {
			$dlPaymentComponent   = null;
			$dlSpeditionComponent = null;
		}

		// FB
		$fbPaymentComponent = null;
		if ($fbpType && $fbpType->isActive() && $fbpType->getFieldValue('enableEcommerce')) {
			$fbPaymentComponent = $this->fbPixelEventControlFactory->create($fbpType, 'AddPaymentInfo', 'track');
			$fbPaymentComponent->showOnlyIfCode3();
			$presenter->addBodyEndComponent($fbPaymentComponent, $fbpType->getKey() . 'AddPaymentInfo');
		}

		if ($presenter->getParameter('do') !== null) {
			// Nastaveni dopravy
			if (isset($this->orderChanges['spedition'])) {
				/** @var Speditions $speditions */
				$speditions = $this->container->getService('eshopOrders.front.speditions');
				$spedition  = $speditions->get((int) $this->orderChanges['spedition']);

				//GA 4
				if ($ga4SpeditionComponent || $dlSpeditionComponent) {
					$arr = [
						'currency'     => $currency,
						'value'        => GTagEventHelper::formatNumber($cartItemsPrice),
						'payment_type' => $spedition ? $spedition->getName() : '',
						'items'        => GTagEventHelper::getGa4ItemsFromProducts($cartItems, $cartDaoProducts),
					];

					if ($discountCode) {
						$arr['coupon'] = $discountCode;
					}

					if ($ga4SpeditionComponent) {
						$ga4SpeditionComponent->setData($arr);
					}

					if ($dlSpeditionComponent) {
						$dlSpeditionComponent->setGTagData([
							'add_shipping_info',
							$arr,
						]);
					}
				}

				// Clarity
				if ($clarityShippingComponent) {
					$clarityShippingComponent->eventName = 'add_shipping_info';
				}
			}

			// Nastaveni platby
			if (isset($this->orderChanges['payment'])) {
				/** @var Payments $payments */
				$payments = $this->container->getService('eshopOrders.front.payments');
				$payment  = $payments->get((int) $this->orderChanges['payment']);

				// FB
				if ($fbPaymentComponent) {
					$code3 = [
						'content_type' => 'product',
						'content_ids'  => array_values(array_map(fn($row) => (string) $row->id, $cartItems)),
						'content_name' => array_values(array_map(fn($row) => $row->title, $cartItems)),
						'value'        => $cartItemsPrice,
						'currency'     => $currency,
					];

					$fbPaymentComponent->setCode3($code3);
				}

				//GA 4
				if ($ga4PaymentComponent || $dlPaymentComponent) {
					$arr = [
						'currency'     => $currency,
						'value'        => GTagEventHelper::formatNumber($cartItemsPrice),
						'payment_type' => $payment ? $payment->getName() : '',
						'items'        => GTagEventHelper::getGa4ItemsFromProducts($cartItems, $cartDaoProducts),
					];

					if ($discountCode) {
						$arr['coupon'] = $discountCode;
					}

					if ($ga4PaymentComponent) {
						$ga4PaymentComponent->setData($arr);
					}

					if ($dlPaymentComponent) {
						$dlPaymentComponent->setGTagData([
							'add_payment_info',
							$arr,
						]);
					}
				}

				// Clarity
				if ($clarityPaymentComponent) {
					$clarityPaymentComponent->eventName = 'add_payment_info';
				}
			}

			if ($presenter->isAjax()) {
				if ($fbPaymentComponent) {
					$fbPaymentComponent->redrawControl('wrap');
				}

				if ($ga4SpeditionComponent) {
					$ga4SpeditionComponent->redrawControl('wrap');
				}

				if ($dlSpeditionComponent) {
					$dlSpeditionComponent->redrawControl('wrap');
				}

				if ($ga4PaymentComponent) {
					$ga4PaymentComponent->redrawControl('wrap');
				}

				if ($dlPaymentComponent) {
					$dlPaymentComponent->redrawControl('wrap');
				}

				if ($clarityShippingComponent) {
					$clarityShippingComponent->redrawControl('wrap');
				}

				if ($clarityPaymentComponent) {
					$clarityPaymentComponent->redrawControl('wrap');
				}
			}

			return;
		}

		// Vytvoreni transakce
		$ga4Component = $this->getGa4Component($presenter, 'begin_checkout', 'ga4');
		$dlComponent  = $dlType ? DataLayerEventHelper::getComponent($presenter, $this->dataLayerControlFactory, 'begin_checkout', $dlType, 'dl') : null;
		$ga4Items     = [];

		foreach ($cartItems as $item) {
			$product = $cartDaoProducts[$item->getProductId()] ?? null;

			$ga4Items[] = GTagEventHelper::getGa4ItemFromProduct($item, $product);
		}

		if ($ga4Items) {
			if ($ga4Component) {
				$ga4Component->setData([
					'currency' => $currency,
					'value'    => GTagEventHelper::formatNumber($cartItemsPrice),
					'coupon'   => $discountCode,
					'items'    => $ga4Items,
				]);
			}

			if ($dlComponent) {
				$dlComponent->setGTagData([
					'begin_checkout', [
						'currency' => $currency,
						'value'    => GTagEventHelper::formatNumber($cartItemsPrice),
						'coupon'   => $discountCode,
						'items'    => $ga4Items,
					],
				]);
			}
		}

		// view item gads
		/** @var GoogleAdsType|null $gAdsType */
		$gAdsType = $this->typesList->getType('googleAds');
		if ($gAdsType && $gAdsType->isActive()) {
			$gAdsComponent = GTagEventHelper::getComponent($presenter, $this->gTagEventControlFactory, 'view_cart', $gAdsType, 'gads');

			if ($gAdsComponent) {
				$gAdsComponent->setData([
					'currency' => $currency,
					'value'    => GTagEventHelper::formatNumber($cartItemsPrice),
					'items'    => $ga4Items,
					'send_to'  => $gAdsType->getFullId(),
				]);
			}
		}

		// clarity view cart
		$clarityType = $this->typesList->getType('clarity');
		if ($clarityType && $clarityType->isActive()) {
			$clarityComponent            = $this->clarityEventControlFactory->create($clarityType);
			$clarityComponent->eventName = 'view_cart';
			$clarityComponent->useSnippet(false);

			$presenter->addBodyEndComponent($clarityComponent, $clarityType->getKey() . '_view_cart');
		}
	}

	public function orderSaveStep2(SaveOrderFormDataEvent $event): void
	{
		$this->orderChanges = [
			'spedition' => $event->data['spedition'],
			'payment'   => $event->data['payment'],
		];
	}

	protected function getComponent(
		BasePresenter $presenter,
		string        $eventName,
		?string       $componentNameSuffix = null,
	): ?GTagEventControl
	{
		$type = GTagEventHelper::getTypeIfAllowed($this->typesList);
		if (!$type) {
			return null;
		}

		return GTagEventHelper::getComponent(
			$presenter,
			$this->gTagEventControlFactory,
			$eventName,
			$type,
			$componentNameSuffix,
		);
	}

	protected function getGa4Component(
		BasePresenter $presenter,
		string        $eventName,
		?string       $componentNameSuffix = null,
	): ?GTagEventControl
	{
		if (!class_exists(EshopCatalogExtension::class) || !MeasuringCodesConfig::load(
				'typesList.googleAnalytics.enableGa4',
			)) {
			return null;
		}

		$type = $this->typesList->getType('googleAnalytics');
		if (!$type || !$type->isActive() || !$type->getFieldValue('ga4_id')) {
			return null;
		}

		return GTagEventHelper::getComponent(
			$presenter,
			$this->gTagEventControlFactory,
			$eventName,
			$type,
			$componentNameSuffix,
		);
	}
}
