<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\FrontModule\Presenters\BasePresenter;
use Core\Model\Event\ControlEvent;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Presenters\DefaultPresenter;
use MeasuringCodes\FrontModule\Components\IClarityEventControlFactory;
use MeasuringCodes\FrontModule\Components\IDataLayerControlFactory;
use MeasuringCodes\FrontModule\Components\IFBPixelEventControlFactory;
use MeasuringCodes\FrontModule\Components\IGTagEventControlFactory;
use MeasuringCodes\FrontModule\Components\TypeControlFactory;
use MeasuringCodes\FrontModule\Model\Dao\GoogleAdsType;
use MeasuringCodes\FrontModule\Model\Helpers\DataLayerHelper;
use MeasuringCodes\FrontModule\Model\Helpers\EcoMailHelper;
use MeasuringCodes\FrontModule\Model\Helpers\Ga4Helper;
use MeasuringCodes\FrontModule\Model\Helpers\GTagEventHelper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;
use Nette\DI\Container;
use function str_contains;

class EshopCatalogSubscriber extends MeasuringCodesEventSubscriber
{
	public function __construct(
		protected TypeControlFactory          $typeControlFactory,
		protected TypesList                   $typesList,
		protected IGTagEventControlFactory    $gTagEventControlFactory,
		protected IFBPixelEventControlFactory $fbPixelEventControlFactory,
		protected IClarityEventControlFactory $clarityEventControlFactory,
		protected Container                   $container,
		protected IDataLayerControlFactory    $dataLayerControlFactory,
		protected Ga4Helper                   $ga4Helper,
		protected DataLayerHelper             $dataLayerHelper,
		protected EcoMailHelper               $ecoMailHelper,
		protected Translator                  $translator,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			DefaultPresenter::class . '::renderCategory' => 'defaultRenderCategory',
			DefaultPresenter::class . '::renderProduct'  => 'defaultRenderProduct',
		];
	}

	public function defaultRenderCategory(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var DefaultPresenter|BasePresenter $presenter */
		$presenter = $event->control;

		if (str_contains((string) $presenter->getParameter('do'), 'cartAddForm')) {
			return;
		}

		$category = property_exists($presenter, 'category') ? $presenter->category : null;

		if ($category) {
			$this->ga4Helper->sendPageView('eshop_category', $presenter);
			$this->dataLayerHelper->sendPageView('eshop_category', $presenter);
			$this->ecoMailHelper->sendStructEvent(
				'kategorie',
				'zobrazení',
				'',
				$category->name,
				(string) $category->id,
				$presenter,
			);
		}

		// Clarity
		$clarityType = $this->typesList->getType('clarity');
		if ($clarityType && $clarityType->isActive()) {
			$clarityComponent            = $this->clarityEventControlFactory->create($clarityType);
			$clarityComponent->eventName = 'view_category';
			$clarityComponent->useSnippet(false);

			$presenter->addBodyEndComponent($clarityComponent, $clarityType->getKey() . '_view_category');
		}
	}

	public function defaultRenderProduct(ControlEvent $event): void
	{
		$data = [];
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var DefaultPresenter|BasePresenter $presenter */
		$presenter = $event->control;
		$product   = property_exists($presenter, 'product') ? $presenter->product : null;
		$isSk      = $this->translator->getLocale() === 'sk';

		if (str_contains((string) $presenter->getParameter('do'), 'cartAddForm') || !$product) {
			return;
		}

		/** @var Currencies $currencies */
		$currencies = $this->container->getService('currency.currencies');
		$currency   = $currencies->getCurrent()->getCode();

		// GADS
		/** @var GoogleAdsType|null $gAdsType */
		$gAdsType = $this->typesList->getType('googleAds');
		if ($gAdsType && $gAdsType->isActive()) {
			$gAdsComponent = GTagEventHelper::getComponent(
				$presenter,
				$this->gTagEventControlFactory,
				'view_item',
				$gAdsType,
			);

			if ($gAdsComponent) {
				$data['items'][]  = GTagEventHelper::getItemFromProduct($product);
				$data['send_to']  = $gAdsType->getFullId();
				$data['currency'] = $currency;
				$data['value']    = GTageventHelper::getItemPriceFromProduct($product);

				$gAdsComponent->setData($data);
			}
		}

		// FB pixel
		$fbpType = $this->typesList->getType('facebookPixel');
		if ($fbpType && $fbpType->isActive() && $fbpType->getFieldValue('enableEcommerce')) {
			$fbpComponent = $this->fbPixelEventControlFactory->create($fbpType, 'ViewContent');

			$code3 = [
				'contents'     => [
					[
						'id'       => (string) $product->getId(),
						'quantity' => 1,
					],
				],
				'content_type' => 'product',
				'content_name' => $product->getName(),
				'value'        => $product->getPrice(),
				'currency'     => $currency,
				'condition'    => $product->condition,
			];


			if ($product->defaultCategory) {
				$categoryName = $product->defaultCategory->getParentPathStringFlipped();

				if ($categoryName)
					$categoryName .= ' > ';
				$categoryName .= $product->defaultCategory->name;

				$code3['content_category'] = trim((string) $categoryName);
			}

			if ($product->getAvailability() && $product->getAvailability()->getOpenGraphText()) {
				$code3['availability'] = $product->getAvailability()->getOpenGraphText();
			}

			$fbpComponent->setCode3($code3);

			$presenter->addBodyEndComponent($fbpComponent, $fbpType->getKey() . 'ViewContent');
		}

		// GA 4
		$this->ga4Helper->sendEvent(
			'view_item',
			[
				'currency' => $currency,
				'value'    => GTagEventHelper::formatNumber($product->getPrice()),
				'items'    => [GTagEventHelper::getGa4ItemFromProduct($product)],
			],
			$presenter,
			'ga4',
		);
		$this->dataLayerHelper->sendEvent(
			'view_item',
			[
				'currency' => $currency,
				'value'    => GTagEventHelper::formatNumber($product->getPrice()),
				'items'    => [GTagEventHelper::getGa4ItemFromProduct($product)],
			],
			$presenter,
			'dl',
		);

		$this->ga4Helper->sendPageView('eshop_product', $presenter);
		$this->dataLayerHelper->sendPageView('eshop_product', $presenter);

		$this->ecoMailHelper->sendEcmProductView(
			(string) $product->id,
			$presenter
		);

		$this->ecoMailHelper->sendStructEvent(
			'produkt',
			'zobrazení',
			'',
			$product->getName(),
			(string) $product->id,
			$presenter,
		);

		// Heureka
		$hrqType = $this->typesList->getType('heurekaConversion' . ($isSk ? 'Sk' : ''));
		if ($hrqType && $hrqType->isActive() && $hrqType->getFieldValue('api_key')) {
			$hrqComponent = $hrqType->factory->create($hrqType);
			$hrqComponent->setRenderFile('productDetail');
			$hrqComponent->setParam('isSk', $isSk);

			$presenter->addBodyEndComponent($hrqComponent, $hrqType->getKey() . 'ProductDetail');
		}

		// Clarity
		$clarityType = $this->typesList->getType('clarity');
		if ($clarityType && $clarityType->isActive()) {
			$clarityComponent            = $this->clarityEventControlFactory->create($clarityType);
			$clarityComponent->eventName = 'view_product';
			$clarityComponent->useSnippet(false);

			$presenter->addBodyEndComponent($clarityComponent, $clarityType->getKey() . '_view_product');
		}
	}
}
