<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Core\Model\Event\FormSuccessEvent;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use EshopOrders\FrontModule\Components\Customer\AccountSettings;
use EshopOrders\Model\Entities\Customer;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\EcoMail;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;
use Tracy\Debugger;
use Users\Model\Entities\User;

class EshopOrdersAccountSettingsSubscriber extends MeasuringCodesEventSubscriber
{
	public function __construct(
		protected TypesList $typesList,
		protected EcoMail   $ecoMail,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			AccountSettings::class . '::formSuccess' => 'formSuccess',
		];
	}

	public function formSuccess(FormSuccessEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var User $user */
		$user = $event->custom['entity'];
		/** @var Customer $customer */
		$customer = $event->custom['entityCustomer'];

		$ecoMailType = $this->typesList->getType('ecoMail');
		if ($user && $customer && $ecoMailType && $ecoMailType->isActive()) {
			$appId  = $ecoMailType->getFieldValue('appId');
			$listId = $ecoMailType->getFieldValue('listId');

			if ($appId && $listId) {
				try {
					$data = $this->ecoMail->getSubscriberFromCustomer($customer);

					$this->ecoMail->updateSubscriber($data['email'] ?: $customer->getUser()->getEmail(), $data, false);
				} catch (\Exception $e) {
					$msg = sprintf("Failed export data to ecomail from AccountSettings: user ='%s' (%s: %s) - %s",
						$user->email, $e->getFile(), $e->getLine(), $e->getMessage());

					Debugger::log($msg, 'eshopordersAccountData');
					LogNotifier::toDevelopers($msg, 'Ecomail - %siteDomain%');
				}
			}
		}
	}
}
