<?php declare(strict_types = 1);

namespace MeasuringCodes\Model\Zbozi;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseFrontEntityService;
use DateTime;
use Exception;
use MeasuringCodes\FrontModule\Model\Zbozi\ZboziCache;
use MeasuringCodes\Model\Entities\ZboziShopRating;
use Nette\InvalidStateException;
use Nette\Utils\Json;
use Tracy\Debugger;

class RatingService extends BaseFrontEntityService
{
	protected $entityClass = ZboziShopRating::class;

	public function __construct(
		protected ZboziCache $zboziCache,
	)
	{
	}

	public function downloadRating(string $key, string $apiKey, string $shopId): void
	{
		try {
			$rating       = $this->fetchRating($apiKey, $shopId);
			$reviewsCount = $this->fetchReviewsCount($apiKey, $shopId);

			$conn = $this->em->getConnection();

			$exist = $conn->fetchOne('SELECT id FROM measuring_codes__zbozi_shop_rating WHERE id = ?', [$key]);
			if ($exist) {
				$conn->update('measuring_codes__zbozi_shop_rating', [
					'rating'        => $rating,
					'reviews_count' => $reviewsCount,
					'updated'       => (new DateTime)->format('Y-m-d H:i:s'),
				], [
					'id' => $key,
				]);
			} else {
				$conn->insert('measuring_codes__zbozi_shop_rating', [
					'id'            => $key,
					'rating'        => $rating,
					'reviews_count' => $reviewsCount,
					'updated'       => (new DateTime)->format('Y-m-d H:i:s'),
				]);
			}

			$this->zboziCache->clearShopRating($key);
		} catch (Exception $e) {
			Debugger::log($e);
		}
	}

	protected function fetchRating(string $apiKey, string $shopId): float
	{
		$response = $this->fetchResponse("https://api.zbozi.cz/v1/shops/$shopId", $apiKey, $shopId);

		if ($response && Arrays::isJson((string) $response)) {

			$response = Json::decode((string) $response, forceArrays: true);
			if (isset($response['data'][0]['rating'])) {
				return (float) $response['data'][0]['rating'];
			}
		}

		throw new InvalidStateException('Data for Zbozi.cz rating not provided');
	}

	protected function fetchReviewsCount(string $apiKey, string $shopId): ?int
	{
		$response = $this->fetchResponse('https://api.zbozi.cz/v1/shop/reviews?timestampFrom=0&limit=1&offset=0', $apiKey, $shopId);

		if ($response && Arrays::isJson((string) $response)) {

			$response = Json::decode((string) $response, forceArrays: true);
			if (isset($response['meta']['count'])) {
				return (int) $response['meta']['count'];
			}
		}

		throw new InvalidStateException('Data for Zbozi.cz reviews coount not provided');
	}

	/**
	 * @return bool|string
	 */
	protected function fetchResponse(string $url, string $apiKey, string $shopId)
	{
		sleep(2);

		$curl = curl_init();

		$authorization = base64_encode("$shopId:$apiKey");

		curl_setopt_array($curl, [
			CURLOPT_URL            => $url,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING       => '',
			CURLOPT_MAXREDIRS      => 10,
			CURLOPT_TIMEOUT        => 0,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST  => 'GET',
			CURLOPT_HEADER         => false,
			CURLOPT_HTTPHEADER     => [
				'Authorization: Basic ' . $authorization,
			],
		]);

		$response = curl_exec($curl);

		curl_close($curl);

		return $response;
	}

}
