<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Helpers;

use Core\Model\UI\FrontPresenter;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopOrders\FrontModule\Model\Dao\CartItem;
use EshopOrders\Model\Entities\OrderItem;
use MeasuringCodes\FrontModule\Components\DataLayerControl;
use MeasuringCodes\FrontModule\Components\IDataLayerControlFactory;
use MeasuringCodes\FrontModule\Model\Dao\Type;
use MeasuringCodes\FrontModule\Model\TypesList;

class DataLayerEventHelper
{
	public static function getTypeIfAllowed(TypesList $typesList): ?Type
	{
		$type = $typesList->getType('dataLayer');
		if (!$type || !$type->isActive()) {
			return null;
		}

		return $type;
	}

	public static function getComponent(
		FrontPresenter           $presenter,
		IDataLayerControlFactory $factory,
		string                   $eventName,
		Type                     $type,
		?string                  $componentNameSuffix = null,
		bool                     $strictEventName = false
	): ?DataLayerControl
	{
		$component = $factory->create();
		$key       = $type->getKey();
		if ($eventName === 'purchase') {
			$component->useSnippet = true;
			$presenter->addHeadComponent($component, $key . '_' . $eventName . ($componentNameSuffix ? '_' . $componentNameSuffix : ''));
		} else {
			$presenter->addBodyEndComponent(
				$component,
				$strictEventName
					? $eventName
					: ($key . '_' . $eventName . ($componentNameSuffix ? '_' . $componentNameSuffix : ''))
			);
		}

		return $component;
	}

	public static function getItemFromProduct(Product $product): array
	{
		$item = [
			'id'       => (string) $product->getId(),
			'name'     => $product->getName(),
			'category' => $product->defaultCategory->name,
			'price'    => self::formatNumber($product->getPrice()),
			'quantity' => 1,
		];

		if ($product->getManufacturer()) {
			$item['brand'] = $product->getManufacturer()->name;
		}

		return $item;
	}

	public static function getItemPriceFromProduct(Product $product): float
	{
		return self::formatNumber($product->getPrice());
	}

	/**
	 * @param CartItem[]|OrderItem[] $items
	 * @param Product[]              $daoProducts
	 */
	public static function getItemsFromProducts(
		array $items,
		array $daoProducts = [],
		bool  $isVatPayer = false
	): array
	{
		$result = [];

		foreach ($items as $item) {
			$product = $daoProducts[$item->getProductId()] ?? null;

			$result[] = self::getGa4ItemFromProduct($item, $product, $isVatPayer);
		}

		return $result;
	}

	public static function getGa4ItemFromProduct(
		object   $item,
		?Product $product = null,
		bool     $isVatPayer = false
	): array
	{
		if ($item instanceof CartItem) {
			$arr = [
				'item_id'   => $item->getProductId() ? (string) $item->getProductId() : ($item->getProduct() ? (string) $item->getProduct()->getId() : ''),
				'item_name' => $item->title,
				'price'     => self::formatNumber($isVatPayer
					? $item->getPriceWithoutVat()
					: $item->getPrice()
				),
				'quantity'  => $item->getQuantity(),
			];
		} else if ($item instanceof OrderItem) {
			$arr = [
				'item_id'   => $item->getProductId() ? (string) $item->getProductId() : ($item->getProduct() ? $item->getProduct()->getId() : null),
				'item_name' => $item->getOrderItemText() ? $item->getOrderItemText()->getName() : '',
				'price'     => self::formatNumber($isVatPayer
					? $item->getPriceWithoutVat()
					: $item->getPrice()
				),
				'quantity'  => $item->getQuantity(),
			];
		} else if ($item instanceof Product) {
			$arr = [
				'item_id'   => (string) $item->getId(),
				'item_name' => $item->getName(),
				'price'     => self::formatNumber($isVatPayer
					? $item->getPriceWithoutVat()
					: $item->getPrice()
				),
				'quantity'  => 1,
			];

			if (!$product) {
				$product = $item;
			}
		} else {
			return [];
		}

		if ($product) {
			if ($product->getManufacturer()) {
				$arr['item_brand'] = $product->getManufacturer()->name;
			}

			if ($product->defaultCategory) {
				$arr['item_category'] = $product->defaultCategory->name;

				$i = 2;
				foreach ($product->defaultCategory->getParentPath() as $parent) {
					$arr['item_category_' . $i] = $parent->name;

					$i++;
					if ($i > 5) {
						break;
					}
				}
			}
		}

		return $arr;
	}

	/**
	 * @param float|int $price
	 */
	public static function formatNumber($price): float
	{
		$number = str_replace(',', '.', (string) $price);

		return round((float) $number, 2);
	}

}
