<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Helpers;

use Core\FrontModule\Model\ABTest;
use Core\FrontModule\Presenters\BasePresenter;
use EshopOrders\FrontModule\Model\Customers;
use MeasuringCodes\FrontModule\Components\DataLayerControl;
use MeasuringCodes\FrontModule\Components\IDataLayerControlFactory;
use MeasuringCodes\FrontModule\Model\Dao\DefaultType;
use MeasuringCodes\FrontModule\Model\TypesList;
use Nette\DI\Container;
use Users\Model\Security\User;

class DataLayerHelper
{
	protected static bool              $pageViewSent = false;
	protected TypesList                $typesList;
	protected IDataLayerControlFactory $dataLayerEventControlFactory;
	protected ABTest                   $abTest;
	protected User                     $user;
	protected Container                $container;

	protected static ?DataLayerControl $pageViewComponentDataLayer = null;

	public function __construct(
		TypesList                $typesList,
		IDataLayerControlFactory $dataLayerEventControlFactory,
		ABTest                   $abTest,
		User                     $user,
		Container                $container
	)
	{
		$this->typesList                    = $typesList;
		$this->dataLayerEventControlFactory = $dataLayerEventControlFactory;
		$this->abTest                       = $abTest;
		$this->user                         = $user;
		$this->container                    = $container;
	}

	public function isAllowed(): bool
	{
		return DataLayerEventHelper::getTypeIfAllowed($this->typesList) !== null;
	}

	public function sendEvent(string $eventName, array $data, BasePresenter $presenter, ?string $componentNameSuffix = 'dl', bool $strictEventName = false): ?DataLayerControl
	{
		$dlType = DataLayerEventHelper::getTypeIfAllowed($this->typesList);
		if (!$dlType) {
			return null;
		}

		$component = DataLayerEventHelper::getComponent($presenter, $this->dataLayerEventControlFactory, $eventName, $dlType, $componentNameSuffix, $strictEventName);
		if ($component) {
			$component->useSnippet = false;
			$component->varKey     = 'gtag';

			$component->setData([
				$eventName,
				$data,
			]);

			return $component;
		}

		return null;
	}

	public function sendPageView(string $pageType, BasePresenter $presenter): void
	{
		if (self::$pageViewSent) {
			return;
		}

		/** @var DefaultType|null $dlType */
		$dlType = $this->typesList->getType('dataLayer');

		/** @var DefaultType|null $mAnalyticsType */
		$mAnalyticsType = $this->typesList->getType('multihubAnalytics');

		if (!($dlType && $dlType->isActive()) && !($mAnalyticsType && $mAnalyticsType->isActive())) {
			return;
		}

		self::$pageViewSent = true;

		$baseData = [
			'page_title'    => $presenter['title']->getFullTitle(),
			'page_location' => $presenter->getHttpRequest()->getUrl()->getAbsoluteUrl(),
			'page_type'     => $pageType,
		];

		try {
			$activeNav = $presenter->getActiveNavigation();

			if ($activeNav && $activeNav->isHomepage) {
				$pageType = 'homepage';
			}
		} catch (\Exception $e) {
		}

		if ($baseData['page_type'] === 'other' && $pageType !== 'other') {
			$baseData['page_type'] = $pageType;
		}

		if (!$baseData['page_title']) {
			$baseData['page_title'] = $presenter['title']->getFullTitle();
		}

		$abTests = $this->abTest->getActive();
		if ($abTests) {
			$baseData['ab_tests'] = $abTests;
		}

		if ($this->user->isLoggedIn() && $this->container->hasService('eshopOrders.front.customers')) {
			/** @var \Users\Model\Entities\User $user */
			$user = $this->user->getIdentity();

			/** @var Customers $customers */
			$customers = $this->container->getService('eshopOrders.front.customers');

			$customer = $customers->getByUser($user);
			if ($customer) {
				$group = $customer->getGroupCustomers();

				$baseData['customer_group']      = $group->name ?? '';
				$baseData['customer_group_type'] = $group->type ?? 'B2C';
			}
		}

		if (!self::$pageViewComponentDataLayer) {
			if ($dlType && $dlType->isActive()) {
				$componentDL = DataLayerEventHelper::getComponent($presenter, $this->dataLayerEventControlFactory, 'page_view', $dlType);
				if ($componentDL) {
					self::$pageViewComponentDataLayer = $componentDL;
					$componentDL->useSnippet          = false;

					$componentDL->setData($baseData);
				}
			} else if ($mAnalyticsType && $mAnalyticsType->isActive()) {
				$componentDL = DataLayerEventHelper::getComponent($presenter, $this->dataLayerEventControlFactory, 'page_view', $mAnalyticsType);
				if ($componentDL) {
					self::$pageViewComponentDataLayer = $componentDL;
					$componentDL->useSnippet          = false;

					$componentDL->setData($baseData);
				}
			}
		}
	}

}
