<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Blog\FrontModule\Presenters\ArticlesPresenter;
use Blog\FrontModule\Presenters\CategoriesPresenter;
use Core\Model\Event\ControlEvent;
use Core\Model\Helpers\Strings;
use MeasuringCodes\FrontModule\Model\Helpers\DataLayerHelper;
use MeasuringCodes\FrontModule\Model\Helpers\Ga4Helper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;

class BlogSubscriber extends MeasuringCodesEventSubscriber
{
	protected TypesList       $typesList;
	protected Ga4Helper       $ga4Helper;
	protected DataLayerHelper $dataLayerHelper;

	public function __construct(
		TypesList       $typesList,
		Ga4Helper       $ga4Helper,
		DataLayerHelper $dataLayerHelper
	)
	{
		$this->typesList       = $typesList;
		$this->ga4Helper       = $ga4Helper;
		$this->dataLayerHelper = $dataLayerHelper;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			ArticlesPresenter::class . '::renderdetail'     => 'renderDetail',
			CategoriesPresenter::class . '::actioncategory' => 'actionCategory',
		];
	}

	public function renderDetail(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var ArticlesPresenter $presenter */
		$presenter = $event->control;

		$article = $presenter->article;

		if (is_object($article)) {
			$this->ga4Helper->sendEvent(
				'article_view',
				[
					'id'        => $article->getId(),
					'title'     => $article->title,
					'alias'     => $article->alias ?: Strings::webalize((string) $article->title),
					'author'    => $article->getAuthor() ? $article->getAuthor()->getName() : '',
					'published' => $article->publishUp->format('Y-m-d'),
					'category'  => $article->category ? $article->category->title : '',
				],
				$presenter,
			);

			$this->ga4Helper->sendPageView('blog_article', $presenter);
			$this->dataLayerHelper->sendPageView('blog_article', $presenter);
		}
	}

	public function actionCategory(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var CategoriesPresenter $presenter */
		$presenter = $event->control;

		$category = $presenter->category;

		if ($category) {
			$this->ga4Helper->sendPageView('blog_category', $presenter);
			$this->dataLayerHelper->sendPageView('blog_category', $presenter);
		}
	}
}
