<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Core\FrontModule\Presenters\BasePresenter;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\Event;
use Core\Model\Helpers\Strings;
use Core\Model\Http\Session;
use Core\Model\UI\FrontPresenter;
use MeasuringCodes\FrontModule\Model\Helpers\DataLayerHelper;
use MeasuringCodes\FrontModule\Model\Helpers\Ga4Helper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;
use Nette\Http\SessionSection;

class CurrencySubscriber extends MeasuringCodesEventSubscriber
{
	public const SESSION_SECTION = 'currenciesEvents';
	public const CURRENCY_CHANGE = 'change';

	protected EntityManagerDecorator $em;
	protected TypesList              $typesList;
	protected SessionSection         $sessionSection;
	protected Ga4Helper              $ga4Helper;
	protected DataLayerHelper        $dataLayerHelper;

	public function __construct(
		EntityManagerDecorator $em,
		TypesList              $typesList,
		Session                $session,
		Ga4Helper              $ga4Helper,
		DataLayerHelper        $dataLayerHelper
	)
	{
		$this->em              = $em;
		$this->typesList       = $typesList;
		$this->sessionSection  = new SessionSection($session, self::SESSION_SECTION);
		$this->ga4Helper       = $ga4Helper;
		$this->dataLayerHelper = $dataLayerHelper;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.currencyChange'             => 'currencyChange',
			FrontPresenter::class . '::beforeRender' => 'beforeRender',
		];
	}

	public function currencyChange(Event $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		$this->sessionSection->set(self::CURRENCY_CHANGE, [
			'time' => time(),
			'from' => $event->data['from'],
			'to'   => $event->data['to'],
		]);
	}

	public function beforeRender(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var BasePresenter $presenter */
		$presenter = $event->control;

		$currencyChanged = $this->sessionSection->get(self::CURRENCY_CHANGE);
		if ($currencyChanged) {
			$this->ga4Helper->sendEvent(
				'change_currency',
				[
					'from' => Strings::upper($currencyChanged['from']),
					'to'   => Strings::upper($currencyChanged['to']),
				],
				$presenter,
			);
			$this->dataLayerHelper->sendEvent(
				'change_currency',
				[
					'from' => Strings::upper($currencyChanged['from']),
					'to'   => Strings::upper($currencyChanged['to']),
				],
				$presenter,
			);
			$this->sessionSection->remove(self::CURRENCY_CHANGE);
		}
	}
}
