<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Event\ControlEvent;
use Core\Model\Helpers\Strings;
use Core\Model\UI\FrontPresenter;
use Currency\Model\Currencies;
use EshopPackages\FrontModule\Presenters\DefaultPresenter;
use MeasuringCodes\FrontModule\Components\IClarityEventControlFactory;
use MeasuringCodes\FrontModule\Components\IDataLayerControlFactory;
use MeasuringCodes\FrontModule\Components\IFBPixelEventControlFactory;
use MeasuringCodes\FrontModule\Components\IGTagEventControlFactory;
use MeasuringCodes\FrontModule\Components\TypeControlFactory;
use MeasuringCodes\FrontModule\Model\Dao\GoogleAdsType;
use MeasuringCodes\FrontModule\Model\Helpers\DataLayerHelper;
use MeasuringCodes\FrontModule\Model\Helpers\EcoMailHelper;
use MeasuringCodes\FrontModule\Model\Helpers\Ga4Helper;
use MeasuringCodes\FrontModule\Model\Helpers\GTagEventHelper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;
use Nette\DI\Container;

class EshopPackageSubscriber extends MeasuringCodesEventSubscriber
{
	protected TypeControlFactory          $typeControlFactory;
	protected IGTagEventControlFactory    $gTagEventControlFactory;
	protected TypesList                   $typesList;
	protected IFBPixelEventControlFactory $fbPixelEventControlFactory;
	protected IDataLayerControlFactory    $dataLayerControlFactory;
	protected IClarityEventControlFactory $clarityEventControlFactory;
	protected Container                   $container;
	protected Ga4Helper                   $ga4Helper;
	protected EcoMailHelper               $ecoMailHelper;
	protected Translator                  $translator;
	protected DataLayerHelper             $dataLayerHelper;

	public function __construct(
		TypeControlFactory          $typeControlFactory,
		TypesList                   $typesList,
		IGTagEventControlFactory    $gTagEventControlFactory,
		IFBPixelEventControlFactory $fpPixelEvenControlFactory,
		IClarityEventControlFactory $clarityEventControlFactory,
		Container                   $container,
		IDataLayerControlFactory    $dataLayerControlFactory,
		Ga4Helper                   $ga4Helper,
		DataLayerHelper             $dataLayerHelper,
		EcoMailHelper               $ecoMailHelper,
		Translator                  $translator
	)
	{
		$this->typeControlFactory         = $typeControlFactory;
		$this->gTagEventControlFactory    = $gTagEventControlFactory;
		$this->typesList                  = $typesList;
		$this->fbPixelEventControlFactory = $fpPixelEvenControlFactory;
		$this->clarityEventControlFactory = $clarityEventControlFactory;
		$this->container                  = $container;
		$this->dataLayerControlFactory    = $dataLayerControlFactory;
		$this->ga4Helper                  = $ga4Helper;
		$this->dataLayerHelper            = $dataLayerHelper;
		$this->ecoMailHelper              = $ecoMailHelper;
		$this->translator                 = $translator;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			/** @phpstan-ignore-next-line */
			DefaultPresenter::class . '::rendereditablePackage' => 'defaultRenderEditablePackage',
		];
	}

	public function defaultRenderEditablePackage(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var FrontPresenter $presenter */
		$presenter = $event->control;
		$isSk      = $this->translator->getLocale() === 'sk';

		if (Strings::contains((string) $presenter->getParameter('do'), 'cartAddForm') || !property_exists($presenter, 'baseProduct') || !$presenter->baseProduct) {
			return;
		}

		$product = $presenter->baseProduct;

		/** @var Currencies $currencies */
		$currencies = $this->container->getService('currency.currencies');
		$currency   = $currencies->getCurrent()->getCode();

		// GADS
		/** @var GoogleAdsType|null $gAdsType */
		$gAdsType = $this->typesList->getType('googleAds');
		if ($gAdsType && $gAdsType->isActive()) {
			$gAdsComponent = GTagEventHelper::getComponent($presenter, $this->gTagEventControlFactory, 'view_item', $gAdsType);

			if ($gAdsComponent) {
				$data['items'][]  = GTagEventHelper::getItemFromProduct($product);
				$data['send_to']  = $gAdsType->getFullId();
				$data['currency'] = $currency;
				$data['value']    = GTageventHelper::getItemPriceFromProduct($product);

				$gAdsComponent->setData($data);
			}
		}

		// FB pixel
		$fbpType = $this->typesList->getType('facebookPixel');
		if ($fbpType && $fbpType->isActive() && $fbpType->getFieldValue('enableEcommerce')) {
			$fbpComponent = $this->fbPixelEventControlFactory->create($fbpType, 'ViewContent');

			$code3 = [
				'contents'     => [
					[
						'id'       => (string) $product->getId(),
						'quantity' => 1,
					],
				],
				'content_type' => 'product',
				'content_name' => $product->getName(),
				'value'        => $product->getPrice(),
				'currency'     => $currency,
				'condition'    => $product->condition,
			];


			if ($product->defaultCategory) {
				$categoryName = $product->defaultCategory->getParentPathStringFlipped();

				if ($categoryName) {
					$categoryName .= ' > ';
				}

				$categoryName .= $product->defaultCategory->name;

				$code3['content_category'] = trim((string) $categoryName);
			}

			if ($product->getAvailability() && $product->getAvailability()->getOpenGraphText()) {
				$code3['availability'] = $product->getAvailability()->getOpenGraphText();
			}

			$fbpComponent->setCode3($code3);

			$presenter->addBodyEndComponent($fbpComponent, $fbpType->getKey() . 'ViewContent');
		}

		// GA 4
		$this->ga4Helper->sendEvent(
			'view_item',
			[
				'currency' => $currency,
				'value'    => GTagEventHelper::formatNumber($product->getPrice()),
				'items'    => [GTagEventHelper::getGa4ItemFromProduct($product)],
			],
			$presenter,
			'ga4',
		);
		$this->dataLayerHelper->sendEvent(
			'view_item',
			[
				'currency' => $currency,
				'value'    => GTagEventHelper::formatNumber($product->getPrice()),
				'items'    => [GTagEventHelper::getGa4ItemFromProduct($product)],
			],
			$presenter,
			'dl',
		);

		$this->ga4Helper->sendPageView('eshop_product', $presenter);
		$this->dataLayerHelper->sendPageView('eshop_product', $presenter);

		$this->ecoMailHelper->sendEcmProductView(
			(string) $product->id,
			$presenter
		);

		$this->ecoMailHelper->sendStructEvent(
			'produkt',
			'zobrazení',
			'',
			$product->getName(),
			(string) $product->id,
			$presenter
		);

		// Heureka
		$hrqType = $this->typesList->getType('heurekaConversion' . ($isSk ? 'Sk' : ''));
		if ($hrqType && $hrqType->isActive() && $hrqType->getFieldValue('api_key')) {
			$hrqComponent = $hrqType->factory->create($hrqType);
			$hrqComponent->setRenderFile('productDetail');
			$hrqComponent->setParam('isSk', $isSk);

			$presenter->addBodyEndComponent($hrqComponent, $hrqType->getKey() . 'ProductDetail');
		}

		// Clarity
		$clarityType = $this->typesList->getType('clarity');
		if ($clarityType && $clarityType->isActive()) {
			$clarityComponent            = $this->clarityEventControlFactory->create($clarityType);
			$clarityComponent->eventName = 'view_product';
			$clarityComponent->useSnippet(false);

			$presenter->addBodyEndComponent($clarityComponent, $clarityType->getKey() . '_view_product');
		}
	}
}
