<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\ControlEvent;
use Core\Model\Helpers\Strings;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Currency\Model\Currencies;
use Currency\Model\Exchange;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\FrontModule\Presenters\FinishedPresenter;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopOrders\Model\Entities\OrderFlag;
use Heureka\ShopCertification;
use MeasuringCodes\FrontModule\Components\IBingEventControlFactory;
use MeasuringCodes\FrontModule\Components\IClarityEventControlFactory;
use MeasuringCodes\FrontModule\Components\IDataLayerControlFactory;
use MeasuringCodes\FrontModule\Components\IFBPixelEventControlFactory;
use MeasuringCodes\FrontModule\Components\IGTagEventControlFactory;
use MeasuringCodes\FrontModule\Components\TypeControlFactory;
use MeasuringCodes\FrontModule\Components\Zbozi;
use MeasuringCodes\FrontModule\Model\Dao\GoogleAdsType;
use MeasuringCodes\FrontModule\Model\Helpers\DataLayerEventHelper;
use MeasuringCodes\FrontModule\Model\Helpers\DataLayerHelper;
use MeasuringCodes\FrontModule\Model\Helpers\Ga4Helper;
use MeasuringCodes\FrontModule\Model\Helpers\GTagEventHelper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\Helpers\TrackingHelper;
use MeasuringCodes\Model\MeasuringCodesConfig;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;
use Nette\DI\Container;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;
use Tracy\Debugger;
use Users\Model\Security\User;

class OrderFinishedSubscriber extends MeasuringCodesEventSubscriber
{
	protected TypeControlFactory          $typeControlFactory;
	protected TypesList                   $typesList;
	protected TrackingHelper              $trackingHelper;
	protected IGTagEventControlFactory    $gTagEventControlFactory;
	protected IFBPixelEventControlFactory $fbPixelEventControlFactory;
	protected IDataLayerControlFactory    $dataLayerControlFactory;
	protected IBingEventControlFactory    $bingEventControlFactory;
	protected IClarityEventControlFactory $clarityEventControlFactory;
	protected Container                   $container;
	protected Translator                  $translator;
	protected Ga4Helper                   $ga4Helper;
	protected DataLayerHelper             $dataLayerHelper;
	protected EntityManagerDecorator      $em;

	public function __construct(
		TypeControlFactory          $typeControlFactory,
		TypesList                   $typesList,
		TrackingHelper              $trackingHelper,
		IGTagEventControlFactory    $gTagEventControlFactory,
		IFBPixelEventControlFactory $fbPixelEventControlFactory,
		IBingEventControlFactory    $bingEventControlFactory,
		IClarityEventControlFactory $clarityEventControlFactory,
		Container                   $container,
		IDataLayerControlFactory    $dataLayerControlFactory,
		Translator                  $translator,
		Ga4Helper                   $ga4Helper,
		DataLayerHelper             $dataLayerHelper,
		EntityManagerDecorator      $em
	)
	{
		$this->typeControlFactory         = $typeControlFactory;
		$this->typesList                  = $typesList;
		$this->trackingHelper             = $trackingHelper;
		$this->gTagEventControlFactory    = $gTagEventControlFactory;
		$this->fbPixelEventControlFactory = $fbPixelEventControlFactory;
		$this->dataLayerControlFactory    = $dataLayerControlFactory;
		$this->bingEventControlFactory    = $bingEventControlFactory;
		$this->clarityEventControlFactory = $clarityEventControlFactory;
		$this->container                  = $container;
		$this->translator                 = $translator;
		$this->ga4Helper                  = $ga4Helper;
		$this->dataLayerHelper            = $dataLayerHelper;
		$this->em                         = $em;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FinishedPresenter::class . '::beforeRender' => 'orderFinishedBeforeRender',
		];
	}

	/**
	 * Vložení měžících kódů pod dokončení objednávky
	 */
	public function orderFinishedBeforeRender(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed() || !$this->container->hasService('eshopCatalog.front.productsFacade')) {
			return;
		}

		$isSk     = $this->translator->getLocale() === 'sk';
		$allowLog = MeasuringCodesConfig::load('allowLog', false);

		/** @var FinishedPresenter $presenter */
		$presenter = $event->control;
		$order     = $presenter->order;

		if (!$order) {
			return;
		}

		/** @var ProductsFacade $productsFacade */
		$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');

		/** @var Sellers $sellers */
		$sellers    = $this->container->getService('eshopCatalog.front.sellersService') ?? null;
		$isVatPayer = $sellers->getSellerForSite($order->site->getIdent())->dic ? true : false;

		$totalPriceWithVat    = $order->getPriceItemsDiscount();
		$totalPriceWithoutVat = round($totalPriceWithVat - $order->getItemsVatPrice(), 2);

		$discount     = array_values($order->getOrderDiscounts()->toArray())[0] ?? null;
		$discountCode = $discount ? $discount->getCode() : null;

		$orderItems  = $order->getOrderItems();
		$daoProducts = $productsFacade->getProducts(array_map(fn($v) => $v->getProductId(), $orderItems->toArray()));

		$paymentIdent = $order->getPaymentIdent();

		$customer      = $order->getCustomer();
		$addrInvoice   = $order->getAddressInvoice();
		$customerIsB2C = !$addrInvoice;
		if ($customer) {
			/** @var ?GroupCustomers $group */
			$group = $customer->getGroupCustomers();

			if ($group && $group->type === GroupCustomers::typeB2C) {
				$customerIsB2C = true;
			} else if (!$group && (!$customer->addressInvoice || !$customer->addressInvoice->getIdNumber())) {
				$customerIsB2C = true;
			}
		} else if ($addrInvoice && !$addrInvoice->getIdNumber()) {
			$customerIsB2C = true;
		}

		// Zbozi
		$zcType = $this->typesList->getType('zboziConversion');
		if ($zcType->isActive()) {
			/** @var Zbozi\IOrderTrackingFactory $zcFactory */
			$zcFactory   = $zcType->factory;
			$zcComponent = $zcFactory->create($zcType, $presenter->order);
			$presenter->addBodyEndComponent($zcComponent, $zcType->getKey());
		}

		// Didli
		$didliType = $this->typesList->getType('didli');
		if ($didliType->isActive() && $this->trackingHelper->canSend('didli', $order->getId())) {
			$didliComponent = $this->typeControlFactory->create($didliType);
			$didliComponent->setParam('products', $order->getOrderItems());
			$presenter->addBodyEndComponent($didliComponent, $didliType->getKey());

			if (!$didliType->isTurnOff()) {
				$this->trackingHelper->send('didli', $order->getId());
			}
		}

		// Heureka
		$hrqType = $this->typesList->getType('heurekaConversion' . ($isSk ? 'Sk' : ''));
		if ($hrqType && $hrqType->isActive() && $this->trackingHelper->canSend('heurekaConversion', $order->getId())) {
			$hrqComponent = $hrqType->factory->create($hrqType);
			$hrqComponent->setParam('orderId', $order->getId())
				->setParam('scriptLink', $isSk ? 'https://im9.cz/sk/js/ext/2-roi-async.js' : 'https://im9.cz/js/ext/1-roi-async.js')
				->setParam('products', $orderItems)
				->setParam('itemsPrice', $order->getPriceItems(true))
				->setParam('currency', $order->getCurrencyCode());

			$presenter->addBodyEndComponent($hrqComponent, $hrqType->getKey());
			if (!$hrqType->isTurnOff()) {
				$this->trackingHelper->send('heurekaConversion', $order->getId());
			}
		}

		// Heureka overeno
		if ($order->hasFlag(OrderFlag::TYPE_QUESTIONING) && $this->trackingHelper->canSend('heurekaOvereno', $order->getId())) {
			$hrqOverenoType = $this->typesList->getType('heurekaOvereno' . ($isSk ? 'Sk' : ''));
			$key            = $hrqOverenoType ? $hrqOverenoType->getFieldValue('klic') : null;

			if ($hrqOverenoType && $hrqOverenoType->isActive() && $key) {
				if ($allowLog) {
					FileSystem::createDir(LOG_DIR . DS . 'heurekaovereno');
					$logFileName = 'heurekaovereno/' . $order->site->getIdent() . '-' . (new DateTime)->format('Y-m-d');
				}

				try {
					$options = [];
					if ($isSk) {
						$options['service'] = ShopCertification::HEUREKA_SK;
					}

					$shopCertification = new ShopCertification($key, $options);

					$shopCertification->setEmail($order->getAddressInvoice()->getEmail());
					$shopCertification->setOrderId($order->getId());

					foreach ($order->getOrderItems() as $orderItem) {
						$productId = $orderItem->getProductId();
						if ($orderItem->getMoreDataValue('isPackageParent') === true) {
							$productId = 'P' . $orderItem->getProductId();
						}

						if ($orderItem->getProductId()) {
							$shopCertification->addProductItemId((string) $productId);
						}
					}

					$result = $shopCertification->logOrder();
					if ($allowLog) {
						Debugger::log(Json::encode([
							'orderId'     => $order->getId(),
							'email'       => $order->getAddressInvoice()->getEmail(),
							'country'     => $isSk ? 'sk' : 'cz',
							'code'        => $result->code,
							'message'     => $result->message,
							'description' => $result->description,
							'resourceId'  => $result->resourceId,
						]),
							$logFileName
						);
					}

					$this->trackingHelper->send('heurekaOvereno', $order->getId());
				} catch (\Exception $e) {
					$msg = sprintf("Failed export conversion on Heureka: orderId='%s' (%s: %s) - %s",
						$order->getId(), $e->getFile(), $e->getLine(), $e->getMessage());

					LogNotifier::toDevelopers($msg, 'Heureka ověřeno zákazníky ' . ($isSk ? '(SK) ' : '') . '- %siteDomain%');

					if ($allowLog) {
						Debugger::log(Json::encode([
							'error'   => true,
							'orderId' => $order->getId(),
							'email'   => $order->getAddressInvoice()->getEmail(),
							'country' => $isSk ? 'sk' : 'cz',
							'message' => $msg,
						]),
							$logFileName
						);
					}
				}
			}
		}

		// Sklik retargeting
		$skcType = $this->typesList->getType('sklikConversion');
		if ($skcType->isActive() && $this->trackingHelper->canSend('sklikConversion', $order->getId())) {
			$skcComponent = $this->typeControlFactory->create($skcType);
			$skcComponent->setParam('orderValue', $order->getPrice());
			$skcComponent->setParam('orderId', $order->getId());

			$presenter->addBodyEndComponent($skcComponent, $skcType->getKey());
			if (!$skcType->isTurnOff()) {
				$this->trackingHelper->send('sklikConversion', $order->getId());
			}
		}

		// Bing purchase
		$bingType = $this->typesList->getType('bing');
		if ($bingType && $bingType->isActive() && $this->trackingHelper->canSend('bingPurchase', $order->getId())) {
			// Purchase
			$bingComponent            = $this->bingEventControlFactory->create($bingType, 'event');
			$bingComponent->eventName = 'purchase';
			$bingComponent->setData([
				'revenue_value' => $isVatPayer
					? GTagEventHelper::formatNumber($totalPriceWithoutVat)
					: GTagEventHelper::formatNumber($totalPriceWithVat),
				'currency'      => $order->getCurrencyCode(),
			]);

			$presenter->addBodyEndComponent($bingComponent, $bingType->getKey() . '_purchase');
			if (!$bingType->isTurnOff()) {
				$this->trackingHelper->send('bingPurchase', $order->getId());
			}

			// Set user
			if ($order->getAddressInvoice()) {
				$bingComponentSet = $this->bingEventControlFactory->create($bingType, 'set');
				$bingComponentSet->setData([
					'pid' => [
						'em' => $order->getAddressInvoice()->getEmail(),
						'ph' => $order->getAddressInvoice()->getPhone(),
					],
				]);

				$presenter->addBodyEndComponent($bingComponentSet, $bingType->getKey() . '_purchaseUser');
			}
		}

		// clarity purchase
		$clarityType = $this->typesList->getType('clarity');
		if ($clarityType && $clarityType->isActive()) {
			$clarityComponent            = $this->clarityEventControlFactory->create($clarityType);
			$clarityComponent->eventName = 'purchase';
			$clarityComponent->useSnippet(false);

			$presenter->addBodyEndComponent($clarityComponent, $clarityType->getKey() . '_purchase');
		}

		// GA 4
		$gaType = GTagEventHelper::getTypeIfAllowed($this->typesList);
		if ($gaType && $this->ga4Helper->isAllowed() && $this->trackingHelper->canSend('ga4Purchase', $order->getId())) {
			$this->ga4Helper->sendEvent(
				'purchase',
				[
					'currency'       => $order->getDefaultCurrency(),
					'transaction_id' => (string) $order->getId(),
					'value'          => $isVatPayer
						? GTagEventHelper::formatNumber($totalPriceWithoutVat)
						: GTagEventHelper::formatNumber($totalPriceWithVat),
					'coupon'         => $discountCode,
					'shipping'       => $order->getSpedition() ? GTagEventHelper::formatNumber($order->getSpedition()->getPrice()) : 0,
					'tax'            => $isVatPayer
						? GTagEventHelper::formatNumber(round($order->getItemsVatPrice(), 2))
						: '0',
					'items'          => GTagEventHelper::getGa4ItemsFromProducts($orderItems->toArray(), $daoProducts, $isVatPayer),
				],
				$presenter,
			);

			if (!$gaType->isTurnOff()) {
				$this->trackingHelper->send('ga4Purchase', $order->getId());
			}

			$this->ga4Helper->sendPageView('eshop_order_finished', $presenter);
		}

		// DL
		$dlType = DataLayerEventHelper::getTypeIfAllowed($this->typesList);
		if ($dlType && $this->trackingHelper->canSend('dataLayerPurchase', $order->getId())) {
			$this->dataLayerHelper->sendEvent(
				'purchase',
				[
					'currency'       => $order->getDefaultCurrency(),
					'transaction_id' => (string) $order->getId(),
					'value'          => $isVatPayer
						? GTagEventHelper::formatNumber($totalPriceWithoutVat)
						: GTagEventHelper::formatNumber($totalPriceWithVat),
					'coupon'         => $discountCode,
					'shipping'       => $order->getSpedition() ? GTagEventHelper::formatNumber($order->getSpedition()->getPrice()) : 0,
					'tax'            => $isVatPayer
						? GTagEventHelper::formatNumber(round($order->getItemsVatPrice(), 2))
						: '0',
					'items'          => GTagEventHelper::getGa4ItemsFromProducts($orderItems->toArray(), $daoProducts, $isVatPayer),
				],
				$presenter,
			);

			$this->trackingHelper->send('dataLayerPurchase', $order->getId());

			$this->dataLayerHelper->sendPageView('eshop_order_finished', $presenter);
		}

		// Google Ads
		/** @var GoogleAdsType|null $adsType */
		$adsType             = $this->typesList->getType('googleAds');
		$adsDisabledPayments = MeasuringCodesConfig::load('typesList.googleAds.disabledPayments') ?: [];
		if ($adsType && $adsType->isActive() && $this->trackingHelper->canSend('googleAdsConversion', $order->getId()) && !in_array($paymentIdent, $adsDisabledPayments, true)) {
			$merchantId = (string) $adsType->getFieldValue('merchantId');
			if ($merchantId) {
				$adsComponent = $adsType->factory->create($adsType);
			} else {
				$adsComponent = $adsType->factory->create($adsType);
			}

			$addrInvoice = $order->getAddressInvoice();
			if ($addrInvoice) {
				$homeAddress            = [];
				$enhancedConversionData = [
					'first_name' => $addrInvoice->getFirstName(),
					'last_name'  => $addrInvoice->getLastName(),
					'email'      => $addrInvoice->getEmail(),
				];
				$customer               = $order->getCustomer();
				$conversionLabelB2C     = $adsType->getConversionLabelB2C();

				$sendTo = [$adsType->getFullConversionId()];

				if ($conversionLabelB2C && $customerIsB2C) {
					$sendTo[] = $conversionLabelB2C;
				}

				if ($customerIsB2C) {
					$conversionLabelAcqB2C = $adsType->getConversionLabelAcqB2C();
					$conversionLabelRetB2C = $adsType->getConversionLabelRetB2C();

					if ($customer) {
						$ordersCount = $this->em->getConnection()->fetchOne("SELECT COUNT(*) FROM eshop_orders__order 
                		WHERE customer_id = ? AND site_id = ?", [$customer->getId(), $order->site->getIdent()]);
					} else {
						$ordersCount = $this->em->getConnection()->fetchOne("SELECT COUNT(o.id) FROM eshop_orders__order o 
                   		WHERE site_id = ?
                   		INNER JOIN eshop_orders__order_address a ON o.address_invoice_id = a.id AND a.email = ?", [$order->site->getIdent(), $addrInvoice->getEmail()]);
					}

					if ($ordersCount === 1 && $conversionLabelAcqB2C) {
						$sendTo[] = $conversionLabelAcqB2C;
					} else if ($conversionLabelRetB2C) {
						$sendTo[] = $conversionLabelRetB2C;
					}
				}

				$country = $addrInvoice->getCountry() ?: null;

				if ($addrInvoice->getPhone()) {
					$enhancedConversionData['phone_number'] = str_replace(
						[' ', '-', '_'],
						'',
						Strings::phoneFormat($addrInvoice->getPhone(), $country ? $country->getId() : null)
					);
				}

				if ($addrInvoice->getStreet()) {
					$homeAddress['street'] = $addrInvoice->getStreet();
				}

				if ($addrInvoice->getCity()) {
					$homeAddress['city'] = $addrInvoice->getCity();
				}

				if ($addrInvoice->getPostal()) {
					$homeAddress['postal_code'] = str_replace([' ', '-', '_'], '', $addrInvoice->getPostal());
				}

				if ($country) {
					$homeAddress['country'] = $country->getId();
				}

				if (!empty($homeAddress)) {
					$enhancedConversionData['address'] = $homeAddress;
				}

				$adsComponent->setParam('enhancedConversionData', Json::encode($enhancedConversionData));

				if ($merchantId) {
					// Purchase
					$items = [];
					foreach ($orderItems as $item) {
						$items[] = [
							'id'       => $item->getProductId(),
							'quantity' => $item->getQuantity(),
							'price'    => $isVatPayer
								? GTagEventHelper::formatNumber($item->getPriceWithoutVat())
								: GTagEventHelper::formatNumber($item->getPrice()),
						];
					}

					foreach ($order->getOrderDiscounts() as $discount) {
						$items[] = [
							'id'       => $discount->getId(),
							'quantity' => $discount->getQuantity() ?: 1,
							'price'    => GTagEventHelper::formatNumber($discount->getPrice()),
						];
					}

					$adsComponent->setParam('eventName', 'purchase')
						->setParam('data', Json::encode([
							'send_to'          => $sendTo,
							'transaction_id'   => (string) $order->getId(),
							'value'            => $isVatPayer
								? GTagEventHelper::formatNumber($totalPriceWithoutVat)
								: GTagEventHelper::formatNumber($totalPriceWithVat),
							'currency'         => $order->getDefaultCurrency(),
							'aw_merchant_id'   => $merchantId,
							'aw_feed_country'  => 'CZ',
							'aw_feed_language' => 'cs',
							'items'            => $items,
						]));
					$presenter->addBodyEndComponent($adsComponent, 'googleMerchantPurchase');
				} else {
					// Conversion
					$adsComponent->setParam('eventName', 'conversion')
						->setParam('data', Json::encode([
							'send_to'        => $sendTo,
							'value'          => $isVatPayer
								? GTagEventHelper::formatNumber($totalPriceWithoutVat)
								: GTagEventHelper::formatNumber($totalPriceWithVat),
							'currency'       => $order->getDefaultCurrency(),
							'transaction_id' => (string) $order->getId(),
						]));
					$presenter->addBodyEndComponent($adsComponent, 'googleAdsConversion');
				}

				if (!$adsType->isTurnOff()) {
					$this->trackingHelper->send('googleAdsConversion', $order->getId());
				}
			}
		}

		// Persoo
		try {
			$persooType = $this->typesList->getType('persoo');
			if ($persooType && $persooType->isActive() && $this->trackingHelper->canSend('persooTransaction', $order->getId())) {
				/** @var ?Currencies $currencies */
				$currencies = $this->container->hasService('currency.currencies') ? $this->container->getService('currency.currencies') : null;
				/** @var ?Exchange $exchange */
				$exchange = $this->container->hasService('currency.exchange') ? $this->container->getService('currency.exchange') : null;

				$data = [
					'pageType'    => 'basket',
					'currency'    => $order->getCurrencyCode(),
					'basketItems' => [],
					'basketTotal' => $order->getPriceItems(true),
					'transaction' => [
						'id' => $order->getId(),
					],
				];

				foreach (['CZK', 'EUR'] as $curr) {
					if (!isset($currencies->getActive()[$curr]))
						continue;

					if ($order->getDefaultCurrency() === $curr)
						$res = $totalPriceWithVat;
					else if ($order->getCurrencyCode() === $curr)
						$res = $order->getPriceItemsDiscount(true);
					else
						$res = $exchange ? $exchange->change($totalPriceWithVat, $curr) : $totalPriceWithVat;

					$data['transaction']['revenue_' . Strings::lower($curr)] = $res;
				}

				foreach ($orderItems as $item) {
					$arr = [
						'quantity' => $item->getQuantity(),
					];

					foreach (['CZK', 'EUR'] as $curr) {
						if (!isset($currencies->getActive()[$curr]))
							continue;
						$key = 'price_' . Strings::lower($curr);

						if ($order->getDefaultCurrency() === $curr)
							$arr[$key] = $item->getPrice();
						else if ($order->getCurrencyCode() === $curr)
							$arr[$key] = $item->getPrice(true);
						else
							$arr[$key] = $exchange ? $exchange->change($item->getPrice(), $curr) : $item->getPrice();
					}

					$data['basketItems'][$item->getProductId()] = $arr;
				}

				/** @var User $user */
				$user = $presenter->getUser();
				if ($user->isLoggedIn()) {
					/** @var \Users\Model\Entities\User $identity */
					$identity      = $user->getIdentity();
					$data['login'] = $identity->getEmail();
					$data['email'] = $identity->getEmail();
				}

				$dlComponent         = $this->dataLayerControlFactory->create();
				$dlComponent->varKey = 'dataLayerPersoo';
				$dlComponent->setData($data);

				$presenter->addHeadStartComponent($dlComponent, 'persooDlOrderFinished');
				if (!$persooType->isTurnOff()) {
					$this->trackingHelper->send('persooTransaction', $order->getId());
				}
			}
		} catch (\Exception $e) {
		}

		// FB pixel
		$fbpType = $this->typesList->getType('facebookPixel');
		if ($fbpType && $fbpType->isActive() && $fbpType->getFieldValue('enableEcommerce') && $this->trackingHelper->canSend('fbPixelPurchase', $order->getId())) {
			$onlyToB2C = (bool) ($fbpType->getFieldValue('purchaseOnyToB2C') ?: false);

			if (($onlyToB2C && $customerIsB2C) || !$onlyToB2C) {
				$fbComponent = $this->fbPixelEventControlFactory->create($fbpType, 'Purchase');
				$data        = [
					'value'        => (string) $order->getPrice(),
					'currency'     => $order->getDefaultCurrency(),
					'contents'     => [],
					'content_type' => 'product',
				];

				foreach ($orderItems as $item) {
					$data['contents'][] = [
						'id'         => (string) $item->getProductId(),
						'value'      => $item->getPriceTotal(),
						'quantity'   => $item->getQuantity(),
						'item_price' => $item->getPriceTotal(),
					];
				}

				foreach ($order->getOrderDiscounts() as $discount) {
					$data['contents'][] = [
						'id'         => (string) $discount->getCode(),
						'value'      => $discount->getPrice(),
						'quantity'   => 1,
						'item_price' => $discount->getPrice(),
					];
				}

				$fbComponent->setCode3($data);

				$presenter->addBodyEndComponent($fbComponent, $fbpType->getKey() . 'Purchase');
				if (!$fbpType->isTurnOff()) {
					$this->trackingHelper->send('fbPixelPurchase', $order->getId());
				}
			}
		}

		// Trusted shops
		$trustedShopsType = $this->typesList->getType('trustedShops');
		if ($trustedShopsType && $trustedShopsType->isActive() && $this->trackingHelper->canSend('trustedShops', $order->getId())) {
			$trustedShopsComponent = $this->typeControlFactory->create($trustedShopsType);
			$trustedShopsComponent->setParam('order', $order);
			$presenter->addBodyEndComponent($trustedShopsComponent, $trustedShopsType->getKey() . 'OrderFinished');
			$trustedShopsComponent->setRenderFile('orderFinished');

			$this->trackingHelper->send('trustedShops', $order->getId());
		}
	}
}
