<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\Event;
use Core\Model\Helpers\Strings;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Presenters\CustomerPresenter;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderItem;
use MeasuringCodes\FrontModule\Model\Helpers\DataLayerHelper;
use MeasuringCodes\FrontModule\Model\Helpers\Ga4Helper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\EcoMail;
use MeasuringCodes\Model\Helpers\TrackingHelper;
use MeasuringCodes\Model\MeasuringCodesConfig;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;
use Nette\DI\Container;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;
use Soukicz\Zbozicz;
use Tracy\Debugger;

class OrderSubscriber extends MeasuringCodesEventSubscriber
{
	public const SESSION_SECTION = 'eshopOrdersEvents';
	public const ORDER_REPEAT    = 'orderRepeat';

	protected Session                $session;
	protected SessionSection         $orderSuccessMessages;
	protected TypesList              $typesList;
	protected Translator             $translator;
	protected EcoMail                $ecoMail;
	protected Container              $container;
	protected Ga4Helper              $ga4Helper;
	protected DataLayerHelper        $dataLayerHelper;
	protected TrackingHelper         $trackingHelper;
	protected EntityManagerDecorator $em;

	public function __construct(
		Session                $session,
		TypesList              $typesList,
		Translator             $translator,
		EcoMail                $ecoMail,
		Container              $container,
		Ga4Helper              $ga4Helper,
		DataLayerHelper        $dataLayerHelper,
		TrackingHelper         $trackingHelper,
		EntityManagerDecorator $em
	)
	{
		$this->session              = $session;
		$this->orderSuccessMessages = $session->getSection('eshopOrders/orderSuccessMessages');
		$this->typesList            = $typesList;
		$this->translator           = $translator;
		$this->ecoMail              = $ecoMail;
		$this->container            = $container;
		$this->ga4Helper            = $ga4Helper;
		$this->dataLayerHelper      = $dataLayerHelper;
		$this->trackingHelper       = $trackingHelper;
		$this->em                   = $em;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderOnSuccess'                => ['orderOnSuccess', 101],
			CustomerPresenter::class . '::beforeRender' => 'customerBeforeRender',
			'eshopOrders.repeatOrder'                   => 'repeatOrder',
		];
	}

	public function orderOnSuccess(OrderEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		$order    = $event->order;
		$allowLog = MeasuringCodesConfig::load('allowLog', false);

		if ($allowLog) {
			FileSystem::createDir(LOG_DIR . DS . 'measuringcodes');
			$date        = (new DateTime)->format('Y-m-d');
			$logFileName = 'measuringcodes/' . $order->site->getIdent() . '-' . $date;
		}

		$this->trackingHelper->clear();

		if ($allowLog) {
			Debugger::log("Order {$order->getId()} on success start", $logFileName);
		}

		if (!$order->hasFlag(OrderFlag::TYPE_QUESTIONING)) {
			if ($allowLog) {
				Debugger::log("Order {$order->getId()} nepotvrzeno questioning", $logFileName);
			}
		}

		$isSk = $this->translator->getLocale() === 'sk';

		// Zbozi
		$zcType     = $this->typesList->getType('zboziConversion');
		$shopId     = $zcType ? $zcType->getFieldValue('id_provozovny') : null;
		$privateKey = $zcType ? $zcType->getFieldValue('klic') : null;
		if ($zcType && $zcType->isActive() && $shopId && $privateKey) {
			try {
				if ($allowLog) {
					Debugger::log("Order {$order->getId()} try zbozi", $logFileName);
				}

				$client = new Zbozicz\Client($shopId, $privateKey, $zcType->getFieldValue('isSandbox'));

				//radeji pouzijeme kod, ale pokud neni, tak alespon ID
				$spedition = $order->getSpedition();
				$spedId    = null;
				if ($spedition && $spedition->getSpedition()) {
					$spedId = $spedition->getSpedition()->zboziId;
					if (!$spedId) {
						$spedId = $spedition->getSpedition()->getId();
					}
				}

				$payment = $order->getPayment();
				$payId   = null;
				if ($payment && $payment->getPayment()) {
					$payId = $payment->getPayment()->getIdent();
					if (!$payId) {
						$payId = $payment->getPayment()->getId();
					}
				}

				$email = null;
				if ($order->hasFlag(OrderFlag::TYPE_QUESTIONING)) {
					$email = $order->getCustomer() ? $order->getCustomer()->getUser()->getEmail() : null;

					if (!$email && $order->getAddressInvoice()) {
						$email = $order->getAddressInvoice()->getEmail();
					}
				}

				$zboziOrder = new Zbozicz\Order($order->getId());
				$zboziOrder
					->setDeliveryType($spedId)
					->setPaymentType($payId);

				if ($email) {
					$zboziOrder->setEmail($email);
				}

				if ($spedition) {
					$zboziOrder->setDeliveryPrice($spedition->getPrice());
				}

				$otherCosts = 0;

				if ($payment) {
					$otherCosts += $payment->getPrice();
				}

				foreach ($order->getOrderDiscounts()->toArray() as $orderDiscount) {
					$otherCosts += $orderDiscount->getPrice();
				}

				$zboziOrder->setOtherCosts($otherCosts);

				foreach ($order->getOrderItems() as $orderItem) {
					$zboziOrder->addCartItem((new Zbozicz\CartItem)
						->setId($orderItem->getProductId() ? (string) $orderItem->getProductId() : null)
						->setName($orderItem->getOrderItemText() ? $orderItem->getOrderItemText()->getName() : null)
						->setUnitPrice($orderItem->getPrice())
						->setQuantity($orderItem->getQuantity())
					);
				}

				$client->sendOrder($zboziOrder);

				if ($allowLog) {
					Debugger::log("Order {$order->getId()} zbozi sent", $logFileName);
				}

				$this->em->getConnection()->executeStatement("INSERT IGNORE INTO eshop_orders__order_flag (`type`, `state`, order_id) VALUES (:type, :state, :order)", [
					'type'  => OrderFlag::typeZboziConversion,
					'state' => 1,
					'order' => $order->getId(),
				]);
			} catch (\Exception $e) {
				$msg = sprintf("Failed export conversion on Zbozi.cz: orderId='%s' (%s: %s) - %s",
					$order->getId(), $e->getFile(), $e->getLine(), $e->getMessage());

				LogNotifier::toDevelopers($msg, 'Zboží - %siteDomain%');

				if ($allowLog) {
					Debugger::log("Order {$order->getId()} zbozi failed - {$e->getMessage()}", $logFileName);
					Debugger::log($e, $logFileName);
				}
			}
		}

		// Ecomail
		$ecoMailType = $this->typesList->getType('ecoMail');
		if ($ecoMailType && $ecoMailType->isActive()) {
			$appId  = $ecoMailType->getFieldValue('appId');
			$listId = $ecoMailType->getFieldValue('listId');

			if ($appId && $listId) {
				try {
					if ($allowLog) {
						Debugger::log("Order {$order->getId()} try ecomail", $logFileName);
					}

					$ecoMailApi     = new \Ecomail($appId);
					$subscriberData = $order->getCustomer()
						? $this->ecoMail->getSubscriberFromCustomer($order->getCustomer())
						: $this->ecoMail->getSubscriberFromCustomer(null, $order);
					$tags           = [];

					if (MeasuringCodesConfig::load('ecoMail.useTagNewsletter') && $order->hasFlag(OrderFlag::TYPE_NEWSLETTER)) {
						$tags[] = 'newsletter';
					}

					if (MeasuringCodesConfig::load('ecoMail.useTagCategories')) {
						$prodIds = array_map(static fn(OrderItem $v) => $v->getProductId(), $order->getOrderItems()->toArray());

						if (!empty($prodIds)) {
							$prodIds = array_filter($prodIds, static fn($v) => (bool) $v);

							/** @var ProductsFacade $productsFacade */
							$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');

							foreach ($productsFacade->getProducts($prodIds) as $product) {
								if ($product->defaultCategory) {
									$tags[] = str_replace('-', '', Strings::webalize((string) $product->defaultCategory->name));
								}
							}
						}
					}

					if (!empty($tags)) {
						if (!isset($subscriberData['tags'])) {
							$subscriberData['tags'] = $tags;
						} else {
							$subscriberData['tags'] = array_merge($subscriberData['tags'], $tags);
						}
					}

					$this->ecoMail->updateSubscriber(
						$subscriberData['email'],
						$subscriberData,
						false
					);

					if ($allowLog) {
						Debugger::log("Ecomail update subscriber {$order->getId()}", $logFileName);
						Debugger::log(json_encode($subscriberData), $logFileName);
					}

					$r = $ecoMailApi->createNewTransaction($this->ecoMail->getTransactionFromOrder($order));

					if ($allowLog) {
						Debugger::log("Order {$order->getId()} ecomail sent", $logFileName);
						Debugger::log(is_scalar($r) ? $r : json_encode($r), $logFileName);
					}

					$this->em->getConnection()->executeStatement("INSERT IGNORE INTO eshop_orders__order_flag (`type`, `state`, order_id) VALUES (:type, :state, :order)", [
						'type'  => OrderFlag::typeEcomailOrder,
						'state' => 1,
						'order' => $order->getId(),
					]);
				} catch (\Exception $e) {
					$msg = sprintf("Failed export data to ecomail: orderId='%s' (%s: %s) - %s",
						$order->getId(), $e->getFile(), $e->getLine(), $e->getMessage());

					$msg .= ' ' . Json::encode($subscriberData);

					LogNotifier::toDevelopers($msg, 'Ecomail - %siteDomain%');
					if ($allowLog) {
						Debugger::log("Order {$order->getId()} ecomail failed - {$msg}", $logFileName);
						Debugger::log($e, $logFileName);
					}
				}
			}
		}
	}

	public function customerBeforeRender(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var CustomerPresenter $presenter */
		$presenter = $event->control;

		if ($presenter->getAction() === 'orderDetail') {
			$this->ga4Helper->sendEvent(
				'view_order',
				[
					'order_id' => $presenter->getParameter('orderId'),
				],
				$presenter,
			);
			$this->dataLayerHelper->sendEvent(
				'view_order',
				[
					'order_id' => $presenter->getParameter('orderId'),
				],
				$presenter,
			);
		}

		$this->ga4Helper->sendPageView('customer', $presenter);
		$this->dataLayerHelper->sendPageView('customer', $presenter);
	}

	public function repeatOrder(Event $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		$sessionSection = new SessionSection($this->session, self::SESSION_SECTION);
		$sessionSection->set(self::ORDER_REPEAT, [
			'time'    => time(),
			'orderId' => $event->data['orderId'],
		]);
	}
}
