<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Http\Session;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;
use Nette\Http\Request;
use Nette\Http\SessionSection;
use Tracy\Debugger;
use Users\Model\Event\UserEvent;

class RegisterFormSubscriber extends MeasuringCodesEventSubscriber
{
	protected EntityManagerDecorator $em;
	protected TypesList              $typesList;
	protected SessionSection         $sessionSection;
	protected Request                $request;

	public function __construct(EntityManagerDecorator $em, TypesList $typesList, Session $session, Request $request)
	{
		$this->em             = $em;
		$this->typesList      = $typesList;
		$this->sessionSection = new SessionSection($session, UserSubscriber::SESSION_SECTION);
		$this->request        = $request;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'users.registerForm.afterSave' => 'formSuccess',
			'users.userCreated'            => 'userCreated',
		];
	}

	public function formSuccess(UserEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		$this->sessionSection->set(UserSubscriber::USER_REGISTERED, [
			'time'    => time(),
			'pageUrl' => $this->request->getUrl()->getAbsoluteUrl(),
		]);

		$ecoMailType = $this->typesList->getType('ecoMail');
		if ($ecoMailType && $ecoMailType->isActive()) {
			$appId  = $ecoMailType->getFieldValue('appId');
			$listId = $ecoMailType->getFieldValue('listId');
			$user   = $event->user;
			$values = $event->data['formValues'];

			if ($appId && $listId) {
				try {
					$ecoMailApi = new \Ecomail($appId);

					$data = [
						'name'    => $user->getName(),
						'surname' => $user->getLastname(),
						'email'   => $user->getEmail(),
						'company' => null,
						'city'    => null,
						'street'  => null,
						'zip'     => null,
						'country' => null,
						'phone'   => null,
						'tags'    => ['zakaznik'],
					];

					if ($values->newsletterSubscribe) {
						$data['tags'][] = 'newsletter';
					}

					$ecoMailApi->addSubscriber($listId, [
						'subscriber_data'        => $data,
						'trigger_autoresponders' => true,
						'update_existing'        => true,
						'resubscribe'            => true,
						'skip_confirmation'      => true,
					]);
				} catch (\Exception $e) {
					$msg = sprintf("Failed export data to ecomail: user ='%s' (%s: %s) - %s",
						$values->email, $e->getFile(), $e->getLine(), $e->getMessage());

					Debugger::log($msg, 'eshopordersConversion');
					LogNotifier::toDevelopers($msg, 'Ecomail - %siteDomain%');
				}
			}
		}
	}

	public function userCreated(UserEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		$this->sessionSection->set(UserSubscriber::USER_REGISTERED, [
			'time'    => time(),
			'pageUrl' => $this->request->getUrl()->getAbsoluteUrl(),
		]);
	}
}
