<?php declare(strict_types = 1);

namespace MeasuringCodes\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderStatus;
use MeasuringCodes\FrontModule\Model\Dao\Type;
use MeasuringCodes\FrontModule\Model\TypesList;
use Nette\Utils\DateTime;
use Nette\Utils\Json;
use Tracy\Debugger;

class EcoMail
{
	protected EntityManagerDecorator $em;
	protected TypesList              $typesList;
	protected Translator             $translator;

	public function __construct(
		EntityManagerDecorator $em,
		TypesList              $typesList,
		Translator             $translator
	)
	{
		$this->em         = $em;
		$this->typesList  = $typesList;
		$this->translator = $translator;
	}

	protected function getEcoMailApi(): ?\Ecomail
	{
		$ecoMailType = $this->typesList->getType('ecoMail');
		if (!$ecoMailType || !$ecoMailType->isActive() || !$ecoMailType->getFieldValue('appId')) {
			return null;
		}

		return new \Ecomail($ecoMailType->getFieldValue('appId'));
	}

	public function getSubscriberFromCustomer(?Customer $customer, ?Order $order = null): array
	{
		$data = [];
		if ($customer) {
			$addr = $customer->getAddressInvoice();
			$user = $customer->getUser();

			if ($addr) {
				$data = [
					'name'    => $addr->getFirstName() ?: $user->getName(),
					'surname' => $addr->getLastName() ?: $user->lastname,
					'email'   => $user->getEmail(),
					'company' => $addr->getCompany(),
					'city'    => $addr->getCity(),
					'street'  => $addr->getStreet(),
					'zip'     => $addr->getPostal(),
					'country' => $addr->getCountry() ? $addr->getCountry()->getText()->name : 'CZ',
					'phone'   => $addr->phone,
					'tags'    => [],
				];
			}

			if ($customer->getUser()->birthday) {
				$data['birthday'] = $customer->getUser()->birthday->format('Y-m-d');
			}

			$useTagCustomer = MeasuringCodesConfig::load('ecoMail.useTagCustomer');
			$useTagVip      = MeasuringCodesConfig::load('ecoMail.useTagVip');

			$isCustomerActive = $customer->getUser()->getFirstSignIn() !== null;

			if ($isCustomerActive) {
				if ($useTagCustomer || $useTagVip > 0) {
					$ordersCount = $this->em->getConnection()->fetchOne("SELECT COUNT(id) FROM eshop_orders__order WHERE customer_id = " . $customer->getId());

					if ($useTagCustomer) {
						$data['tags'][] = 'zakaznik';
					}

					if ($useTagVip > 0 && $ordersCount >= $useTagVip) {
						$data['tags'][] = 'vip';
					}
				}

				if (MeasuringCodesConfig::load('ecoMail.useTagCustomerGroup') && $customer->getGroupCustomers() && $customer->getGroupCustomers()->short) {
					$data['tags'][] = $customer->getGroupCustomers()->short;
				}
			}

			if (MeasuringCodesConfig::load('ecoMail.useTagCountry') && $addr && $addr->getCountry()) {
				$data['tags'][] = mb_strtolower($addr->getCountry()->getId());
			}

			if (MeasuringCodesConfig::load('ecoMail.useTagNewsletter') && $customer->getUser()->newsletterSubscribed) {
				$data['tags'][] = 'newsletter';
			}
		} else if ($order) {
			$addr = $order->getAddressInvoice();

			if ($addr) {
				$data = [
					'name'    => $addr->getFirstName(),
					'surname' => $addr->getLastName(),
					'email'   => $addr->getEmail(),
					'company' => $addr->getCompany(),
					'city'    => $addr->getCity(),
					'street'  => $addr->getStreet(),
					'zip'     => $addr->getPostal(),
					'country' => $addr->getCountry() ? $addr->getCountry()->getText()->name : 'CZ',
					'phone'   => $addr->getPhone(),
					'tags'    => [],
				];
			}

			if (MeasuringCodesConfig::load('ecoMail.useTagCountry') && $addr->getCountry()) {
				$data['tags'][] = mb_strtolower($addr->getCountry()->getId());
			}
		}

		return $data;
	}

	public function getTransactionFromOrder(Order $order): array
	{
		$timestamp = $order->getCreatedTime()->getTimestamp();

		$transactionItems = [];
		$addr             = $order->getAddressInvoice();

		foreach ($order->getOrderItems()->toArray() as $orderItem) {
			/** @var OrderItem $orderItem */
			$transactionItems[] = [
				'code'      => $orderItem->getProductId(),
				'title'     => $orderItem->getOrderItemText()->getName(),
				'price'     => round((float) $orderItem->getPrice(), 2),
				'amount'    => (int) $orderItem->getQuantity(),
				'timestamp' => $timestamp,
			];
		}

		return [
			'transaction'       => [
				'order_id'  => $order->getIdent(),
				'email'     => $addr->getEmail(),
				'shop'      => $order->site->getIdent(),
				'amount'    => round((float) $order->getPriceItems(), 2),
				'shipping'  => round((float) $order->getPaySpedPrice(), 2),
				'city'      => $addr->getCity(),
				'country'   => $addr->getCountry() ? $addr->getCountry()->getText()->name : 'CZ',
				'timestamp' => $timestamp,
			],
			'transaction_items' => $transactionItems,
		];
	}

	public function checkInactive(int $days, int $listId, string $siteIdent, string $lang, ?string $tag = null): void
	{
		Sites::$currentIdentOverride = $siteIdent;
		Sites::$currentLangOverride  = $lang;
		$this->translator->setLocale($lang);

		$ecoMailType = $this->typesList->getType('ecoMail');
		if (!$ecoMailType || !$ecoMailType->isActive() || !$ecoMailType->getFieldValue('appId')) {
			return;
		}

		if (!$tag) {
			$tag = 'neaktivni';
		}

		$this->em->getConfiguration()->setSQLLogger(null);

		$appId  = $ecoMailType->getFieldValue('appId');
		$today  = (new DateTime());
		$toDate = (new DateTime())->modify('-' . $days . ' days')
			->setTime(0, 0, 0);
		$emails = [];

		foreach ($this->em->getRepository(Order::class)->createQueryBuilder('o')
			         ->select('cAddr.email, os.created')
			         ->innerJoin('o.orderStatuses', 'os', Join::WITH, 'os.status = :status AND os.created >= :toDate')
			         ->innerJoin('o.customer', 'c')
			         ->innerJoin('c.addressInvoice', 'cAddr')
			         ->andWhere('o.site = :site')
			         ->setParameters([
				         'status' => OrderStatus::STATUS_CREATED,
				         'toDate' => $toDate,
				         'site'   => $siteIdent,
			         ])
			         ->orderBy('os.created', 'DESC')
			         ->getQuery()->getArrayResult() as $row) {
			if (isset($emails[$row['email']])) {
				continue;
			}

			$emails[$row['email']] = $row['created'];
		}

		$ecoMailApi = new \Ecomail($appId);
		$i          = 0;

		foreach ($emails as $email => $created) {
			$diff = $today->diff($created);

			if ($diff->days >= $days) {
				/** @var array $data */
				$data = $ecoMailApi->getSubscriber((string) $listId, $email);

				if (isset($data['error'])) {
					foreach ($data['message']->errors as $err) {
						Debugger::log('ERROR - ' . $err, '_ecomailCheckInactive');
					}

					continue;
				}

				if (!isset($data['subscriber'])) {
					Debugger::log($email . ' - Subscriber data not found', '_ecomailCheckInactive');

					continue;
				}

				if (!isset($data['subscriber']['tags'])) {
					$data['subscriber']['tags'] = [];
				}

				if (!in_array($tag, $data['subscriber']['tags'], true)) {
					$data['subscriber']['tags'][] = $tag;
				}

				$ecoMailApi->updateSubscriber((string) $listId, [
					'email'           => $email,
					'subscriber_data' => $data['subscriber'],
				]);
				Debugger::log("Update - set inactive " . $email, '_ecomailCheckInactive');

				$i++;
				if ($i % 30 === 0) {
					sleep(2);
				}
			}
		}
	}

	public function updateSubscriber(string $email, array $data, bool $resubscribe = false, ?string $forceAction = null, bool $skipConfirmation = true): void
	{
		$ecoMailApi = $this->getEcoMailApi();
		/** @var ?Type $ecoMailType */
		$ecoMailType = $this->typesList->getType('ecoMail');

		if (!$ecoMailType) {
			return;
		}

		$listId = $ecoMailType->getFieldValue('listId');

		if (!$ecoMailApi || !$listId) {
			return;
		}

		/** @var array $subscriberData */
		$subscriberData = $data;

		$subscriber = $ecoMailApi->getSubscriber($listId, $email);

		if (isset($subscriber['subscriber'])) {
			$subscriberData['email'] = $subscriber['email'];
			$action                  = 'update';
		} else {
			$action = 'create';
		}

		if ($forceAction) {
			$action = $forceAction;
		}

		if (is_array($subscriberData['tags'])) {
			$subscriberData['tags'] = array_unique($subscriberData['tags']);
		}

		if (!isset($subscriberData['email'])) {
			$subscriberData['email'] = $email;
		}

		if ($action === 'update') {
			$r = $ecoMailApi->updateSubscriber($listId, [
				'email'           => $email,
				'subscriber_data' => $subscriberData,
			]);

			if (isset($r['error'])) {
				Debugger::log($email, '_ecomailUpdateSubscriber');
				Debugger::log(Json::encode($r), '_ecomailUpdateSubscriber');
				Debugger::log(Json::encode($subscriberData), '_ecomailUpdateSubscriber');
			}
		} else {
			$r = $ecoMailApi->addSubscriber($listId, [
				'subscriber_data'        => $subscriberData,
				'trigger_autoresponders' => true,
				'update_existing'        => true,
				'resubscribe'            => $resubscribe,
				'skip_confirmation'      => $skipConfirmation,
			]);

			if (isset($r['error'])) {
				Debugger::log($email, '_ecomailAddSubscriber');
				Debugger::log(Json::encode($r), '_ecomailAddSubscriber');
				Debugger::log(Json::encode($subscriberData), '_ecomailAddSubscriber');
			}
		}
	}

	public function exportCustomers(string $siteIdent, string $lang): void
	{
		Sites::$currentIdentOverride = $siteIdent;
		Sites::$currentLangOverride  = $lang;

		$this->translator->setLocale($lang);

		/** @var ?Type $ecoMailType */
		$ecoMailType = $this->typesList->getType('ecoMail');

		if (!$ecoMailType) {
			return;
		}

		$appId  = $ecoMailType->getFieldValue('appId');
		$listId = $ecoMailType->getFieldValue('listId');

		if (!$listId || !$appId) {
			return;
		}

		$conn = $this->em->getConnection();

		$ordersWithNewsletter = [];
		$customers            = [];
		$useTagNewsletter     = MeasuringCodesConfig::load('ecoMail.useTagNewsletter');

		if ($useTagNewsletter) {
			foreach ($conn->executeQuery("SELECT order_id FROM eshop_orders__order_flag 
                              WHERE `type` = 'agreed_newsletter' AND `state` = 1")->iterateAssociative() as $row) {
				/** @var array $row */
				$ordersWithNewsletter[$row['order_id']] = true;
			}
		}

		foreach ($conn->executeQuery("SELECT o.id, o.customer_id FROM eshop_orders__order o
                    WHERE o.customer_id IS NOT NULL AND o.site_id = ? AND o.lang = ?", [$siteIdent, $lang])->iterateAssociative() as $row) {
			if (!isset($customers[$row['customer_id']])) {
				$customers[$row['customer_id']] = [
					'tags' => [],
				];
			}

			if ($useTagNewsletter && isset($ordersWithNewsletter[$row['id']]) && !in_array('newsletter', $customers[$row['customer_id']]['tags'], true)) {
				$customers[$row['customer_id']]['tags'][] = 'newsletter';
			}
		}

		$i = 0;
		foreach ($customers as $customerId => $data) {
			/** @var ?Customer $customer */
			$customer = $this->em->getRepository(Customer::class)->find($customerId);
			if (!$customer) {
				continue;
			}

			$subscriberData = $this->getSubscriberFromCustomer($customer);

			$subscriberData['tags'] = array_unique(array_merge($subscriberData['tags'], $data['tags']));

			$this->updateSubscriber($customer->getUser()->getEmail(), $subscriberData, false);

			$i++;
			if ($i % 30 === 0) {
				$this->em->clear();
				sleep(2);
			}
		}
	}
}
