<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Helpers;

use Core\FrontModule\Presenters\BasePresenter;
use Exception;
use MeasuringCodes\FrontModule\Components\GTagEventControl;
use MeasuringCodes\FrontModule\Components\IGTagEventControlFactory;
use MeasuringCodes\FrontModule\Model\Dao\GoogleAdsType;
use MeasuringCodes\FrontModule\Model\TypesList;

class Ga4Helper
{
	protected static ?GTagEventControl $pageViewComponentGA4  = null;
	protected static ?GTagEventControl $pageViewComponentGAds = null;

	public function __construct(
		protected TypesList                $typesList,
		protected IGTagEventControlFactory $gTagEventControlFactory,
	)
	{
	}

	public function isAllowed(): bool
	{
		return GTagEventHelper::getTypeGa4IfAllowed($this->typesList) !== null;
	}

	public function sendEvent(
		string        $eventName,
		array         $data,
		BasePresenter $presenter,
		?string       $componentNameSuffix = 'ga4',
	): ?GTagEventControl
	{
		$ga4Type = GTagEventHelper::getTypeGa4IfAllowed($this->typesList);
		if (!$ga4Type) {
			return null;
		}

		$component = GTagEventHelper::getComponent(
			$presenter,
			$this->gTagEventControlFactory,
			$eventName,
			$ga4Type,
			$componentNameSuffix,
		);
		if ($component) {
			$component->useSnippet(false);
			if (!isset($data['send_to'])) {
				$data['send_to'] = $ga4Type->getFieldValue('ga4_id');
			}

			$component->setData($data);

			return $component;
		}

		return null;
	}

	public function sendPageView(string $pageType, BasePresenter $presenter): void
	{
		$baseData = [
			'page_title'    => $presenter['title']->getFullTitle(),
			'page_location' => $presenter->getHttpRequest()->getUrl()->getAbsoluteUrl(),
			'page_type'     => $pageType,
		];

		try {
			$activeNav = $presenter->getActiveNavigation();

			if ($activeNav && $activeNav->isHomepage) {
				$pageType = 'homepage';
			}
		} catch (Exception) {
		}

		if ($baseData['page_type'] === 'other' && $pageType !== 'other') {
			$baseData['page_type'] = $pageType;
		}

		if (!$baseData['page_title']) {
			$baseData['page_title'] = $presenter['title']->getFullTitle();
		}

		if (!self::$pageViewComponentGA4) {
			$ga4Type = GTagEventHelper::getTypeGa4IfAllowed($this->typesList);

			if ($ga4Type) {
				$componentGA4 = GTagEventHelper::getComponent(
					$presenter,
					$this->gTagEventControlFactory,
					'page_view',
					$ga4Type,
				);
				if ($componentGA4) {
					self::$pageViewComponentGA4 = $componentGA4;
					$componentGA4->useSnippet(false);

					$componentGA4->setData(
						$baseData + [
							'send_to' => $ga4Type->getFieldValue('ga4_id'),
						],
					);
				}
			}
		}

		if (!self::$pageViewComponentGAds) {
			/** @var GoogleAdsType|null $gAdsType */
			$gAdsType = $this->typesList->getType('googleAds');

			if ($gAdsType && $gAdsType->isActive()) {
				$componentGAds = GTagEventHelper::getComponent(
					$presenter,
					$this->gTagEventControlFactory,
					'page_view',
					$gAdsType,
				);
				if ($componentGAds) {
					self::$pageViewComponentGAds = $componentGAds;
					$componentGAds->useSnippet(false);

					$componentGAds->setData(
						$baseData + [
							'send_to' => $gAdsType->getFullId(),
						],
					);
				}
			}
		}
	}

}
