<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Core\FrontModule\Presenters\BasePresenter;
use Core\Model\Event\ControlEvent;
use Core\Model\Http\Session;
use Core\Model\UI\FrontPresenter;
use Currency\Model\Currencies;
use EshopCatalog\DI\EshopCatalogExtension;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopOrders\FrontModule\Model\CartFacade;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\FrontModule\Presenters\DefaultPresenter;
use MeasuringCodes\FrontModule\Components\GTagEventControl;
use MeasuringCodes\FrontModule\Components\IDataLayerControlFactory;
use MeasuringCodes\FrontModule\Components\IEcoTrackControlFactory;
use MeasuringCodes\FrontModule\Components\IFBPixelEventControlFactory;
use MeasuringCodes\FrontModule\Components\IGTagEventControlFactory;
use MeasuringCodes\FrontModule\Model\Helpers\DataLayerHelper;
use MeasuringCodes\FrontModule\Model\Helpers\Ga4Helper;
use MeasuringCodes\FrontModule\Model\Helpers\GTagEventHelper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesConfig;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;
use Nette\DI\Container;
use Nette\Http\SessionSection;

class EshopCartSubscriber extends MeasuringCodesEventSubscriber
{
	protected array $cartChanges = [
		'added'   => [],
		'updated' => [],
	];

	protected array $orderChanges = [];

	public function __construct(
		public Container                      $container,
		protected TypesList                   $typesList,
		protected IGTagEventControlFactory    $gTagEventControlFactory,
		protected IFBPixelEventControlFactory $fbPixelEventControlFactory,
		protected IDataLayerControlFactory    $dataLayerControlFactory,
		protected Session                     $session,
		protected Ga4Helper                   $ga4Helper,
		protected IEcoTrackControlFactory     $ecoTrackControlFactory,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FrontPresenter::class . '::beforeRender'  => 'beforeRender',
			'eshopOrders.cartAddItem'                 => 'onAddItem',
			'eshopOrders.cartUpdateItem'              => 'onUpdateItem',
			DefaultPresenter::class . '::renderorder' => 'renderOrder',
			'eshopOrders.saveOrderFormDataStep2'      => 'orderSaveStep2',
		];
	}

	public function beforeRender(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed() || !$this->container->hasService('eshopCatalog.front.productsFacade')) {
			return;
		}

		if (!MeasuringCodesConfig::load('allowCartEvents')) {
			return;
		}

		/** @var BasePresenter $presenter */
		$presenter = $event->control;

		/** @var ProductsFacade $productsFacade */
		$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');

		/** @var Currencies $currencies */
		$currencies = $this->container->getService('currency.currencies');
		$currency   = $currencies->getCurrent()->getCode();

		$items = [
			'add_to_cart'      => [],
			'remove_from_cart' => [],
		];

		foreach ($this->cartChanges['added'] as $id => $change) {
			$items['add_to_cart'][$id] = $change;
		}

		foreach ($this->cartChanges['updated'] as $id => $change) {
			if ($change > 0) {
				$items['add_to_cart'][$id] = $change;
			} else {
				$items['remove_from_cart'][$id] = abs($change);
			}
		}

		foreach ($items as $eventName => $v) {
			// GA
			$component   = $this->getComponent($presenter, $eventName);
			$dlComponent = $this->dataLayerControlFactory->create();
			$dlEventName = $eventName === 'add_to_cart' ? 'addToCart' : 'removeFromCart';
			$presenter->addBodyEndComponent($dlComponent, 'dl' . $dlEventName);
			$dlComponent->useSnippet = true;

			$data   = [];
			$dataDl = [];
			foreach ($v as $id => $quantity) {
				$id      = (int) $id;
				$product = $productsFacade->getProduct($id);

				if ($product) {
					$tmp             = GTagEventHelper::getItemFromProduct($product);
					$tmp['quantity'] = $quantity;

					$data[] = $tmp;

					$tmp             = DataLayerHelper::getProduct($product);
					$tmp['quantity'] = $quantity;
					$dataDl[]        = $tmp;
				}
			}

			if ($component && $data) {
				$component->setData([
					'items'   => $data,
					'send_to' => $component->getType()->getFieldValue('ga_id'),
				]);
			}
			if (!empty($dataDl)) {
				$dlComponent->setData([
					'event'     => $dlEventName,
					'ecommerce' => [
						'currencyCode' => $currency,
						'add'          => [
							'products' => $dataDl,
						],
					],
				]);
			}

			// GA 4
			if ($this->ga4Helper->isAllowed()) {
				$gaItems = [];
				$value   = 0;

				if (!empty($v)) {
					foreach ($v as $id => $quantity) {
						$product = $productsFacade->getProduct((int) $id);

						if ($product) {
							$item             = GTagEventHelper::getGa4ItemFromProduct($product);
							$item['quantity'] = $quantity;
							$value            += $item['price'] * $item['quantity'];
							$gaItems[]        = $item;
						}
					}
				}

				$evName       = $eventName === 'add_to_cart' ? 'add_to_cart' : 'remove_from_cart';
				$ga4Component = $this->getGa4Component($presenter, $evName, 'ga4');
				if ($ga4Component && !empty($gaItems)) {
					$ga4Component->setData([
						'currency' => $currency,
						'value'    => GTagEventHelper::formatNumber($value),
						'items'    => $gaItems,
						'send_to'  => $ga4Component->getType()->getFieldValue('ga4_id'),
					]);
				}
			}

			// Ajax
			if ($presenter->isAjax()) {
				if ($component) {
					$component->redrawControl('wrap');
				}

				$dlComponent->redrawControl('wrap');

				if (isset($ga4Component)) {
					$ga4Component->redrawControl('wrap');
				}
			}
		}

		// FB pixel
		$fbpPixel = $this->typesList->getType('facebookPixel');
		if ($fbpPixel->isActive() && $fbpPixel->getFieldValue('enableEcommerce')) {
			$fbpComponent = $this->fbPixelEventControlFactory->create($fbpPixel, 'AddToCart');
			$fbpComponent->showOnlyIfCode3();
			$presenter->addBodyEndComponent($fbpComponent, $fbpPixel->getKey() . 'AddToCart');

			if (!empty($items['add_to_cart'])) {
				$totalValue = 0;
				$data       = [
					'currency'     => $currency,
					'content_type' => 'product',
					'contents'     => [],
				];

				foreach ($items['add_to_cart'] as $id => $quantity) {
					if (!$id) {
						continue;
					}

					$product    = $productsFacade->getProduct((int) $id);
					$price      = round($product->getPrice() * $quantity, 2);
					$totalValue += $price;

					$data['contents'][] = [
						'id'       => (string) $id,
						'quantity' => $quantity,
					];
				}

				$data['value'] = $totalValue;

				$fbpComponent->setCode3($data);

				if ($presenter->isAjax()) {
					$fbpComponent->redrawControl('wrap');
				}
			}
		}

		$eshopOrdersSession = new SessionSection($this->session, OrderSubscriber::SESSION_SECTION);
		$repeatOrder        = $eshopOrdersSession->get(OrderSubscriber::ORDER_REPEAT);
		if ($repeatOrder) {
			$this->ga4Helper->sendEvent(
				'repeat_order',
				[
					'order_id' => $repeatOrder['orderId'],
				],
				$presenter,
			);

			$eshopOrdersSession->remove(OrderSubscriber::ORDER_REPEAT);
		}

		// Ecomail cart tracking
		if (
			MeasuringCodesConfig::load('typesList.ecoMail.enableCartTracking')
			&& $this->container->hasService('eshopOrders.front.cartFacade')
		) {
			/** @var CartFacade $cartFacade */
			$cartFacade  = $this->container->getService('eshopOrders.front.cartFacade');
			$ecoMailType = $this->typesList->getType('ecoMail');

			if ($ecoMailType && $ecoMailType->isActive()) {
				$ecoMailCartComponent             = $this->ecoTrackControlFactory->create($ecoMailType);
				$ecoMailCartComponent->useSnippet = true;

				$data = [
					'schema' => '',
					'data'   => [
						'action'   => 'Basket',
						'products' => [],
					],
				];

				$i = 0;
				foreach ($cartFacade->getCart()->getCartItems() as $item) {
					$row = [
						'productId' => $item->id,
						'url'       => $item->getLink(),
						'name'      => $item->title,
						'price'     => $item->getPrice(),
					];

					if ($item->getImage()) {
						$row['img_url'] = $presenter->baseUrl . ltrim($item->getImage()->getFilePath(), '/');
					}

					if ($item->getProduct()) {
						$row['description'] = $item->getProduct()->getDescription();
					}

					$data['data']['products'][] = $row;
					$i++;

					if ($i > 10) {
						break;
					}
				}

				$ecoMailCartComponent->eventName = 'trackUnstructEvent';
				$ecoMailCartComponent->data      = $data;

				$presenter->addBodyEndComponent($ecoMailCartComponent, $ecoMailType->getKey() . 'Cart');

				if ($presenter->isAjax()) {
					$ecoMailCartComponent->redrawControl('wrap');
				}
			}
		}
	}

	public function onAddItem(AddedCartItemEvent $item): void
	{
		$this->cartChanges['added'][$item->item->productId] = $item->item->quantity;
	}

	public function onUpdateItem(UpdatedCartItemEvent $item): void
	{
		$this->cartChanges['updated'][$item->productId] = $item->quantity - $item->beforeQuantity;
	}

	public function renderOrder(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var DefaultPresenter $presenter */
		$presenter = $event->control;
		/** @var ProductsFacade $productsFacade */
		$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');

		/** @var Currencies $currencies */
		$currencies = $this->container->getService('currency.currencies');
		$currency   = $currencies->getCurrent()->getCode();

		$cart           = $presenter->cartsService->getCurrentCart();
		$cartItemsPrice = $cart->getCartItemsPrice();

		$discount     = array_values($cart->discounts)[0] ?? null;
		$discountCode = $discount ? $discount->id : null;

		$cartItems       = $cart->getCartItems();
		$cartDaoProducts = $productsFacade->getProducts(array_map(fn($v) => $v->getProductId(), $cartItems));

		// InitiateCheckout
		$fbpType = $this->typesList->getType('facebookPixel');
		if ($presenter->getParameter('do') === null && empty($presenter->getHttpRequest()->getPost())) {
			if ($fbpType->isActive() && $fbpType->getFieldValue('enableEcommerce')) {
				$fbpComponent = $this->fbPixelEventControlFactory->create($fbpType, 'InitiateCheckout', 'track');

				$code3 = [
					'currency'  => $currency,
					'num_items' => count($cartItems),
					'value'     => $cartItemsPrice,
				];

				$fbpComponent->setCode3($code3);

				$presenter->addBodyEndComponent($fbpComponent, $fbpType->getKey() . 'InitiateCheckout');
			}
		}

		// GA
		$speditionComponent    = $this->getComponent($presenter, 'set_checkout_option', 'spedition');
		$paymentComponent      = $this->getComponent($presenter, 'set_checkout_option', 'payment');
		$ga4PaymentComponent   = $this->getGa4Component($presenter, 'add_payment_info');
		$ga4SpeditionComponent = $this->getGa4Component($presenter, 'add_shipping_info');

		$spedDlComponent = $this->dataLayerControlFactory->create();
		$presenter->addBodyEndComponent($spedDlComponent, 'dlCheckoutOptionSped');
		$spedDlComponent->useSnippet = true;

		$payDlComponent = $this->dataLayerControlFactory->create();
		$presenter->addBodyEndComponent($payDlComponent, 'dlCheckoutOptionPay');
		$payDlComponent->useSnippet = true;

		// FB
		$fbPaymentComponent = null;
		if ($fbpType->isActive() && $fbpType->getFieldValue('enableEcommerce')) {
			$fbPaymentComponent = $this->fbPixelEventControlFactory->create($fbpType, 'AddPaymentInfo', 'track');
			$fbPaymentComponent->showOnlyIfCode3();
			$presenter->addBodyEndComponent($fbPaymentComponent, $fbpType->getKey() . 'AddPaymentInfo');
		}

		if ($presenter->getParameter('do') !== null) {
			// Nastaveni dopravy
			if (isset($this->orderChanges['spedition'])) {
				/** @var Speditions $speditions */
				$speditions = $this->container->getService('eshopOrders.front.speditions');
				$spedition  = $speditions->get((int) $this->orderChanges['spedition']);

				if ($speditionComponent) {
					$speditionComponent->setData([
						'checkout_step'   => 1,
						'checkout_option' => 'shipping method',
						'value'           => $spedition->getName(),
						'send_to'         => $speditionComponent->getType()->getFieldValue('ga_id'),
					]);
				}

				$spedDlComponent->setData([
					'event'     => 'checkoutOption',
					'ecommerce' => [
						'checkout_option' => [
							'actionField' => [
								'step'   => 1,
								'option' => $spedition->getName(),
							],
						],
					],
				]);

				//GA 4
				if ($ga4SpeditionComponent) {
					$arr = [
						'currency'     => $currency,
						'value'        => GTagEventHelper::formatNumber($cartItemsPrice),
						'payment_type' => $spedition->getName(),
						'items'        => GTagEventHelper::getGa4ItemsFromProducts($cartItems, $cartDaoProducts),
						'send_to'      => $ga4SpeditionComponent->getType()->getFieldValue('ga4_id'),
					];

					if ($discountCode) {
						$arr['coupon'] = $discountCode;
					}

					$ga4SpeditionComponent->setData($arr);
				}
			}

			// Nastaveni platby
			if (isset($this->orderChanges['payment'])) {
				/** @var Payments $payments */
				$payments = $this->container->getService('eshopOrders.front.payments');
				$payment  = $payments->get((int) $this->orderChanges['payment']);

				if ($paymentComponent) {
					$paymentComponent->setData([
						'checkout_step'   => 2,
						'checkout_option' => 'payment method',
						'value'           => $payment->getName(),
						'send_to'         => $paymentComponent->getType()->getFieldValue('ga_id'),
					]);
				}

				$payDlComponent->setData([
					'event'     => 'checkoutOption',
					'ecommerce' => [
						'checkout_option' => [
							'actionField' => [
								'step'   => 2,
								'option' => $payment->getName(),
							],
						],
					],
				]);

				// FB
				if ($fbPaymentComponent) {
					$code3 = [
						'content_type' => 'product',
						'content_ids'  => array_values(array_map(fn($row) => (string) $row->id, $cartItems)),
						'content_name' => array_values(array_map(fn($row) => $row->title, $cartItems)),
						'value'        => $cartItemsPrice,
						'currency'     => $currency,
					];

					$fbPaymentComponent->setCode3($code3);
				}

				//GA 4
				if ($ga4PaymentComponent) {
					$arr = [
						'currency'     => $currency,
						'value'        => GTagEventHelper::formatNumber($cartItemsPrice),
						'payment_type' => $payment->getName(),
						'items'        => GTagEventHelper::getGa4ItemsFromProducts($cartItems, $cartDaoProducts),
						'send_to'      => $ga4PaymentComponent->getType()->getFieldValue('ga4_id'),
					];

					if ($discountCode) {
						$arr['coupon'] = $discountCode;
					}

					$ga4PaymentComponent->setData($arr);
				}
			}

			if ($presenter->isAjax()) {
				if ($speditionComponent)
					$speditionComponent->redrawControl('wrap');
				if ($paymentComponent)
					$paymentComponent->redrawControl('wrap');
				$spedDlComponent->redrawControl('wrap');
				$payDlComponent->redrawControl('wrap');
				if ($fbPaymentComponent)
					$fbPaymentComponent->redrawControl('wrap');
				if ($ga4SpeditionComponent)
					$ga4SpeditionComponent->redrawControl('wrap');
				if ($ga4PaymentComponent)
					$ga4PaymentComponent->redrawControl('wrap');
			}

			return;
		}

		// Vytvoreni transakce
		$component   = $this->getComponent($presenter, 'begin_checkout');
		$dlComponent = $this->dataLayerControlFactory->create();
		$presenter->addBodyEndComponent($dlComponent, 'dlCheckout');
		$ga4Component = $this->getGa4Component($presenter, 'begin_checkout', 'ga4');
		$items        = [];
		$dlItems      = [];
		$ga4Items     = [];

		if (!$component) {
			return;
		}

		foreach ($cartItems as $item) {
			$product = $cartDaoProducts[$item->getProductId()] ?? null;

			if ($product) {
				$tmp             = GTagEventHelper::getItemFromProduct($product);
				$tmp['quantity'] = (int) $item->getQuantity();
				$tmp['price']    = GTagEventHelper::formatNumber($item->getPrice());

				$items[] = $tmp;

				$tmp             = DataLayerHelper::getProduct($product);
				$tmp['quantity'] = $item->getQuantity();
				$dlItems[]       = $tmp;
			}

			$ga4Items[] = GTagEventHelper::getGa4ItemFromProduct($item, $product);
		}

		if ($items) {
			$component->setData([
				'items'   => $items,
				'coupon'  => '',
				'send_to' => $component->getType()->getFieldValue('ga_id'),
			]);
		}

		$dlComponent->setData([
			'event'     => 'checkout',
			'ecommerce' => [
				'checkout' => [
					'actionField' => [
						'step'   => 1,
						'option' => '',
					],
					'products'    => $dlItems,
				],
			],
		]);

		if ($ga4Component && $ga4Items) {
			$ga4Component->setData([
				'currency' => $currency,
				'value'    => GTagEventHelper::formatNumber($cartItemsPrice),
				'coupon'   => $discountCode,
				'items'    => $ga4Items,
				'send_to'  => $ga4Component->getType()->getFieldValue('ga4_id'),
			]);
		}
	}

	public function orderSaveStep2(SaveOrderFormDataEvent $event): void
	{
		$this->orderChanges = [
			'spedition' => $event->data['spedition'],
			'payment'   => $event->data['payment'],
		];
	}

	protected function getComponent(
		BasePresenter $presenter,
		string        $eventName,
		?string       $componentNameSuffix = null,
	): ?GTagEventControl
	{
		$type = GTagEventHelper::getTypeIfAllowed($this->typesList);
		if (!$type) {
			return null;
		}

		return GTagEventHelper::getComponent(
			$presenter,
			$this->gTagEventControlFactory,
			$eventName,
			$type,
			$componentNameSuffix,
		);
	}

	protected function getGa4Component(
		BasePresenter $presenter,
		string        $eventName,
		?string       $componentNameSuffix = null,
	): ?GTagEventControl
	{
		if (!class_exists(EshopCatalogExtension::class) || !MeasuringCodesConfig::load(
				'typesList.googleAnalytics.enableGa4',
			)) {
			return null;
		}

		$type = $this->typesList->getType('googleAnalytics');
		if (!$type || !$type->isActive() || !$type->getFieldValue('ga4_id')) {
			return null;
		}

		return GTagEventHelper::getComponent(
			$presenter,
			$this->gTagEventControlFactory,
			$eventName,
			$type,
			$componentNameSuffix,
		);
	}
}
