<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Core\FrontModule\Presenters\BasePresenter;
use Core\Model\Application\AppState;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\SystemConfig;
use Core\Model\UI\FrontPresenter;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Components\ProductsList;
use EshopCatalog\FrontModule\Model\Helpers\ShowedProducts;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Presenters\DefaultPresenter as EshopCatalogDefaultPresenter;
use EshopCatalog\Model\Helpers\Helper;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Customers;
use Exception;
use Forms\DI\FormsExtension;
use Forms\FrontModule\Components\FormControl;
use Forms\FrontModule\Model\Dao;
use Forms\Model\Entities\Form;
use Forms\Model\Entities\Submission;
use MeasuringCodes\FrontModule\Components\DataLayerControl;
use MeasuringCodes\FrontModule\Components\DefaultTypeControl;
use MeasuringCodes\FrontModule\Components\GTagEventControl;
use MeasuringCodes\FrontModule\Components\IDataLayerControlFactory;
use MeasuringCodes\FrontModule\Components\IGTagEventControlFactory;
use MeasuringCodes\FrontModule\Components\TypeControlFactory;
use MeasuringCodes\FrontModule\Model\Dao\GoogleAdsType;
use MeasuringCodes\FrontModule\Model\Helpers\Ga4Helper;
use MeasuringCodes\FrontModule\Model\Helpers\GTagEventHelper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;
use Nette\DI\Container;
use Nette\Http\Request;
use Users\Model\Security\User;

class FrontPresenterSubscriber extends MeasuringCodesEventSubscriber
{
	public function __construct(
		protected TypeControlFactory       $typeControlFactory,
		protected TypesList                $typesList,
		protected Container                $container,
		protected IGTagEventControlFactory $gTagEventControlFactory,
		protected IDataLayerControlFactory $dataLayerControlFactory,
		protected Request                  $request,
		protected Ga4Helper                $ga4Helper,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FrontPresenter::class . '::startup'                  => 'startup',
			FrontPresenter::class . '::beforeRender'             => 'beforeRender',
			FrontPresenter::class . '::getBodyEndComponentsPack' => 'beforeRenderBodyEnd',
			FormControl::class . '::create'                      => 'formControlCreate',
			FormControl::class . '::formSuccess'                 => 'formFormControlSuccess',
		];
	}

	public function startup(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var BasePresenter $presenter */
		$presenter = $event->control;

		if (class_exists(FormsExtension::class)) {
			$dlComponent             = $this->dataLayerControlFactory->create();
			$dlComponent->useSnippet = true;
			$presenter->addBodyEndComponent($dlComponent, 'dlFormSend');
		}

		$presenter->onRender[] = function($presenter) {
			$this->ga4Helper->sendPageView('other', $presenter);
		};
	}

	/**
	 * Vložení obecných kódů na všechny stránky
	 */
	public function beforeRender(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var BasePresenter $presenter */
		$presenter = $event->control;

		if (class_exists(ShowedProducts::class)) {
			if ($presenter->getParameter('do') === 'cartAddForm-form-submit') {
				return;
			}
			$action = $presenter->getAction(true);

			// Persoo
			$persooType = $this->typesList->getType('persoo');
			if ($persooType && $persooType->isActive() && $action !== ':EshopOrders:Front:Finished:orderFinished') {
				$data = [];

				/** @var ProductsList|null $productsList */
				$productsList = $presenter->getComponent('list', false);
				if ($productsList && $productsList instanceof ProductsList) {
					foreach ($productsList->getProducts() as $prod)
						$data['impressedProducts']['locationMain'][] = (string) $prod->getId();
				}

				if ($this->container->hasService('currency.currencies')) {
					/** @var Currencies $currencies */
					$currencies = $this->container->getService('currency.currencies');

					$data['currency'] = $currencies->getCurrent()->getCode();
				}

				try {
					$activeNav   = $presenter->getActiveNavigation();
					$addCategory = false;

					if ($action === ':EshopCatalog:Front:Default:product' && $presenter instanceof EshopCatalogDefaultPresenter && $presenter->product) {
						$data['pageType'] = 'detail';
						$data['itemID']   = $presenter->product->getId();

						if ($presenter->product->getManufacturer())
							$data['brand'] = $presenter->product->getManufacturer()->name;

						$addCategory = true;
					} else if ($action === ':EshopCatalog:Front:Default:category' && $presenter instanceof EshopCatalogDefaultPresenter && $presenter->category) {
						$data['pageType'] = 'category';
						$addCategory      = true;
					} else if ($action === ':EshopCatalog:Front:Default:search') {
						$data['pageType'] = 'search';
					} else if ($action === ':EshopOrders:Front:Default:order') {
						$data['pageType'] = 'basket';
					} else if ($activeNav->isHomepage) {
						$data['pageType'] = 'homepage';
					} else {
						$data['pageType'] = 'other';
					}

					if ($addCategory) {
						if (isset($activeNav->componentParams['virtualCategory'])) {
							$data['categoryID'] = 'v' . $activeNav->componentParams['virtualCategory'];
							$data['hierarchy']  = 'v' . $activeNav->componentParams['virtualCategory'];
						} else if (isset($presenter->category) && $presenter->category) {
							$data['categoryID'] = $presenter->category->getId();

							$catPath = array_keys($presenter->category->getParentPath()) ?: [];
							if ($catPath) {
								$catPath = array_reverse($catPath);
							}

							$catPath[]         = $presenter->category->getId();
							$data['hierarchy'] = implode(':', $catPath);
						}
					}

					if ($this->container->hasService('eshopOrders.carts')) {
						/** @var Carts $cartService */
						$cartService = $this->container->getService('eshopOrders.carts');
						$cart        = $cartService->getCurrentCart();

						if ($cart->getCartItems()) {
							$data['basketItems'] = [];
							foreach ($cart->getCartItems() as $item) {
								$data['basketItems'][$item->getProductId()] = [
									'quantity' => $item->getQuantity(),
									'price'    => $item->getPrice(),
								];
							}
						}
						$data['basketTotal'] = $cart->getPriceTotal();
					}
				} catch (Exception) {
				}

				/** @var User $user */
				$user = $presenter->getUser();
				if ($user->isLoggedIn()) {
					/** @var \Users\Model\Entities\User $identity */
					$identity      = $user->getIdentity();
					$data['login'] = $identity->getEmail();
					$data['email'] = $identity->getEmail();

					try {
						if ($this->container->hasService('eshopOrders.front.customers')) {
							/** @var Customers $customers */
							$customers = $this->container->getService('eshopOrders.front.customers');
							$customer  = $customers->getByUser($identity);

							if ($customer && $customer->getGroupCustomers()) {
								$data['priceLevel'] = $customer->getGroupCustomers()->getId();

								$productsSale = $customer->getGroupCustomers()->getProductsSale();
								if ($productsSale && $productsSale > 0)
									$data['userPercentSale'] = (float) $productsSale;
							}
						}
					} catch (Exception) {
					}

					if (!isset($data['userPercentSale']) && $this->container->hasService('eshopCatalog.helper')) {
						/** @var Helper $eshopCatalogHelper */
						$eshopCatalogHelper      = $this->container->getService('eshopCatalog.helper');
						$percentSale             = $eshopCatalogHelper->getRegisterUserSale($identity);
						$data['userPercentSale'] = (float) $percentSale;
					}
				}

				$dlComponent         = $this->dataLayerControlFactory->create();
				$dlComponent->varKey = 'dataLayerPersoo';
				$dlComponent->setData($data);

				$presenter->addHeadComponent($dlComponent, 'persooDl');
			}
		}

		// Vlozeni obecnych na kazdou stranku
		foreach ($this->typesList->getTypes() as $type) {
			if (!$type->isActive() || empty($type->getPositions()))
				continue;

			foreach ($type->getPositions() as $pos) {
				if ($pos === 'headStart') {
					$presenter->addHeadStartComponent($this->typeControlFactory->create($type), $type->getKey());
				} else if ($pos === 'head') {
					$component = $this->typeControlFactory->create($type);
					$presenter->addHeadComponent($component, $type->getKey());

					if ($type->getKey() === 'googleAnalytics') {
						/** @var GoogleAdsType|null $googleAdsType */
						$googleAdsType = $this->typesList->getType('googleAds');

						if ($googleAdsType && $googleAdsType->isActive()) {
							$component->setParam('googleAdsType', $googleAdsType);
							$component->setParam('googleAdsId', $googleAdsType->getFullId());
							$component->setParam(
								'googleAdsEnhanced',
								$googleAdsType->getFieldValue('allowEnhancedConversions'),
							);
						}
					}
				} else if ($pos === 'bodyStart') {
					$presenter->addBodyStartComponent($this->typeControlFactory->create($type), $type->getKey());
				} else if ($pos === 'bodyEnd') {
					$presenter->addBodyEndComponent($this->typeControlFactory->create($type), $type->getKey());
				}
			}
		}

		// Sklik retargeting
		$skrType = $this->typesList->getType('sklikRetargeting');
		if ($skrType->isActive()) {
			$skrComponent = $this->typeControlFactory->create($skrType);

			// Nastavit detail pokud je aktivni detail produktu nebo kategorie
			if ($presenter->getAction(
					true,
				) === ':EshopCatalog:Front:Default:product' && $presenter instanceof EshopCatalogDefaultPresenter && $presenter->product) {
				$skrComponent->setParam('productDetailId', $presenter->product->getId());
			} else if ($presenter->getAction(true) === ':EshopCatalog:Front:Default:category'
				&& $presenter instanceof EshopCatalogDefaultPresenter
				&& $presenter->category && isset($presenter->category->attrs['productsComparison']['exports']['zbozi'])) {
				$skrComponent->setParam(
					'categoryDetailId',
					$presenter->category->attrs['productsComparison']['exports']['zbozi']['categoryText'],
				);
			}

			$presenter->addBodyEndComponent($skrComponent, $skrType->getKey());
		}
	}

	public function beforeRenderBodyEnd(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var BasePresenter $presenter */
		$presenter = $event->control;

		if ($this->container->hasService('eshopCatalog.front.productsFacade')
			&& class_exists(ShowedProducts::class)
			&& !empty(ShowedProducts::getShowedProductsIds())
			&& $presenter->getParameter('do') !== 'cartAddForm-form-submit') {

			/** @var ProductsFacade $productsFacade */
			$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');
			$products       = $productsFacade->getProducts(ShowedProducts::getShowedProductsIds());

			// GADS
			/** @var GoogleAdsType|null $gAdsType */
			$gAdsType = $this->typesList->getType('googleAds');
			if ($gAdsType && $gAdsType->isActive()) {
				$gAdsComponent = GTagEventHelper::getComponent(
					$presenter,
					$this->gTagEventControlFactory,
					'view_item_list',
					$gAdsType,
				);

				if ($gAdsComponent) {
					$gAdsComponent->setData([
						'items'   => array_values(
							array_map(static fn($p,
							) => GTagEventHelper::getItemFromProduct($p), $products),
						),
						'send_to' => $gAdsType->getFullId(),
					]);

					if ($presenter->isAjax()) {
						$gAdsComponent->redrawControl('wrap');
					}
				}
			}

			// GA 4
			if ($this->ga4Helper->isAllowed()) {
				$items = [];
				foreach ($products as $product) {
					$item = GTagEventHelper::getGa4ItemFromProduct($product, $product);

					if ($item) {
						$items[] = $item;
					}
				}

				$this->ga4Helper->sendEvent(
					'view_item_list',
					[
						'items' => $items,
					],
					$presenter,
					'ga4',
				);
			}
		}
	}

	public function formControlCreate(CreateFormEvent $event): void
	{
		/** @var Dao\Form|null $form */
		$form = $event->data['formDao'];

		if (!$this->isMeasuringCodesAllowed() || !$form) {
			return;
		}

		/** @var BasePresenter|null $presenter */
		$presenter = $event->control->getPresenterIfExists();
		if (!$presenter) {
			return;
		}

		$ms = $form->measuringCodes;

		if ($ms['sklikConversionId'] && $ms['sklikConversionId']->value) {
			$skcType = $this->typesList->getType('sklikConversion');
			if ($skcType && $skcType->isActive()) {
				$skcComponent             = $this->typeControlFactory->create($skcType);
				$skcComponent->useSnippet = true;

				$presenter->addBodyEndComponent($skcComponent, $skcType->getKey() . 'Form' . ucfirst($form->ident));
			}
		}

		if ($ms['googleAdsConversionLabel'] && $ms['googleAdsConversionLabel']->value) {
			$googleAdsType = $this->typesList->getType('googleAds');
			$ga            = $this->typesList->getType('googleAnalytics');
			if ($googleAdsType && $ga && $googleAdsType->isActive() && $ga->isActive()) {
				/** @var GTagEventControl $adsComponent */
				$adsComponent = GTagEventHelper::getComponent(
					$presenter,
					$this->gTagEventControlFactory,
					'conversion',
					$ga,
					'Form' . ucfirst($form->ident),
				);
				$adsComponent->useSnippet();
			}
		}
	}

	public function formFormControlSuccess(FormSuccessEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var Form $form */
		$form = $event->custom['entity'];

		/** @var BasePresenter $presenter */
		$presenter = $event->presenter;

		/** @var DataLayerControl|null $component */
		$component = $presenter->getComponent('dlFormSend_bodyEnd', false);

		if ($component) {
			$ident = $form->getIdent();
			/** @var Submission|null $submission */
			$submission = $event->custom['submission'] ?? null;

			$allowCookieSections   = AppState::getState('allowCookieSections', []);
			$cookieBarEnable       = SystemConfig::load('cookieBar.enable', false) && !SystemConfig::load(
					'cookieBar.fakeMode',
					false,
				);
			$enablePartialFakeMode = in_array('sklikConversion', SystemConfig::load('cookieBar.partialFakeMode', []));
			$enableConsentMode     = in_array(
				'sklikConversion',
				SystemConfig::load('cookieBar.servicesWithConsentMode', []),
			);
			$analyticalAllowed     = in_array('analytical', $allowCookieSections);

			$consent = null;
			/** @phpstan-ignore-next-line */
			if (!$cookieBarEnable || ($cookieBarEnable && $enablePartialFakeMode) || ($cookieBarEnable && $enableConsentMode && $analyticalAllowed)) {
				$consent = 1;
			} else {
				$consent = 0;
			}

			$component->setData([
				'event'     => 'formSend' . ucfirst($ident),
				'name'      => $form->getText()->title,
				'pagepath'  => $this->request->getUrl()->getAbsoluteUrl(),
				'messageId' => $submission ? $submission->getId() : null,
			]);

			$ga4Component = $this->ga4Helper->sendEvent(
				'formSend',
				[
					'ident'     => $ident,
					'name'      => $form->getText()->title,
					'pagepath'  => $this->request->getUrl()->getAbsoluteUrl(),
					'messageId' => $submission ? $submission->getId() : null,
				],
				$presenter,
			);

			$ms = $form->getMeasuringCodesByKey();

			$adsComponent = null;
			if (isset($ms['googleAdsConversionLabel']) && $ms['googleAdsConversionLabel']->value) {
				/** @var GoogleAdsType|null $googleAdsType */
				$googleAdsType = $this->typesList->getType('googleAds');
				$ga            = $this->typesList->getType('googleAnalytics');
				if ($googleAdsType && $ga && $googleAdsType->isActive() && $ga->isActive()) {
					/** @var GTagEventControl|null $adsComponent */
					$adsComponent = $presenter->getComponent(
						$ga->getKey() . '_conversion_Form' . ucfirst($form->getIdent()) . '_bodyEnd',
						false,
					);

					if ($adsComponent) {
						$adsComponent->setData([
							'send_to'        => $googleAdsType->getFullId(
								) . '/' . $ms['googleAdsConversionLabel']->value,
							'transaction_id' => (string) $form->getIdent(),
						]);
					}
				}
			}

			$skcComponent = null;
			if (isset($ms['sklikConversionId']) && $ms['sklikConversionId']->value) {
				$skcType = $this->typesList->getType('sklikConversion');
				if ($skcType && $skcType->isActive()) {
					/** @var DefaultTypeControl|null $skcComponent */
					$skcComponent = $presenter->getComponent(
						$skcType->getKey() . 'Form' . ucfirst($form->getIdent()) . '_bodyEnd',
						false,
					);

					if ($skcComponent) {
						$skcComponent->setParam('customConfig', [
							'id'      => $ms['sklikConversionId']->value,
							'value'   => null,
							'consent' => $consent,
						]);
					}
				}
			}

			if ($presenter->isAjax()) {
				$component->redrawControl('wrap');

				if ($ga4Component) {
					$ga4Component->redrawControl('wrap');
				}

				if ($adsComponent) {
					$adsComponent->redrawControl('wrap');
				}

				if ($skcComponent) {
					$skcComponent->redrawControl('wrap');
				}
			}
		}
	}
}
