<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\Event;
use Core\Model\Helpers\Strings;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Presenters\CustomerPresenter;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderItem;
use Exception;
use MeasuringCodes\FrontModule\Model\Helpers\Ga4Helper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\EcoMail;
use MeasuringCodes\Model\Helpers\TrackingHelper;
use MeasuringCodes\Model\MeasuringCodesConfig;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;
use Nette\DI\Container;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;
use Soukicz\Zbozicz;
use Tracy\Debugger;

class OrderSubscriber extends MeasuringCodesEventSubscriber
{
	final public const SESSION_SECTION = 'eshopOrdersEvents';
	final public const ORDER_REPEAT    = 'orderRepeat';
	protected SessionSection $orderSuccessMessages;

	public function __construct(
		protected Session        $session,
		protected TypesList      $typesList,
		protected Translator     $translator,
		protected EcoMail        $ecoMail,
		protected Container      $container,
		protected Ga4Helper      $ga4Helper,
		protected TrackingHelper $trackingHelper,
	)
	{
		$this->orderSuccessMessages = $session->getSection('eshopOrders/orderSuccessMessages');
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderOnSuccess'                => ['orderOnSuccess', 101],
			CustomerPresenter::class . '::beforeRender' => 'customerBeforeRender',
			'eshopOrders.repeatOrder'                   => 'repeatOrder',
		];
	}

	public function orderOnSuccess(OrderEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		$order    = $event->order;
		$allowLog = MeasuringCodesConfig::load('allowLog', false);

		if ($allowLog) {
			FileSystem::createDir(LOG_DIR . DS . 'measuringcodes');
			$date        = (new DateTime)->format('Y-m-d');
			$logFileName = 'measuringcodes/' . $order->site->getIdent() . '-' . $date;
		}

		$this->trackingHelper->clear();

		if (!$order->hasFlag(OrderFlag::TYPE_QUESTIONING)) {
			if ($allowLog) {
				Debugger::log("Order {$order->getId()} nepotvrzeno questioning", $logFileName);
			}

			return;
		}

		if ($allowLog) {
			Debugger::log("Order {$order->getId()} on success start", $logFileName);
		}

		$isSk = $this->translator->getLocale() === 'sk';

		// Zbozi
		$zcType     = $this->typesList->getType('zboziConversion');
		$shopId     = $zcType->getFieldValue('id_provozovny');
		$privateKey = $zcType->getFieldValue('klic');
		if ($zcType && $zcType->isActive() && $shopId && $privateKey) {
			try {
				if ($allowLog) {
					Debugger::log("Order {$order->getId()} try zbozi", $logFileName);
				}

				$client = new Zbozicz\Client($shopId, $privateKey, $zcType->getFieldValue('isSandbox'));

				//radeji pouzijeme kod, ale pokud neni, tak alespon ID
				$spedition = $order->getSpedition();
				$spedId    = $spedition->getSpedition()->zboziId;
				if (!$spedId) {
					$spedId = $spedition->getSpedition()->getId();
				}

				$payment = $order->getPayment();
				$payId   = $payment->getPayment()->getIdent();
				if (!$payId) {
					$payId = $payment->getPayment()->getId();
				}

				$zboziOrder = new Zbozicz\Order($order->getId());
				$zboziOrder
					->setEmail($order->getAddressDelivery()->getEmail())
					->setDeliveryType($spedId)
					->setDeliveryPrice($spedition->getPrice())
					->setPaymentType($payId);

				$otherCosts = 0;
				$otherCosts += $payment->getPrice();
				if ($order->getOrderDiscounts()) {
					foreach ($order->getOrderDiscounts() as $orderDiscount) {
						$otherCosts += $orderDiscount->getPrice();
					}
				}
				$zboziOrder->setOtherCosts($otherCosts);

				foreach ($order->getOrderItems() as $orderItem) {
					$zboziOrder->addCartItem(
						(new Zbozicz\CartItem)
							->setId($orderItem->getProductId())
							->setName($orderItem->getOrderItemText()->getName())
							->setUnitPrice($orderItem->getPrice())
							->setQuantity($orderItem->getQuantity()),
					);
				}

				$client->sendOrder($zboziOrder);

				if ($allowLog) {
					Debugger::log("Order {$order->getId()} zbozi sent", $logFileName);
				}
			} catch (Exception $e) {
				$msg = sprintf(
					"Failed export conversion on Zbozi.cz: orderId='%s' (%s: %s) - %s",
					$order->getId(),
					$e->getFile(),
					$e->getLine(),
					$e->getMessage(),
				);

				LogNotifier::toDevelopers($msg, 'Zboží - %siteDomain%');

				if ($allowLog) {
					Debugger::log("Order {$order->getId()} zbozi failed - {$e->getMessage()}", $logFileName);
					Debugger::log($e, $logFileName);
				}
			}
		}

		// Ecomail
		$ecoMailType = $this->typesList->getType('ecoMail');
		if ($ecoMailType && $ecoMailType->isActive()) {
			$appId  = $ecoMailType->getFieldValue('appId');
			$listId = $ecoMailType->getFieldValue('listId');

			if ($appId && $listId && $order->getCustomer()) {
				try {
					if ($allowLog) {
						Debugger::log("Order {$order->getId()} try ecomail", $logFileName);
					}

					$ecoMailApi     = new \Ecomail($appId);
					$subscriberData = $this->ecoMail->getSubscriberFromCustomer($order->getCustomer());
					$tags           = [];

					if (MeasuringCodesConfig::load('ecoMail.useTagNewsletter') && $order->hasFlag(
							OrderFlag::TYPE_NEWSLETTER,
						)) {
						$tags[] = 'newsletter';
					}

					if (MeasuringCodesConfig::load('ecoMail.useTagCategories')) {
						$prodIds = array_map(fn(OrderItem $v) => $v->getProductId(),
							$order->getOrderItems()
								->toArray());

						if (!empty($prodIds)) {
							$prodIds = array_filter($prodIds, fn($v) => $v);

							/** @var ProductsFacade $productsFacade */
							$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');

							foreach ($productsFacade->getProducts($prodIds) as $product) {
								if ($product->defaultCategory) {
									$tags[] = str_replace(
										'-',
										'',
										Strings::webalize((string) $product->defaultCategory->name),
									);
								}
							}
						}
					}

					if (!empty($tags)) {
						if (!isset($subscriberData['tags'])) {
							$subscriberData['tags'] = $tags;
						} else {
							$subscriberData['tags'] = array_merge($subscriberData['tags'], $tags);
						}
					}

					$subscribed = $ecoMailApi->getSubscriber($listId, $subscriberData['email']);
					if (isset($subscribed['subscriber'])) {
						if (isset($subscribed['subscriber']['tags'], $subscriberData['tags'])) {
							$subscriberData['tags'] = array_unique(
								array_merge($subscribed['subscriber']['tags'], $subscriberData['tags']),
							);
						}

						$ecoMailApi->updateSubscriber($listId, [
							'email'           => $subscriberData['email'],
							'subscriber_data' => $subscriberData,
						]);
					} else {
						if (MeasuringCodesConfig::load('ecoMail.useTagNewsletterNew') && $order->hasFlag(
								OrderFlag::TYPE_NEWSLETTER,
							)) {
							$subscriberData['tags'][] = 'newsletter_new';
						}

						$ecoMailApi->addSubscriber($listId, [
							'subscriber_data'        => $subscriberData,
							'trigger_autoresponders' => true,
							'update_existing'        => true,
							'resubscribe'            => false,
							'skip_confirmation'      => true,
						]);
					}

					$ecoMailApi->createNewTransaction($this->ecoMail->getTransactionFromOrder($order));

					if ($allowLog) {
						Debugger::log("Order {$order->getId()} ecomail sent", $logFileName);
					}
				} catch (Exception $e) {
					$msg = sprintf(
						"Failed export data to ecomail: orderId='%s' (%s: %s) - %s",
						$order->getId(),
						$e->getFile(),
						$e->getLine(),
						$e->getMessage(),
					);

					$msg .= ' ' . Json::encode($subscriberData);

					LogNotifier::toDevelopers($msg, 'Ecomail - %siteDomain%');
					if ($allowLog) {
						Debugger::log("Order {$order->getId()} ecomail failed - {$msg}", $logFileName);
						Debugger::log($e, $logFileName);
					}
				}
			}
		}
	}

	public function customerBeforeRender(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var CustomerPresenter $presenter */
		$presenter = $event->control;

		if ($presenter->getAction() === 'orderDetail') {
			$this->ga4Helper->sendEvent(
				'view_order',
				[
					'order_id' => $presenter->getParameter('orderId'),
				],
				$presenter,
			);
		}

		$this->ga4Helper->sendPageView('customer', $presenter);
	}

	public function repeatOrder(Event $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		$sessionSection = new SessionSection($this->session, self::SESSION_SECTION);
		$sessionSection->set(self::ORDER_REPEAT, [
			'time'    => time(),
			'orderId' => $event->data['orderId'],
		]);
	}
}
