<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Core\FrontModule\Presenters\BasePresenter;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\ControlEvent;
use Core\Model\Http\Session;
use Core\Model\UI\FrontPresenter;
use MeasuringCodes\FrontModule\Model\Helpers\Ga4Helper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesEventSubscriber;
use Nette\Http\Request;
use Nette\Http\SessionSection;
use Users\Model\Event\UserEvent;

class UserSubscriber extends MeasuringCodesEventSubscriber
{
	final public const SESSION_SECTION = 'userEvents';
	final public const USER_REGISTERED = 'userRegistered';
	final public const USER_LOGGED_IN  = 'userLoggedIn';
	final public const USER_LOGOUT     = 'userLogout';
	protected SessionSection $sessionSection;

	public function __construct(
		protected EntityManagerDecorator $em,
		protected TypesList              $typesList,
		Session                          $session,
		protected Request                $request,
		protected Ga4Helper              $ga4Helper,
	)
	{
		$this->sessionSection = new SessionSection($session, self::SESSION_SECTION);
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'user.loggedIn'                          => 'loggedIn',
			'user.logout'                            => 'logout',
			FrontPresenter::class . '::beforeRender' => 'beforeRender',
		];
	}

	public function loggedIn(UserEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		$this->sessionSection->set(self::USER_LOGGED_IN, [
			'time'    => time(),
			'pageUrl' => $this->request->getUrl()->getAbsoluteUrl(),
		]);
	}

	public function logout(UserEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		$this->sessionSection->set(self::USER_LOGOUT, [
			'time'    => time(),
			'pageUrl' => $this->request->getUrl()->getAbsoluteUrl(),
		]);
	}

	public function beforeRender(ControlEvent $event): void
	{
		if (!$this->isMeasuringCodesAllowed()) {
			return;
		}

		/** @var BasePresenter $presenter */
		$presenter = $event->control;

		$userRegistered = $this->sessionSection->get(self::USER_REGISTERED);
		if ($userRegistered) {
			$this->ga4Helper->sendEvent(
				'sign_up',
				[
					'method'   => 'email',
					'page_url' => $userRegistered['pageUrl'],
				],
				$presenter,
			);

			$this->sessionSection->remove(self::USER_REGISTERED);
		}

		$userLoggedIn = $this->sessionSection->get(self::USER_LOGGED_IN);
		if ($userLoggedIn) {
			$this->ga4Helper->sendEvent(
				'log_in',
				[
					'method'   => 'email',
					'page_url' => $userLoggedIn['pageUrl'],
				],
				$presenter,
			);

			$this->sessionSection->remove(self::USER_LOGGED_IN);
		}

		$userLogout = $this->sessionSection->get(self::USER_LOGOUT);
		if ($userLogout) {
			$this->ga4Helper->sendEvent(
				'log_out',
				[
					'page_url' => $userLogout['pageUrl'],
				],
				$presenter,
			);

			$this->sessionSection->remove(self::USER_LOGOUT);
		}
	}
}
