<?php declare(strict_types = 1);

namespace MeasuringCodes\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\DBAL\Logging\Middleware;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderStatus;
use MeasuringCodes\FrontModule\Model\TypesList;
use Nette\Utils\DateTime;
use Psr\Log\NullLogger;
use Tracy\Debugger;

class EcoMail
{
	public function __construct(protected EntityManagerDecorator $em, protected TypesList $typesList)
	{
	}

	protected function getEcoMailApi(): ?\Ecomail
	{
		$ecoMailType = $this->typesList->getType('ecoMail');
		if (!$ecoMailType || !$ecoMailType->isActive() || !$ecoMailType->getFieldValue('appId')) {
			return null;
		}

		return new \Ecomail($ecoMailType->getFieldValue('appId'));
	}

	public function getSubscriberFromCustomer(Customer $customer): array
	{
		$addr = $customer->getAddressInvoice();

		$tags = [];
		$data = [
			'name'    => $addr->getFirstName(),
			'surname' => $addr->getLastName(),
			'email'   => $addr->email,
			'company' => $addr->getCompany(),
			'city'    => $addr->getCity(),
			'street'  => $addr->getStreet(),
			'zip'     => $addr->getPostal(),
			'country' => $addr->getCountry() ? $addr->getCountry()->getName() : 'CZ',
			'phone'   => $addr->phone,
		];

		$useTagCustomer = MeasuringCodesConfig::load('ecoMail.useTagCustomer');
		$useTagVip      = MeasuringCodesConfig::load('ecoMail.useTagVip');

		if ($useTagCustomer || $useTagVip > 0) {
			$ordersCount = $this->em->getConnection()
				->fetchOne("SELECT COUNT(id) FROM eshop_orders__order WHERE customer_id = " . $customer->getId());

			if ($useTagCustomer) {
				$tags[] = 'zakaznik';
			}

			if ($ordersCount >= $useTagVip) {
				$tags[] = 'vip';
			}
		}

		if (MeasuringCodesConfig::load('ecoMail.useTagCountry') && $addr->getCountry()) {
			$tags[] = mb_strtolower($addr->getCountry()->getId());
		}

		if (!empty($tags)) {
			$data['tags'] = $tags;
		}

		return $data;
	}

	public function getTransactionFromOrder(Order $order): array
	{
		$timestamp = $order->getCreatedTime()->getTimestamp();

		$transactionItems = [];
		$addr             = $order->getAddressInvoice();

		foreach ($order->getOrderItems()->toArray() as $orderItem) {
			/** @var OrderItem $orderItem */
			$transactionItems[] = [
				'code'      => $orderItem->getProductId(),
				'title'     => $orderItem->getOrderItemText()->getName(),
				'price'     => round((float) $orderItem->getPrice(), 2),
				'amount'    => (int) $orderItem->getQuantity(),
				'timestamp' => $timestamp,
			];
		}

		return [
			'transaction'       => [
				'order_id'  => $order->getId(),
				'email'     => $addr->getEmail(),
				'shop'      => $order->site->getIdent(),
				'amount'    => round((float) $order->getPriceItems(), 2),
				'shipping'  => round((float) $order->getPaySpedPrice(), 2),
				'city'      => $addr->getCity(),
				'country'   => $addr->getCountry() ? $addr->getCountry()->getName() : 'CZ',
				'timestamp' => $timestamp,
			],
			'transaction_items' => $transactionItems,
		];
	}

	public function checkInactive(int $days, int $length, int $listId, ?string $tag = null): void
	{
		$ecoMailType = $this->typesList->getType('ecoMail');
		if (!$ecoMailType || !$ecoMailType->isActive() || !$ecoMailType->getFieldValue('appId')) {
			return;
		}

		if (!$tag) {
			$tag = 'neaktivni';
		}

		$this->em->getConfiguration()->setMiddlewares([new Middleware(new NullLogger)]);

		$appId  = $ecoMailType->getFieldValue('appId');
		$today  = (new DateTime());
		$toDate = (new DateTime())->modify('-' . ($days + $length) . ' days')
			->setTime(0, 0, 0);
		$emails = [];
		foreach ($this->em->getRepository(Order::class)->createQueryBuilder('o')
			         ->select('cAddr.email, os.created')
			         ->innerJoin('o.orderStatuses', 'os', Join::WITH, 'os.status = :status AND os.created >= :toDate')
			         ->innerJoin('o.customer', 'c')
			         ->innerJoin('c.addressInvoice', 'cAddr')
			         ->setParameters([
				         'status' => OrderStatus::STATUS_CREATED,
				         'toDate' => $toDate,
			         ])
			         ->orderBy('os.created', 'DESC')
			         ->getQuery()->getArrayResult() as $row) {
			if (isset($emails[$row['email']])) {
				continue;
			}

			$emails[$row['email']] = $row['created'];
		}

		$ecoMailApi = new \Ecomail($appId);
		$i          = 0;

		foreach ($emails as $email => $created) {
			$diff = $today->diff($created);

			if ($diff->days >= $days) {
				$data = $ecoMailApi->getSubscriber((string) $listId, $email);

				if (isset($data['error'])) {
					foreach ($data['message']->errors as $err) {
						Debugger::log('ERROR - ' . $err, '_ecomailCheckInactive');
					}

					continue;
				}

				if (!isset($data['subscriber'])) {
					Debugger::log($email . ' - Subscriber data not found', '_ecomailCheckInactive');

					continue;
				}

				if (!isset($data['subscriber']['tags'])) {
					$data['subscriber']['tags'] = [];
				}

				if (!in_array($tag, $data['subscriber']['tags'])) {
					$data['subscriber']['tags'][] = $tag;
				}

				$ecoMailApi->updateSubscriber((string) $listId, [
					'email'           => $email,
					'subscriber_data' => $data['subscriber'],
				]);
				Debugger::log("Update - set inactive " . $email, '_ecomailCheckInactive');

				$i++;
				if ($i % 30 === 0) {
					sleep(2);
				}
			}
		}
	}

	public function updateSubscriber(string $email, array $data, bool $resubscribe = false): void
	{
		$ecoMailApi  = $this->getEcoMailApi();
		$ecoMailType = $this->typesList->getType('ecoMail');
		$listId      = $ecoMailType->getFieldValue('listId');

		if (!$ecoMailApi || !$listId) {
			return;
		}

		$subscriber = $ecoMailApi->getSubscriber($listId, $email);

		if (isset($subscriber['subscriber'])) {
			$subscriberData = array_merge_recursive($subscriber['subscriber'], $data);
			$action         = 'update';
		} else {
			$subscriberData = $data;
			$action         = 'create';
		}

		if (is_array($subscriberData['tags'])) {
			$subscriberData['tags'] = array_unique($subscriberData['tags']);
		}

		if (!isset($subscriberData['email'])) {
			$subscriberData['email'] = $email;
		}

		if ($action === 'update') {
			$ecoMailApi->updateSubscriber($listId, [
				'email'           => $email,
				'subscriber_data' => $subscriberData,
			]);
		} else {
			$ecoMailApi->addSubscriber($listId, [
				'subscriber_data'        => $subscriberData,
				'trigger_autoresponders' => true,
				'update_existing'        => true,
				'resubscribe'            => $resubscribe,
				'skip_confirmation'      => true,
			]);
		}
	}
}
