<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Core\Model\Event\ControlEvent;
use Core\Model\UI\FrontPresenter;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopOrders\FrontModule\Model\Payments;
use EshopOrders\FrontModule\Model\Speditions;
use EshopOrders\FrontModule\Presenters\DefaultPresenter;
use MeasuringCodes\FrontModule\Components\GTagEventControl;
use MeasuringCodes\FrontModule\Components\IDataLayerControlFactory;
use MeasuringCodes\FrontModule\Components\IFBPixelEventControlFactory;
use MeasuringCodes\FrontModule\Components\IGTagEventControlFactory;
use MeasuringCodes\FrontModule\Model\Helpers\DataLayerHelper;
use MeasuringCodes\FrontModule\Model\Helpers\GTagEventHelper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesConfig;
use Nette\Application\IPresenter;
use Nette\DI\Container;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class EshopCartSubscriber implements EventSubscriberInterface
{
	protected IGTagEventControlFactory $gTagEventControlFactory;

	protected IFBPixelEventControlFactory $fbPixelEventControlFactory;

	protected IDataLayerControlFactory $dataLayerControlFactory;

	protected TypesList $typesList;

	protected array $cartChanges = [
		'added'   => [],
		'updated' => [],
	];

	protected array $orderChanges = [];

	public Container $container;

	public function __construct(Container                   $container, TypesList $typesList, IGTagEventControlFactory $gTagEventControlFactory,
	                            IFBPixelEventControlFactory $fbPixelEventControlFactory, IDataLayerControlFactory $dataLayerControlFactory)
	{
		$this->container                  = $container;
		$this->gTagEventControlFactory    = $gTagEventControlFactory;
		$this->typesList                  = $typesList;
		$this->fbPixelEventControlFactory = $fbPixelEventControlFactory;
		$this->dataLayerControlFactory    = $dataLayerControlFactory;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FrontPresenter::class . '::beforeRender'  => 'beforeRender',
			'eshopOrders.cartAddItem'                 => 'onAddItem',
			'eshopOrders.cartUpdateItem'              => 'onUpdateItem',
			DefaultPresenter::class . '::renderorder' => 'renderOrder',
			'eshopOrders.saveOrderFormDataStep2'      => 'orderSaveStep2',
		];
	}

	public function beforeRender(ControlEvent $event): void
	{
		if ((!MeasuringCodesConfig::load('enabledInTestMode', false) && CORE_TEST_MODE) || !MeasuringCodesConfig::load('enabled', false) || !$this->container->hasService('eshopCatalog.front.productsFacade'))
			return;

		/** @var FrontPresenter $presenter */
		$presenter = $event->control;

		/** @var ProductsFacade $productsFacade */
		$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');

		/** @var Carts $cartService */
		$cartService = $this->container->getService('eshopOrders.carts');

		/** @var Currencies $currencies */
		$currencies = $this->container->getService('currency.currencies');
		$currency   = $currencies ? $currencies->getCurrent()->getCode() : 'CZK';

		$items = [
			'add_to_cart'      => [],
			'remove_from_cart' => [],
		];

		foreach ($this->cartChanges['added'] as $id => $change) {
			$items['add_to_cart'][$id] = $change;
		}

		foreach ($this->cartChanges['updated'] as $id => $change) {
			$tmp = $cartService->getCartItem($id);

			if ($tmp) {
				if ($change > 0) {
					$items['add_to_cart'][$tmp->productId] = $change;
				} else {
					$items['remove_from_cart'][$tmp->productId] = abs($change);
				}
			}
		}

		foreach ($items as $eventName => $v) {
			$component   = $this->getComponent($presenter, $eventName);
			$dlComponent = $this->dataLayerControlFactory->create();
			$dlEventName = $eventName == 'add_to_cart' ? 'addToCart' : 'removeFromCart';
			$presenter->addBodyEndComponent($dlComponent, 'dl' . $dlEventName);
			$dlComponent->useSnippet = true;

			if (!$component) {
				return;
			}

			$data   = [];
			$dataDl = [];
			foreach ($v as $id => $quantity) {
				$product = $productsFacade->getProduct($id);

				$tmp             = GTagEventHelper::getItemFromProduct($product);
				$tmp['quantity'] = $quantity;

				$data[] = $tmp;

				$tmp             = DataLayerHelper::getProduct($product);
				$tmp['quantity'] = $quantity;
				$dataDl[]        = $tmp;
			}

			if ($data) {
				$component->setData([
					'items' => $data,
				]);
			}
			if (!empty($dataDl)) {
				$dlComponent->setData([
					'event'     => $dlEventName,
					'ecommerce' => [
						'currencyCode' => $currency,
						'add'          => [
							'products' => $dataDl,
						],
					],
				]);
			}

			if ($presenter->isAjax()) {
				$component->redrawControl('wrap');
				$dlComponent->redrawControl('wrap');
			}
		}

		// FB pixel
		$fbpPixel = $this->typesList->getType('facebookPixel');
		if ($fbpPixel->isActive() && $fbpPixel->getFieldValue('enableEcommerce')) {
			$fbpComponent = $this->fbPixelEventControlFactory->create($fbpPixel, 'AddToCart');
			$fbpComponent->showOnlyIfCode3();
			$presenter->addBodyEndComponent($fbpComponent, $fbpPixel->getKey() . 'AddToCart');

			if (!empty($items['add_to_cart'])) {
				$totalValue = 0;
				$data       = [
					'currency'     => $currency,
					'content_type' => 'product',
					'contents'     => [],
				];

				foreach ($items['add_to_cart'] as $id => $quantity) {
					$product    = $productsFacade->getProduct($id);
					$price      = round($product->getPrice() * $quantity, 2);
					$totalValue += $price;

					$data['contents'][] = [
						'id'       => strval($id),
						'quantity' => $quantity,
					];
				}

				$data['value'] = $totalValue;

				$fbpComponent->setCode3($data);

				if ($presenter->isAjax()) {
					$fbpComponent->redrawControl('wrap');
				}
			}
		}
	}

	public function onAddItem(AddedCartItemEvent $item): void
	{
		$this->cartChanges['added'][$item->item->productId] = $item->item->quantity;
	}

	public function onUpdateItem(UpdatedCartItemEvent $item): void
	{
		$this->cartChanges['updated'][$item->itemId] = $item->quantity - $item->beforeQuantity;
	}

	public function renderOrder(ControlEvent $event): void
	{
		if ((!MeasuringCodesConfig::load('enabledInTestMode', false) && CORE_TEST_MODE) || !MeasuringCodesConfig::load('enabled', false))
			return;

		/** @var DefaultPresenter $presenter */
		$presenter = $event->control;
		/** @var ProductsFacade $productsFacade */
		$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');

		/** @var Currencies $currencies */
		$currencies = $this->container->getService('currency.currencies');
		$currency   = $currencies ? $currencies->getCurrent()->getCode() : 'CZK';

		// InitiateCheckout
		$fbpType = $this->typesList->getType('facebookPixel');
		if ($presenter->getParameter('do', null) === null && empty($presenter->getHttpRequest()->getPost())) {
			if ($fbpType->isActive() && $fbpType->getFieldValue('enableEcommerce')) {
				$fbpComponent = $this->fbPixelEventControlFactory->create($fbpType, 'InitiateCheckout', 'track');

				$code3 = [
					'currency'  => $currency,
					'num_items' => count($presenter->cartsService->getCurrentCart()->getCartItems()),
					'value'     => $presenter->cartsService->getCurrentCart()->getCartItemsPrice(),
				];

				$fbpComponent->setCode3($code3);

				$presenter->addBodyEndComponent($fbpComponent, $fbpType->getKey() . 'InitiateCheckout');
			}
		}

		// GA
		$speditionComponent = $this->getComponent($presenter, 'set_checkout_option', 'spedition');
		$paymentComponent   = $this->getComponent($presenter, 'set_checkout_option', 'payment');

		$spedDlComponent = $this->dataLayerControlFactory->create();
		$presenter->addBodyEndComponent($spedDlComponent, 'dlCheckoutOptionSped');
		$spedDlComponent->useSnippet = true;

		$payDlComponent = $this->dataLayerControlFactory->create();
		$presenter->addBodyEndComponent($payDlComponent, 'dlCheckoutOptionPay');
		$payDlComponent->useSnippet = true;

		// FB
		$fbPaymentComponent = null;
		if ($fbpType->isActive() && $fbpType->getFieldValue('enableEcommerce')) {
			$fbPaymentComponent = $this->fbPixelEventControlFactory->create($fbpType, 'AddPaymentInfo', 'track');
			$fbPaymentComponent->showOnlyIfCode3();
			$presenter->addBodyEndComponent($fbPaymentComponent, $fbpType->getKey() . 'AddPaymentInfo');
		}

		if ($presenter->getParameter('do') !== null) {
			// Nastaveni dopravy
			if (isset($this->orderChanges['spedition'])) {
				/** @var Speditions $speditions */
				$speditions = $this->container->getService('eshopOrders.front.speditions');
				$spedition  = $speditions->get((int) $this->orderChanges['spedition']);

				if ($speditionComponent) {
					$speditionComponent->setData([
						'checkout_step'   => 1,
						'checkout_option' => 'shipping method',
						'value'           => $spedition->getName(),
					]);
				}

				$spedDlComponent->setData([
					'event'     => 'checkoutOption',
					'ecommerce' => [
						'checkout_option' => [
							'actionField' => [
								'step'   => 1,
								'option' => $spedition->getName(),
							],
						],
					],
				]);
			}

			// Nastaveni platby
			if (isset($this->orderChanges['payment'])) {
				/** @var Payments $payments */
				$payments = $this->container->getService('eshopOrders.front.payments');
				$payment  = $payments->get((int) $this->orderChanges['payment']);

				if ($paymentComponent)
					$paymentComponent->setData([
						'checkout_step'   => 2,
						'checkout_option' => 'payment method',
						'value'           => $payment->getName(),
					]);

				if ($payDlComponent)
					$payDlComponent->setData([
						'event'     => 'checkoutOption',
						'ecommerce' => [
							'checkout_option' => [
								'actionField' => [
									'step'   => 2,
									'option' => $payment->getName(),
								],
							],
						],
					]);

				// FB
				if ($fbPaymentComponent) {
					$code3 = [
						'content_type' => 'product',
						'content_ids'  => array_values(array_map(fn($row) => (string) $row->id, $presenter->cartsService->getCurrentCart()->getCartItems())),
						'content_name' => array_values(array_map(fn($row) => $row->title, $presenter->cartsService->getCurrentCart()->getCartItems())),
						'value'        => $presenter->cartsService->getCurrentCart()->getCartItemsPrice(),
						'currency'     => $currency,
					];

					$fbPaymentComponent->setCode3($code3);
				}
			}

			if ($presenter->isAjax()) {
				if ($speditionComponent)
					$speditionComponent->redrawControl('wrap');
				if ($paymentComponent)
					$paymentComponent->redrawControl('wrap');
				if ($spedDlComponent)
					$spedDlComponent->redrawControl('wrap');
				if ($payDlComponent)
					$payDlComponent->redrawControl('wrap');
				if ($fbPaymentComponent)
					$fbPaymentComponent->redrawControl('wrap');
			}

			return;
		}

		// Vytvoreni transakce
		$component   = $this->getComponent($presenter, 'begin_checkout');
		$dlComponent = $this->dataLayerControlFactory->create();
		$presenter->addBodyEndComponent($dlComponent, 'dlCheckout');
		$items   = [];
		$dlItems = [];

		if (!$component) {
			return;
		}

		foreach ($presenter->cartsService->getCurrentCart()->getCartItems() as $item) {
			$product = $productsFacade->getProduct($item->getProductId());

			$tmp             = GTagEventHelper::getItemFromProduct($product);
			$tmp['quantity'] = (int) $item->getQuantity();
			$tmp['price']    = GTagEventHelper::formatNumber($item->getPrice());

			$items[] = $tmp;

			$tmp             = DataLayerHelper::getProduct($product);
			$tmp['quantity'] = $item->getQuantity();
			$dlItems[]       = $tmp;
		}

		$component->setData([
			'items'  => $items,
			'coupon' => '',
		]);

		$dlComponent->setData([
			'event'     => 'checkout',
			'ecommerce' => [
				'checkout' => [
					'actionField' => [
						'step'   => 1,
						'option' => '',
					],
					'products'    => $dlItems,
				],
			],
		]);
	}

	public function orderSaveStep2(SaveOrderFormDataEvent $event): void
	{
		$this->orderChanges = [
			'spedition' => $event->data['spedition'],
			'payment'   => $event->data['payment'],
		];
	}

	protected function getComponent(IPresenter $presenter, string $eventName, ?string $componentNameSuffix = null): ?GTagEventControl
	{
		$type = GTagEventHelper::getTypeIfAllowed($this->typesList);
		if (!$type) {
			return null;
		}

		return GTagEventHelper::getComponent($presenter, $this->gTagEventControlFactory, $eventName, $type, $componentNameSuffix);
	}
}
