<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Core\Model\Event\ControlEvent;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Presenters\DefaultPresenter;
use MeasuringCodes\FrontModule\Components\IDataLayerControlFactory;
use MeasuringCodes\FrontModule\Components\IFBPixelEventControlFactory;
use MeasuringCodes\FrontModule\Components\IGTagEventControlFactory;
use MeasuringCodes\FrontModule\Components\ITypeControlFactory;
use MeasuringCodes\FrontModule\Model\Helpers\GTagEventHelper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesConfig;
use Nette\DI\Container;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class EshopCatalogSubscriber implements EventSubscriberInterface
{
	protected ITypeControlFactory $typeControlFactory;

	protected IGTagEventControlFactory $gTagEventControlFactory;

	protected TypesList $typesList;

	protected IFBPixelEventControlFactory $fbPixelEventControlFactory;

	protected IDataLayerControlFactory $dataLayerControlFactory;

	protected Container $container;

	public function __construct(ITypeControlFactory $typeControlFactory, TypesList $typesList, IGTagEventControlFactory $gTagEventControlFactory,
	                            IFBPixelEventControlFactory $fpPixelEvenControlFactory, Container $container,
	                            IDataLayerControlFactory $dataLayerControlFactory)
	{
		$this->typeControlFactory         = $typeControlFactory;
		$this->gTagEventControlFactory    = $gTagEventControlFactory;
		$this->typesList                  = $typesList;
		$this->fbPixelEventControlFactory = $fpPixelEvenControlFactory;
		$this->container                  = $container;
		$this->dataLayerControlFactory    = $dataLayerControlFactory;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			DefaultPresenter::class . '::rendercategory' => 'defaultRenderCategory',
			DefaultPresenter::class . '::renderproduct'  => 'defaultRenderProduct',
		];
	}

	public function defaultRenderCategory(ControlEvent $event): void
	{
		if (!MeasuringCodesConfig::load('enabled', false))
			return;

		/** @var DefaultPresenter $presenter */
		$presenter = $event->control;

		if ($presenter->getParameter('do', null) === 'cartAddForm-form-submit') {
			return;
		}
	}

	public function defaultRenderProduct(ControlEvent $event): void
	{
		if ((!MeasuringCodesConfig::load('enabledInTestMode', false) && CORE_TEST_MODE) || !MeasuringCodesConfig::load('enabled', false))
			return;

		/** @var DefaultPresenter $presenter */
		$presenter = $event->control;

		if ($presenter->getParameter('do', null) === 'cartAddForm-form-submit' || !$presenter->product) {
			return;
		}

		$product = $presenter->product;

		/** @var Currencies $currencies */
		$currencies = $this->container->getService('currency.currencies');
		$currency   = $currencies ? $currencies->getCurrent()->getCode() : 'CZK';

		// GA
		$gaType = GTagEventHelper::getTypeIfAllowed($this->typesList);
		if ($gaType) {
			$gaComponent     = GTagEventHelper::getComponent($presenter, $this->gTagEventControlFactory, 'view_item', $gaType);
			$data['items'][] = GTagEventHelper::getItemFromProduct($presenter->product);

			$gaComponent->setData($data);
		}

		// FB pixel
		$fbpType = $this->typesList->getType('facebookPixel');
		if ($fbpType->isActive() && $fbpType->getFieldValue('enableEcommerce')) {
			$fbpComponent = $this->fbPixelEventControlFactory->create($fbpType, 'ViewContent');

			$code3 = [
				'contents'     => [
					[
						'id'       => strval($product->getId()),
						'quantity' => 1,
					],
				],
				'content_type' => 'product',
				'content_name' => $product->getName(),
				'value'        => $product->getPrice(),
				'currency'     => $currency,
				'condition'    => $product->condition,
			];


			if ($product->defaultCategory) {
				$categoryName = $product->defaultCategory->getParentPathStringFlipped();

				if ($categoryName)
					$categoryName .= ' > ';
				$categoryName .= $product->defaultCategory->name;

				$code3['content_category'] = trim((string) $categoryName);
			}

			if ($product->getAvailability() && $product->getAvailability()->getOpenGraphText()) {
				$code3['availability'] = $product->getAvailability()->getOpenGraphText();
			}

			$fbpComponent->setCode3($code3);

			$presenter->addBodyEndComponent($fbpComponent, $fbpType->getKey() . 'ViewContent');
		}
	}
}
