<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\UI\FrontPresenter;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Components\ProductsList;
use EshopCatalog\FrontModule\Model\Helpers\ShowedProducts;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Presenters\DefaultPresenter;
use EshopCatalog\Model\Helpers\Helper;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Customers;
use Forms\DI\FormsExtension;
use Forms\FrontModule\Components\FormControl;
use Forms\Model\Entities\Form;
use MeasuringCodes\FrontModule\Components\DataLayerControl;
use MeasuringCodes\FrontModule\Components\IDataLayerControlFactory;
use MeasuringCodes\FrontModule\Components\IGTagEventControlFactory;
use MeasuringCodes\FrontModule\Components\ITypeControlFactory;
use MeasuringCodes\FrontModule\Model\Helpers\GTagEventHelper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesConfig;
use Nette\DI\Container;
use Nette\Http\Request;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class FrontPresenterSubscriber implements EventSubscriberInterface
{
	protected ITypeControlFactory $typeControlFactory;

	protected IGTagEventControlFactory $gTagEventControlFactory;

	protected IDataLayerControlFactory $dataLayerControlFactory;

	protected TypesList $typesList;

	protected Request $request;

	protected Container $container;

	public function __construct(ITypeControlFactory $typeControlFactory, TypesList $typesList, Container $container,
	                            IGTagEventControlFactory $gTagEventControlFactory, IDataLayerControlFactory $dataLayerControlFactory,
	                            Request $request)
	{
		$this->typeControlFactory      = $typeControlFactory;
		$this->typesList               = $typesList;
		$this->gTagEventControlFactory = $gTagEventControlFactory;
		$this->container               = $container;
		$this->dataLayerControlFactory = $dataLayerControlFactory;
		$this->request                 = $request;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FrontPresenter::class . '::startup'                  => 'startup',
			FrontPresenter::class . '::beforeRender'             => 'beforeRender',
			FrontPresenter::class . '::getBodyEndComponentsPack' => 'beforeRenderBodyEnd',
			FormControl::class . '::formSuccess'                 => 'formFormControlSuccess',
		];
	}

	public function startup(ControlEvent $event): void
	{

		if ((!MeasuringCodesConfig::load('enabledInTestMode', false) && CORE_TEST_MODE) || !MeasuringCodesConfig::load('enabled', false))
			return;

		/** @var FrontPresenter $presenter */
		$presenter = $event->control;

		if (class_exists(FormsExtension::class)) {
			$dlComponent             = $this->dataLayerControlFactory->create();
			$dlComponent->useSnippet = true;
			$presenter->addBodyEndComponent($dlComponent, 'dlFormSend');
		}
	}

	/**
	 * Vložení obecných kódů na všechny stránky
	 *
	 * @param ControlEvent $event
	 */
	public function beforeRender(ControlEvent $event): void
	{
		if ((!MeasuringCodesConfig::load('enabledInTestMode', false) && CORE_TEST_MODE) || !MeasuringCodesConfig::load('enabled', false))
			return;

		/** @var FrontPresenter $presenter */
		$presenter = $event->control;

		if (class_exists(ShowedProducts::class)) {
			if ($presenter->getParameter('do', null) === 'cartAddForm-form-submit') {
				return;
			}
			$action = $presenter->getAction(true);

			// Persoo
			$persooType = $this->typesList->getType('persoo');
			if ($persooType && $persooType->isActive() && $action !== ':EshopOrders:Front:Finished:orderFinished') {
				$data = [];

				/** @var ProductsList $productsList */
				$productsList = $presenter->getComponent('list', false);
				if ($productsList && $productsList instanceof ProductsList) {
					foreach ($productsList->getProducts() as $prod)
						$data['impressedProducts']['locationMain'][] = (string) $prod->getId();
				}

				if ($this->container->hasService('currency.currencies')) {
					/** @var Currencies $currencies */
					$currencies = $this->container->getService('currency.currencies');

					$data['currency'] = $currencies->getCurrent()->getCode();
				}

				try {
					/** @var DaoNavigationItem $activeNav */
					$activeNav   = $presenter->getActiveNavigation();
					$addCategory = false;

					if ($action === ':EshopCatalog:Front:Default:product' && $presenter->product) {
						/** @var DefaultPresenter $presenter */
						$data['pageType'] = 'detail';
						$data['itemID']   = $presenter->product->getId();

						if ($presenter->product->getManufacturer())
							$data['brand'] = $presenter->product->getManufacturer()->name;

						$addCategory = true;
					} else if ($action === ':EshopCatalog:Front:Default:category' && $presenter->category) {
						$data['pageType'] = 'category';
						$addCategory      = true;
					} else if ($action === ':EshopCatalog:Front:Default:search') {
						$data['pageType'] = 'search';
					} else if ($action === ':EshopOrders:Front:Default:order') {
						$data['pageType'] = 'basket';
					} else if ($activeNav->isHomepage) {
						$data['pageType'] = 'homepage';
					} else {
						$data['pageType'] = 'other';
					}

					if ($addCategory) {
						if (isset($activeNav->componentParams['virtualCategory'])) {
							$data['categoryID'] = 'v' . $activeNav->componentParams['virtualCategory'];
							$data['hierarchy']  = 'v' . $activeNav->componentParams['virtualCategory'];
						} else if ($presenter->category) {
							$data['categoryID'] = $presenter->category->getId();

							$catPath = array_keys($presenter->category->getParentPath()) ?: [];
							if ($catPath) {
								$catPath = array_reverse($catPath);
							}

							$catPath[]         = $presenter->category->getId();
							$data['hierarchy'] = implode(':', $catPath);
						}
					}

					if ($this->container->hasService('eshopOrders.carts')) {
						/** @var Carts $cartService */
						$cartService = $this->container->getService('eshopOrders.carts');
						$cart        = $cartService->getCurrentCart();

						if ($cart->getCartItems()) {
							$data['basketItems'] = [];
							foreach ($cart->getCartItems() as $item) {
								$data['basketItems'][$item->getProductId()] = [
									'quantity' => $item->getQuantity(),
									'price'    => $item->getPrice(),
								];
							}
						}
						$data['basketTotal'] = $cart->getPriceTotal();
					}
				} catch (\Exception $e) {
				}

				$user = $presenter->getUser();
				if ($user->isLoggedIn()) {
					$data['login'] = $user->getIdentity()->getEmail();
					$data['email'] = $user->getIdentity()->getEmail();

					try {
						if ($this->container->hasService('eshopOrders.front.customers')) {
							/** @var Customers $customers */
							$customers = $this->container->getService('eshopOrders.front.customers');
							$customer  = $customers->getByUser($user->getIdentity());

							if ($customer && $customer->getGroupCustomers()) {
								$data['priceLevel'] = $customer->getGroupCustomers()->getId();

								$productsSale = $customer->getGroupCustomers()->getProductsSale();
								if ($productsSale && $productsSale > 0)
									$data['userPercentSale'] = (float) $productsSale;
							}
						}
					} catch (\Exception $e) {
					}

					if (!isset($data['userPercentSale']) && $this->container->hasService('eshopCatalog.helper')) {
						/** @var Helper $eshopCatalogHelper */
						$eshopCatalogHelper      = $this->container->getService('eshopCatalog.helper');
						$percentSale             = $eshopCatalogHelper->getRegisterUserSale($user->getIdentity());
						$data['userPercentSale'] = (float) $percentSale;
					}
				}

				$dlComponent         = $this->dataLayerControlFactory->create();
				$dlComponent->varKey = 'dataLayerPersoo';
				if ($data)
					$dlComponent->setData($data);

				$presenter->addHeadComponent($dlComponent, 'persooDl');
			}
		}

		// Vlozeni obechnych na kazdou stranku
		foreach ($this->typesList->getTypes() as $type) {
			if (!$type->isActive() || empty($type->getPositions()))
				continue;

			foreach ($type->getPositions() as $pos) {
				if ($pos === 'headStart') {
					$presenter->addHeadStartComponent($this->typeControlFactory->create($type), $type->getKey());
				} else if ($pos === 'head') {
					$component = $this->typeControlFactory->create($type);
					$presenter->addHeadComponent($component, $type->getKey());

					if ($type->getKey() === 'googleAnalytics') {
						$googleAdsType = $this->typesList->getType('googleAds');

						if ($googleAdsType && $googleAdsType->isActive()) {
							$component->setParam('googleAds', $googleAdsType->getFieldValue('ads_id'));
							$component->setParam('googleAdsEnhanced', $googleAdsType->getFieldValue('allowEnhancedConversions'));
						}
					}
				} else if ($pos === 'bodyStart') {
					$presenter->addBodyStartComponent($this->typeControlFactory->create($type), $type->getKey());
				} else if ($pos === 'bodyEnd') {
					$presenter->addBodyEndComponent($this->typeControlFactory->create($type), $type->getKey());
				}
			}
		}

		// Sklik retargeting
		$skrType = $this->typesList->getType('sklikRetargeting');
		if ($skrType->isActive()) {
			$skrComponent = $this->typeControlFactory->create($skrType);

			// Nastavit detail pokud je aktivni detail produktu nebo kategorie
			if ($presenter->getAction(true) === ':EshopCatalog:Front:Default:product' && $presenter->product) {
				$skrComponent->setParam('productDetailId', $presenter->product->getId());
			} else if ($presenter->getAction(true) === ':EshopCatalog:Front:Default:category'
				&& $presenter->category && isset($presenter->category->attrs['productsComparison']['exports']['zbozi'])) {
				$skrComponent->setParam('categoryDetailId', $presenter->category->attrs['productsComparison']['exports']['zbozi']['categoryText']);
			}

			$presenter->addBodyEndComponent($skrComponent, $skrType->getKey());
		}
	}

	public function beforeRenderBodyEnd(ControlEvent $event): void
	{
		if ((!MeasuringCodesConfig::load('enabledInTestMode', false) && CORE_TEST_MODE) || !MeasuringCodesConfig::load('enabled', false))
			return;

		/** @var FrontPresenter $presenter */
		$presenter = $event->control;

		if ($this->container->hasService('eshopCatalog.front.productsFacade')) {
			if (class_exists(ShowedProducts::class)) {
				if ($presenter->getParameter('do', null) === 'cartAddForm-form-submit') {
					return;
				}

				/** @var ProductsFacade $productsFacade */
				$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');
				$products       = $productsFacade->getProducts(ShowedProducts::getShowedProductsIds());

				// GA
				$gaType = GTagEventHelper::getTypeIfAllowed($this->typesList);
				if ($gaType && !empty(ShowedProducts::getShowedProductsIds())) {
					$gaComponent = GTagEventHelper::getComponent($presenter, $this->gTagEventControlFactory, 'view_item_list', $gaType);

					if ($gaComponent) {
						$gaComponent->setData([
							'items' => array_values(array_map(fn($p) => GTagEventHelper::getItemFromProduct($p), $products)),
						]);

						if ($presenter->isAjax()) {
							$gaComponent->redrawControl('wrap');
						}
					}
				}
			}
		}
	}

	public function formFormControlSuccess(FormSuccessEvent $event): void
	{
		/** @var Form $form */
		$form = $event->custom['entity'];

		/** @var FrontPresenter $presenter */
		$presenter = $event->presenter;

		/** @var DataLayerControl $component */
		$component = $presenter->getComponent('dlFormSend_bodyEnd', false);

		if ($component) {
			$ident = $form->getIdent();

			$component->setData([
				'event'    => 'formSend' . ucfirst($ident),
				'name'     => $form->getText()->title,
				'pagepath' => $this->request->getUrl()->getAbsoluteUrl(),
			]);

			if ($presenter->isAjax())
				$component->redrawControl('wrap');
		}
	}
}
