<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Core\Model\Event\ControlEvent;
use Core\Model\Helpers\Strings;
use Currency\Model\Currencies;
use Currency\Model\Exchange;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\FrontModule\Presenters\FinishedPresenter;
use EshopOrders\Model\Entities\OrderItem;
use MeasuringCodes\FrontModule\Components\IDataLayerControlFactory;
use MeasuringCodes\FrontModule\Components\IFBPixelEventControlFactory;
use MeasuringCodes\FrontModule\Components\IGTagEventControlFactory;
use MeasuringCodes\FrontModule\Components\ITypeControlFactory;
use MeasuringCodes\FrontModule\Components\TypeControl;
use MeasuringCodes\FrontModule\Model\Helpers\GTagEventHelper;
use MeasuringCodes\Model\Helpers\TrackingHelper;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\MeasuringCodesConfig;
use Nette\DI\Container;
use Nette\Localization\ITranslator;
use Nette\Utils\Json;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use MeasuringCodes\FrontModule\Components\Zbozi;

class OrderFinishedSubscriber implements EventSubscriberInterface
{
	protected ITypeControlFactory $typeControlFactory;

	protected TypesList $typesList;

	protected TrackingHelper $trackingHelper;

	protected IGTagEventControlFactory $gTagEventControlFactory;

	protected IFBPixelEventControlFactory $fbPixelEventControlFactory;

	protected IDataLayerControlFactory $dataLayerControlFactory;

	protected Container $container;

	protected ITranslator $translator;

	public function __construct(ITypeControlFactory      $typeControlFactory, TypesList $typesList, TrackingHelper $trackingHelper,
	                            IGTagEventControlFactory $gTagEventControlFactory, IFBPixelEventControlFactory $fbPixelEventControlFactory,
	                            Container                $container, IDataLayerControlFactory $dataLayerControlFactory, ITranslator $translator)
	{
		$this->typeControlFactory         = $typeControlFactory;
		$this->typesList                  = $typesList;
		$this->trackingHelper             = $trackingHelper;
		$this->gTagEventControlFactory    = $gTagEventControlFactory;
		$this->fbPixelEventControlFactory = $fbPixelEventControlFactory;
		$this->dataLayerControlFactory    = $dataLayerControlFactory;
		$this->container                  = $container;
		$this->translator                 = $translator;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FinishedPresenter::class . '::beforeRender' => 'orderFinishedBeforeRender',
		];
	}

	/**
	 * Vložení měžících kódů pod dokončení objednávky
	 *
	 * @param ControlEvent $event
	 */
	public function orderFinishedBeforeRender(ControlEvent $event): void
	{
		if ((!MeasuringCodesConfig::load('enabledInTestMode', false) && CORE_TEST_MODE) || !MeasuringCodesConfig::load('enabled', false) || !$this->container->hasService('eshopCatalog.front.productsFacade'))
			return;

		$isSk = $this->translator->getLocale() == 'sk';

		/** @var FinishedPresenter $presenter */
		$presenter = $event->control;
		$order     = $presenter->order;

		if (!$order) {
			return;
		}

		/** @var ProductsFacade $productsFacade */
		$productsFacade = $this->container->getService('eshopCatalog.front.productsFacade');

		/** @var Sellers $sellers */
		$sellers    = $this->container->getService('eshopCatalog.front.sellersService') ?? null;
		$isVatPayer = $sellers->getSellerForSite($order->site->getIdent())->dic ? true : false;

		// Zbozi
		$zcType = $this->typesList->getType('zboziConversion');
		if ($zcType->isActive()) {
			/** @var Zbozi\IOrderTrackingFactory $zcFactory */
			$zcFactory   = $zcType->factory;
			$zcComponent = $zcFactory->create($zcType, $presenter->order);
			$presenter->addBodyEndComponent($zcComponent, $zcType->getKey());
		}

		// Heureka
		$hrqType = $this->typesList->getType('heurekaConversion' . ($isSk ? 'Sk' : ''));
		if ($hrqType->isActive() && $this->trackingHelper->canSend('heurekaConversion', $order->getId())) {
			$hrqComponent = $hrqType->factory->create($hrqType);
			$hrqComponent->setParam('orderId', $order->getId())
				->setParam('scriptLink', $isSk ? 'https://im9.cz/sk/js/ext/2-roi-async.js' : 'https://im9.cz/js/ext/1-roi-async.js')
				->setParam('products', $order->getOrderItems());

			$presenter->addBodyEndComponent($hrqComponent, $hrqType->getKey());
			$this->trackingHelper->send('heurekaConversion', $order->getId());
		}

		// Sklik retargeting
		$skcType = $this->typesList->getType('sklikConversion');
		if ($skcType->isActive() && $this->trackingHelper->canSend('sklikConversion', $order->getId())) {
			$skcComponent = $this->typeControlFactory->create($skcType);
			$skcComponent->setParam('orderValue', $order->getPrice());
			$skcComponent->setParam('orderId', $order->getId());

			$presenter->addBodyEndComponent($skcComponent, $skcType->getKey());
			$this->trackingHelper->send('sklikConversion', $order->getId());
		}

		// GA
		$gaType = GTagEventHelper::getTypeIfAllowed($this->typesList);
		if ($gaType && $this->trackingHelper->canSend('gaConversion', $order->getId())) {
			$gaComponent = GTagEventHelper::getComponent($presenter, $this->gTagEventControlFactory, 'purchase', $gaType);

			$data = [
				'transaction_id' => strval($order->getId()),
				'affiliation'    => $order->site->getIdent(),
				'value'          => $isVatPayer
					? GTagEventHelper::formatNumber(round($order->getPriceItemsDiscount() - $order->getItemsVatPrice(), 2))
					: GTagEventHelper::formatNumber($order->getPriceItemsDiscount()),
				'currency'       => $order->getDefaultCurrency(),
				'tax'            => $isVatPayer
					? GTagEventHelper::formatNumber(round($order->getItemsVatPrice(), 2))
					: '0',
				'shipping'       => GTagEventHelper::formatNumber((float) $order->getPaySpedPrice()),
				'items'          => [],
			];

			foreach ($order->getOrderItems() as $item) {
				$product = $productsFacade->getProduct($item->getProductId());

				$tmp             = GTagEventHelper::getItemFromProduct($product);
				$tmp['quantity'] = (int) $item->getQuantity();
				$tmp['price']    = $isVatPayer
					? GTagEventHelper::formatNumber($item->getPriceWithoutVat())
					: GTagEventHelper::formatNumber($item->getPrice());

				$data['items'][] = $tmp;
			}

			foreach ($order->getOrderDiscounts() as $discount) {
				$data['items'][] = [
					'id'       => strval($discount->getCode()),
					'name'     => $discount->getName(),
					'price'    => GTagEventHelper::formatNumber($discount->getPrice()),
					'quantity' => (int) $discount->getQuantity() ?: 1,
				];
			}

			$gaComponent->setData($data);
			$this->trackingHelper->send('gaConversion', $order->getId());
		}

		// Google Ads
		$adsType = $this->typesList->getType('googleAds');
		if ($adsType && $adsType->isActive() && $this->trackingHelper->canSend('googleAdsConversion', $order->getId())) {
			$merchantId = (string) $adsType->getFieldValue('merchantId');/** @var TypeControl $adsComponent */;
			if ($merchantId) {
				$adsComponent = $adsType->factory->create($adsType);
			} else {
				$adsComponent = $adsType->factory->create($adsType);
			}

			$addrInvoice            = $order->getAddressInvoice();
			$homeAddress            = [];
			$enhancedConversionData = [
				'first_name' => $addrInvoice->getFirstName(),
				'last_name'  => $addrInvoice->getLastName(),
				'email'      => $addrInvoice->getEmail(),
			];
			if ($addrInvoice->getPhone())
				$enhancedConversionData['phone_number'] = $addrInvoice->getPhone();
			if ($addrInvoice->getStreet())
				$homeAddress['street'] = $addrInvoice->getStreet();
			if ($addrInvoice->getCity())
				$homeAddress['city'] = $addrInvoice->getCity();
			if ($addrInvoice->getPostal())
				$homeAddress['postal_code'] = $addrInvoice->getPostal();
			if ($addrInvoice->getCountry())
				$homeAddress['country'] = $addrInvoice->getCountry()->getIso3166_1();

			if (!empty($homeAddress))
				$enhancedConversionData['home_address'] = $homeAddress;

			$adsComponent->setParam('enhancedConversionData', Json::encode($enhancedConversionData));

			if ($merchantId) {
				// Purchase
				$items = [];
				foreach ($order->getOrderItems() as $item) {
					$items[] = [
						'id'       => $item->getProductId(),
						'quantity' => $item->getQuantity(),
						'price'    => $isVatPayer
							? GTagEventHelper::formatNumber($item->getPriceWithoutVat())
							: GTagEventHelper::formatNumber($item->getPrice()),
					];
				}

				foreach ($order->getOrderDiscounts() as $discount) {
					$items[] = [
						'id'       => $discount->getId(),
						'quantity' => $discount->getQuantity() ?: 1,
						'price'    => GTagEventHelper::formatNumber($discount->getPrice()),
					];
				}

				$adsComponent->setParam('eventName', 'purchase')
					->setParam('data', Json::encode([
						'send_to'          => 'AW-' . $adsType->getFieldValue('ads_id') . '/' . $adsType->getFieldValue('conversionLabel'),
						'transaction_id'   => (string) $order->getId(),
						'value'            => $isVatPayer
							? GTagEventHelper::formatNumber(round($order->getPriceItemsDiscount() - $order->getItemsVatPrice(), 2))
							: GTagEventHelper::formatNumber($order->getPriceItemsDiscount()),
						'currency'         => $order->getDefaultCurrency(),
						'aw_merchant_id'   => $merchantId,
						'aw_feed_country'  => 'CZ',
						'aw_feed_language' => 'cs',
						'items'            => $items,
					]));
				$presenter->addBodyEndComponent($adsComponent, 'googleMerchantPurchase');
			} else {
				// Conversion
				$adsComponent->setParam('eventName', 'conversion')
					->setParam('data', Json::encode([
						'send_to'        => 'AW-' . $adsType->getFieldValue('ads_id') . '/' . $adsType->getFieldValue('conversionLabel'),
						'value'          => $isVatPayer
							? GTagEventHelper::formatNumber(round($order->getPriceItemsDiscount() - $order->getItemsVatPrice(), 2))
							: GTagEventHelper::formatNumber($order->getPriceItemsDiscount()),
						'currency'       => $order->getDefaultCurrency(),
						'transaction_id' => (string) $order->getId(),
					]));
				$presenter->addBodyEndComponent($adsComponent, 'googleAdsConversion');
			}

			$this->trackingHelper->send('googleAdsConversion', $order->getId());
		}

		// Persoo
		try {
			$persooType = $this->typesList->getType('persoo');
			if ($persooType && $persooType->isActive() && $this->trackingHelper->canSend('persooTransaction', $order->getId())) {
				/** @var ?Currencies $currencies */
				$currencies = $this->container->hasService('currency.currencies') ? $this->container->getService('currency.currencies') : null;
				/** @var ?Exchange $exchange */
				$exchange = $this->container->hasService('currency.exchange') ? $this->container->getService('currency.exchange') : null;

				$data = [
					'pageType'    => 'basket',
					'currency'    => $order->getCurrencyCode(),
					'basketItems' => [],
					'basketTotal' => $order->getPriceItems(true),
					'transaction' => [
						'id' => $order->getId(),
					],
				];

				foreach (['CZK', 'EUR'] as $curr) {
					if (!isset($currencies->getActive()[$curr]))
						continue;

					if ($order->getDefaultCurrency() === $curr)
						$res = $order->getPriceItemsDiscount();
					else if ($order->getCurrencyCode() === $curr)
						$res = $order->getPriceItemsDiscount(true);
					else
						$res = $exchange ? $exchange->change($order->getPriceItemsDiscount(), $curr) : $order->getPriceItemsDiscount();

					$data['transaction']['revenue_' . Strings::lower($curr)] = $res;
				}

				foreach ($order->getOrderItems()->toArray() as $item) {
					/** @var OrderItem $item */
					$arr = [
						'quantity' => $item->getQuantity(),
					];

					foreach (['CZK', 'EUR'] as $curr) {
						if (!isset($currencies->getActive()[$curr]))
							continue;
						$key = 'price_' . Strings::lower($curr);

						if ($order->getDefaultCurrency() === $curr)
							$arr[$key] = $item->getPrice();
						else if ($order->getCurrencyCode() === $curr)
							$arr[$key] = $item->getPrice(true);
						else
							$arr[$key] = $exchange ? $exchange->change($item->getPrice(), $curr) : $item->getPrice();
					}

					$data['basketItems'][$item->getProductId()] = $arr;
				}

				$user = $presenter->getUser();
				if ($user->isLoggedIn()) {
					$data['login'] = $user->getIdentity()->getEmail();
					$data['email'] = $user->getIdentity()->getEmail();
				}

				$dlComponent         = $this->dataLayerControlFactory->create();
				$dlComponent->varKey = 'dataLayerPersoo';
				$dlComponent->setData($data);

				$presenter->addHeadStartComponent($dlComponent, 'persooDlOrderFinished');
				$this->trackingHelper->send('persooTransaction', $order->getId());
			}
		} catch (\Exception $e) {
		}

		// FB pixel
		$fbpType = $this->typesList->getType('facebookPixel');
		if ($fbpType->isActive() && $fbpType->getFieldValue('enableEcommerce') && $this->trackingHelper->canSend('fbPixelPurchase', $order->getId())) {
			$fbComponent = $this->fbPixelEventControlFactory->create($fbpType, 'Purchase');
			$data        = [
				'value'        => strval($order->getPrice()),
				'currency'     => $order->getDefaultCurrency(),
				'contents'     => [],
				'content_type' => 'product',
			];

			foreach ($order->getOrderItems() as $item) {
				$data['contents'][] = [
					'id'         => strval($item->getProductId()),
					'value'      => $item->getPriceTotal(),
					'quantity'   => $item->getQuantity(),
					'item_price' => $item->getPriceTotal(),
				];
			}

			foreach ($order->getOrderDiscounts() as $discount) {
				$data['contents'][] = [
					'id'         => strval($discount->getCode()),
					'value'      => $discount->getPrice(),
					'quantity'   => 1,
					'item_price' => $discount->getPrice(),
				];
			}

			$fbComponent->setCode3($data);

			$presenter->addBodyEndComponent($fbComponent, $fbpType->getKey() . 'Purchase');
			$this->trackingHelper->send('fbPixelPurchase', $order->getId());
		}
	}
}
