<?php declare(strict_types = 1);

namespace MeasuringCodes\FrontModule\Model\Subscribers;

use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\OrderFlag;
use Heureka\ShopCertification;
use MeasuringCodes\FrontModule\Model\TypesList;
use MeasuringCodes\Model\EcoMail;
use MeasuringCodes\Model\MeasuringCodesConfig;
use Nette\Http\Session;
use Nette\Http\SessionSection;
use Nette\Localization\ITranslator;
use Soukicz\Zbozicz;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderSubscriber implements EventSubscriberInterface
{
	protected SessionSection $orderSuccessMessages;

	protected TypesList $typesList;

	protected ITranslator $translator;

	public function __construct(Session $session, TypesList $typesList, ITranslator $translator)
	{
		$this->orderSuccessMessages = $session->getSection('eshopOrders/orderSuccessMessages');
		$this->typesList            = $typesList;
		$this->translator           = $translator;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderOnSuccess' => ['orderOnSuccess', 101],
		];
	}

	public function orderOnSuccess(OrderEvent $event): void
	{
		if ((!MeasuringCodesConfig::load('enabledInTestMode', false) && CORE_TEST_MODE) || !MeasuringCodesConfig::load('enabled', false))
			return;

		$order = $event->order;
		if (!$order || !$order->hasFlag(OrderFlag::TYPE_QUESTIONING))
			return;

		$isSk = $this->translator->getLocale() == 'sk';

		// Zbozi
		$zcType     = $this->typesList->getType('zboziConversion');
		$shopId     = $zcType->getFieldValue('id_provozovny');
		$privateKey = $zcType->getFieldValue('klic');
		if ($zcType && $zcType->isActive() && $shopId && $privateKey) {
			try {
				$client = new Zbozicz\Client($shopId, $privateKey, $zcType->getFieldValue('isSandbox'));

				//radeji pouzijeme kod, ale pokud neni, tak alespon ID
				$spedition = $order->getSpedition();
				$spedId    = $spedition->getSpedition()->zboziId;
				if (!$spedId) {
					$spedId = $spedition->getSpedition()->getId();
				}

				$payment = $order->getPayment();
				$payId   = $payment->getPayment()->getIdent();
				if (!$payId) {
					$payId = $payment->getPayment()->getId();
				}

				$zboziOrder = new Zbozicz\Order($order->getId());
				$zboziOrder
					->setEmail($order->getAddressDelivery()->getEmail())
					->setDeliveryType($spedId)
					->setDeliveryPrice($spedition->getPrice())
					->setPaymentType($payId);

				$otherCosts = 0;
				$otherCosts += $payment->getPrice();
				if ($order->getOrderDiscounts()) {
					foreach ($order->getOrderDiscounts() as $orderDiscount) {
						$otherCosts += $orderDiscount->getPrice();
					}
				}
				$zboziOrder->setOtherCosts($otherCosts);

				foreach ($order->getOrderItems() as $orderItem) {
					$zboziOrder->addCartItem((new Zbozicz\CartItem)
						->setId($orderItem->getProductId())
						->setName($orderItem->getOrderItemText()->getName())
						->setUnitPrice($orderItem->getPrice())
						->setQuantity($orderItem->getQuantity())
					);
				}

				$client->sendOrder($zboziOrder);
			} catch (\Exception $e) {
				$msg = sprintf("Failed export conversion on Zbozi.cz: orderId='%s' (%s: %s) - %s",
					$order->getId(), $e->getFile(), $e->getLine(), $e->getMessage());

				Debugger::log($msg, 'eshopordersConversion');
				LogNotifier::toDevelopers($msg, 'Zboží - %siteDomain%');
			}
		}

		// Heureka
		$hrqType = $this->typesList->getType('heurekaOvereno' . ($isSk ? 'Sk' : ''));
		$key     = $hrqType->getFieldValue('klic');
		if ($hrqType && $hrqType->isActive() && $key) {

			try {
				$options = [];
				if ($isSk) {
					$options['service'] = ShopCertification::HEUREKA_SK;
				}

				$shopCertification = new ShopCertification($key, $options);

				$shopCertification->setEmail($order->getAddressDelivery()->getEmail());
				$shopCertification->setOrderId($order->getId());


				foreach ($order->getOrderItems() as $orderItem) {
					$shopCertification->addProductItemId($orderItem->getProductId());
				}

				$shopCertification->logOrder();
			} catch (\Exception $e) {
				$msg = sprintf("Failed export conversion on Heureka: orderId='%a' (%s: %s) - %s",
					$order->getId(), $e->getFile(), $e->getLine(), $e->getMessage());

				Debugger::log($msg, 'eshopordersConversion');
				LogNotifier::toDevelopers($msg, 'Heureka ověřeno zákazníky - %siteDomain%');
			}
		}

		// Ecomail
		$ecoMailType = $this->typesList->getType('ecoMail');
		if ($ecoMailType && $ecoMailType->isActive()) {
			$appId  = $ecoMailType->getFieldValue('appId');
			$listId = $ecoMailType->getFieldValue('listId');

			if ($appId && $listId) {
				try {
					$ecoMailApi = new \Ecomail($appId);

					$ecoMailApi->addSubscriber($listId, [
						'subscriber_data'        => EcoMail::getSubscriberFromCustomer($order->getCustomer()),
						'trigger_autoresponders' => true,
						'update_existing'        => true,
						'resubscribe'            => false,
						'skip_confirmation'      => true,
					]);

					$ecoMailApi->createNewTransaction(EcoMail::getTransactionFromOrder($order));
				} catch (\Exception $e) {
					$msg = sprintf("Failed export data to ecomail: orderId='%a' (%s: %s) - %s",
						$order->getId(), $e->getFile(), $e->getLine(), $e->getMessage());

					Debugger::log($msg, 'eshopordersConversion');
					LogNotifier::toDevelopers($msg, 'Ecomail - %siteDomain%');
				}
			}
		}
	}
}
