<?php declare(strict_types = 1);

namespace Mojedpd\AdminModule\Model\Subscribers;


use Contributte\EventDispatcher\EventSubscriber;
use Core\Model\Event\ControlEvent;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use Mojedpd\Model\OrderApiService;
use Mojedpd\Model\OrderCsvService;
use Mojedpd\Model\OrdersExported;
use Nette\Application\Responses\FileResponse;
use Nette\Localization\ITranslator;
use Nette\Utils\DateTime;

class MojedpdOrdersSubscriber implements EventSubscriber
{
	/** @var OrdersExported */
	protected $ordersExported;

	/** @var OrderApiService */
	protected $apiService;

	/** @var OrderCsvService */
	protected $csvServices;

	/** @var ITranslator */
	protected $translator;

	/** @var array */
	protected $mojedpd;

	public function __construct(array $mojedpd, OrdersExported $ordersExported, OrderApiService $apiService, OrderCsvService $csvService, ITranslator $translator)
	{
		$this->ordersExported = $ordersExported;
		$this->apiService     = $apiService;
		$this->csvServices    = $csvService;
		$this->translator     = $translator;
		$this->mojedpd        = $mojedpd;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.admin.ordersDefaultNavigation'         => 'ordersDefaultNavigation',
			'eshopOrders.admin.ordersDefaultNavigationCallback' => 'ordersDefaultNavigationCallback',
			OrdersGrid::class . '::onAttach'                    => 'ordersGridAttached',
		];
	}

	public function ordersGridAttached(ControlEvent $event)
	{
		/** @var OrdersGrid $control */
		$control = $event->control;

		$control->addExportCallback('dpdPickup', 'mojedpd.ordersGrid.exportCsv', function(array $ids) use ($control) {
			$orders = $this->ordersExported->getOrdersNotExported($ids);

			$result = $this->csvServices->generateOrderCsv($orders);

			if (isset($result['file']) && file_exists($result['file'])) {
				$arr = [
					'name' => 'DPD export',
					'url'  => $control->getPresenter()->link(':Core:Admin:FileResponse:', base64_encode(serialize([
						'file'     => $result['file'],
						'filename' => 'DPD-export-' . (new DateTime())->format('Y-m-d') . '.csv',
					]))),
				];

				$control->getPresenter()->payload->fileRequests[] = $arr;
			}

			if ($result['error'] === 0)
				$control->getPresenter()->flashMessageSuccess($this->translator->translate('mojedpd.export.success'));
			else if ($result['error'] > 0 && $result['ok'] > 0)
				$control->getPresenter()->flashMessageWarning($this->translator->translate('mojedpd.export.partial', null, $result));
			else
				$control->getPresenter()->flashMessageDanger($this->translator->translate('mojedpd.export.error'));

			$control->getPresenter()->redrawControl('flashes');
		});
	}

	public function ordersDefaultNavigation(OrdersNavigationEvent $event): void
	{
		$event->addNavigation(
			'mojedpd.ordersNavigation.exportMojedpd',
			['navigationCallback!', ['key' => 'mojedpdExport']],
			'fas fa-truck-loading',
			'ajax'
		);
	}

	public function ordersDefaultNavigationCallback(OrderNavigationCallbackEvent $event): void
	{
		if ($event->key !== 'mojedpdExport')
			return;

		$orders = $this->ordersExported->getOrdersNotExported();
		if (!$orders) {
			$event->presenter->flashMessageSuccess($this->translator->translate('mojedpd.export.noOrders'));
			$event->presenter->redrawControl('flashes');

			return;
		}

		set_time_limit(120);
		switch ($this->mojedpd['exported']) {
			case 'api':
				$result = $this->apiService->generateOrderOverApi($orders);
				break;
			case 'file':
				$result = $this->csvServices->generateOrderCsv($orders);

				if (isset($result['file'])) {
					$arr = [
						'name' => 'DPD export',
						'url'  => $event->presenter->link(':Core:Admin:FileResponse:', base64_encode(serialize([
							'file'     => $result['file'],
							'filename' => 'DPD-export-' . (new DateTime())->format('Y-m-d') . '.csv',
						]))),
					];

					$event->presenter->payload->fileRequests[] = $arr;
				}

				break;
			default:
				$event->presenter->flashMessageWarning($this->translator->translate('mojedpd.export.noParamter'));
		}


		if ($result['error'] === 0)
			$event->presenter->flashMessageSuccess($this->translator->translate('mojedpd.export.success'));
		else if ($result['error'] > 0 && $result['ok'] > 0)
			$event->presenter->flashMessageWarning($this->translator->translate('mojedpd.export.partial', null, $result));
		else
			$event->presenter->flashMessageDanger($this->translator->translate('mojedpd.export.error'));

		$event->presenter->redrawControl('flashes');
	}
}
