<?php declare(strict_types = 1);

namespace Mojedpd\AdminModule\Model\Subscribers;

use Contributte\Translation\Exceptions\InvalidArgument;
use Contributte\Translation\Translator;
use Core\Model\Event\ControlEvent;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use EshopOrders\AdminModule\Model\Speditions;
use Mojedpd\Model\Entities\MojedpdOrder;
use Mojedpd\Model\MojeDpdConfig;
use Mojedpd\Model\OrderApiService;
use Mojedpd\Model\OrderCsvService;
use Mojedpd\Model\OrdersExported;
use Nette\Application\UI\InvalidLinkException;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class MojedpdOrdersSubscriber implements EventSubscriberInterface
{
	public function __construct(
		protected array           $mojedpd,
		protected OrdersExported  $ordersExported,
		protected OrderApiService $apiService,
		protected OrderCsvService $csvServices,
		protected Translator      $translator,
		protected Speditions      $speditionsService,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.admin.ordersDefaultNavigation'         => 'ordersDefaultNavigation',
			'eshopOrders.admin.ordersDefaultNavigationCallback' => 'ordersDefaultNavigationCallback',
			OrdersGrid::class . '::onAttach'                    => ['ordersGridAttached', 255],
			OrderForm::class . '::onAttach'                     => 'orderFormOnAttach',
		];
	}

	public function orderFormOnAttach(ControlEvent $event): void
	{
		/** @var OrderForm $control */
		$control = $event->control;

		if ($control->order && MojedpdOrder::filterIsDpd((string) $control->order->getSpeditionIdent())) {
			$control->template->speditionExportInfoTemplate = __DIR__ . '/OrderFormExportInfo.latte';
			$control->template->mojeDpdExported             = $this->ordersExported->get($control->order->getId());
		}
	}

	public function ordersGridAttached(ControlEvent $event): void
	{
		/** @var OrdersGrid $control */
		$control = $event->control;

		// handle pro tlacitka v popupu
		$control->addHandleCallback('mojeDpdExport', function(?array $args = null) use ($control): void {
			if (!$args) {
				return;
			}

			$force = isset($args['force']) && $args['force'] == 1;

			if ($args['ids']) {
				$orders = $this->ordersExported->getOrders($args['ids'], !$force);
				$this->runExport($orders, $control);
			}
		});

		$control->addExportCallback('dpdApiExport', 'mojedpd.ordersGrid.dpdApiExport', function(array $ids, $quantity) use ($control): void {
			$orders = $this->ordersExported->getOrders($ids);
			if ($orders === []) {
				$control->presenter->flashMessageSuccess($this->translator->translate('mojedpd.export.noOrders'));
				$control->presenter->redrawControl('flashes');

				return;
			}

			set_time_limit(120);
			$result = $this->apiService->generateOrderOverApi($orders, max(1, (int) $quantity));

			if ($result['error'] === 0) {
				$control->presenter
					->flashMessageSuccess($this->translator->translate('mojedpd.export.success'));
			} else if ($result['error'] > 0 && $result['ok'] > 0) {
				$control->presenter
					->flashMessageWarning($this->translator->translate('mojedpd.export.partial', null, $result));
			} else {
				$control->presenter->flashMessageDanger($this->translator->translate('mojedpd.export.error'));
			}

			$control->presenter->redrawControl('flashes');
		});

		$control->addExportCallback('dpdApiLabels', 'mojedpd.ordersGrid.dpdApiLabels', function(array $ids) use ($control): void {
			$orders = $this->ordersExported->getOrders($ids, false);
			if ($orders === []) {
				$control->presenter->flashMessageSuccess($this->translator->translate('mojedpd.export.noOrders'));
				$control->presenter->redrawControl('flashes');

				return;
			}

			set_time_limit(120);
			$result = $this->apiService->generateLabelPdf($orders);

			if ($result['error'] === 0) {
				foreach ($result['files'] as $file) {
					$arr                                         = [
						'name' => 'DPD labels',
						'url'  => $control->presenter->link(':Core:Admin:FileResponse:', base64_encode(serialize([
							'file'     => $file,
							'filename' => basename($file),
						]))),
					];
					$control->presenter->payload->fileRequests[] = $arr;
				}

				$control->presenter->flashMessageSuccess($this->translator->translate('mojedpd.export.labelsSuccess'));
			} else {
				$control->presenter
					->flashMessageDanger($this->translator->translate('mojedpd.export.labelsErrors'));
			}

			$control->presenter->redrawControl('flashes');
		});

		$control->addExportCallback('dpdPickup', 'mojedpd.ordersGrid.exportCsv', function(array $ids) use ($control): void {
			$orders = $this->ordersExported->getOrders($ids);

			$preparedIds = [];
			foreach ($orders as $order) {
				$preparedIds[] = $order->getOrder()->getId();
			}

			$exported = array_diff($ids, $preparedIds);

			if ($exported !== []) {
				$control->template->customModalTemplate = __DIR__ . '/ExportedModal.latte';
				$control->template->exported            = $exported;
				$control->template->ids                 = $ids;
				$control->template->modalTitle          = $control->t('default.notice');
				$control->template->modal               = 'customModal';
				$control->redrawControl('modal');
			} else {
				$this->runExport($orders, $control);
			}
		});
	}

	public function ordersDefaultNavigation(OrdersNavigationEvent $event): void
	{
		if (MojeDpdConfig::load('ordersNavigationShowExportAll')) {
			$event->addNavigation(
				'mojedpd.ordersNavigation.exportMojedpd',
				['navigationCallback!', ['key' => 'mojedpdExport']],
				'fas fa-truck-loading',
				'ajax'
			);
		}
	}

	public function ordersDefaultNavigationCallback(OrderNavigationCallbackEvent $event): void
	{
		if ($event->key !== 'mojedpdExport' || MojeDpdConfig::loadScalar('ordersNavigationShowExportAll') === false) {
			return;
		}

		$orders = $this->ordersExported->getOrders();

		$ids = [];
		foreach ($orders as $order) {
			$ids[] = $order->getOrder()->getId();
		}

		if ($ids !== []) {
			$allowed = $this->speditionsService->checkOrdersGridMenuExportOnlyInStatus($ids);

			foreach ($orders as $k => $order) {
				if (!in_array($order->getOrder()->getId(), $allowed)) {
					unset($orders[$k]);
				}
			}
		}

		if ($orders === []) {
			$event->presenter->flashMessageSuccess($this->translator->translate('mojedpd.export.noOrders'));
			$event->presenter->redrawControl('flashes');

			return;
		}

		set_time_limit(120);
		switch ($this->mojedpd['exported']) {
			case 'api':
				$result = $this->apiService->generateOrderOverApi($orders);
				break;
			case 'file':
				$result = $this->runExport($orders);

				if (isset($result['file'])) {
					$arr = [
						'name' => 'DPD export',
						'url'  => $event->presenter->link(':Core:Admin:FileResponse:', base64_encode(serialize([
							'file'     => $result['file'],
							'filename' => 'DPD-export-' . (new DateTime())->format('Y-m-d') . '.csv',
						]))),
					];

					$event->presenter->payload->fileRequests[] = $arr;
				}

				break;
			default:
				$event->presenter->flashMessageWarning($this->translator->translate('mojedpd.export.noParamter'));
		}


		if (isset($result)) {
			if ($result['error'] === 0) {
				$event->presenter->flashMessageSuccess($this->translator->translate('mojedpd.export.success'));
			} else if ($result['error'] > 0 && $result['ok'] > 0) {
				$event->presenter->flashMessageWarning($this->translator->translate('mojedpd.export.partial', null, $result));
			} else {
				$event->presenter->flashMessageDanger($this->translator->translate('mojedpd.export.error'));
			}
		}

		$event->presenter->redrawControl('flashes');
	}

	/**
	 * @param OrdersGrid|null $control
	 *
	 * @param MojedpdOrder[]  $orders
	 *
	 * @throws InvalidLinkException
	 * @throws InvalidArgument
	 */
	protected function runExport(array $orders, $control = null): array
	{
		$result = $this->csvServices->generateOrderCsv($orders);

		if ($control) {
			if (isset($result['file']) && file_exists($result['file'])) {
				$arr = [
					'name' => 'DPD export',
					'url'  => $control->presenter->link(':Core:Admin:FileResponse:', base64_encode(serialize([
						'file'     => $result['file'],
						'filename' => 'DPD-export-' . (new DateTime())->format('Y-m-d') . '.csv',
					]))),
				];

				$control->presenter->payload->fileRequests[] = $arr;
			}

			if ($result['error'] === 0 && $result['ok'] > 0) {
				$control->presenter->flashMessageSuccess($this->translator->translate('mojedpd.export.success'));
			} else if ($result['error'] > 0 && $result['ok'] > 0) {
				$control->presenter->flashMessageWarning($this->translator->translate('mojedpd.export.partial', null, $result));
			} else if ($result['error'] == 0 && $result['ok'] == 0) {
				$control->presenter->flashMessageWarning($this->translator->translate('mojedpd.export.noOrders'));
			} else {
				$control->presenter->flashMessageDanger($this->translator->translate('mojedpd.export.error'));
			}

			$control->presenter->redrawControl('flashes');
		}

		return $result;
	}
}
