<?php declare(strict_types = 1);

namespace Mojedpd\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Exception;
use libphonenumber\PhoneNumberUtil;
use Mojedpd\Model\Entities\MojedpdOrder;
use Mojedpd\Model\Utils\Helpers;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Tracy\Debugger;

class OrderCsvService
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected Branches               $branches,
		protected EventDispatcher        $eventDispatcher,
	)
	{
	}

	/**
	 * @param MojedpdOrder[] $orders
	 */
	public function generateOrderCsv(array $orders): array
	{
		$result = [
			'ok'    => 0,
			'error' => 0,
		];

		$this->em->beginTransaction();
		try {
			$tmpDir        = TMP_DIR . '/mojedpdExport/' . (new DateTime())->format('Y-m-d-H-i-s');
			$exportCsvType = (int) MojeDpdConfig::load('exportCsvType', 2);
			FileSystem::createDir($tmpDir);

			if ($exportCsvType === 0) {
				$file = fopen($tmpDir . '/dpdPrivate.csv', 'w');
				if ($file === false) {
					$result['error']++;

					return $result;
				}
				$event = new Event([
					'data'   => $orders,
					'file'   => &$file,
					'result' => &$result,
				]);
				$this->eventDispatcher->dispatch($event, 'mojedpd.export.generateOrdersCsv');
				fclose($file);
				$result['file'] = $tmpDir . '/dpdPrivate.csv';
			} else if ($exportCsvType === 1) {
				$file = fopen($tmpDir . '/dpdPrivate.csv', 'w');
				if ($file === false) {
					$result['error']++;

					return $result;
				}
				fwrite($file, implode(';', ['Name', 'Address', 'City', 'Zip', 'Country', 'DPD Service', 'Phone',
						'Email', 'Currency', 'Is COD', 'COD price', 'Order']) . PHP_EOL);
				foreach ($orders as $v) {
					$d = $v->getOrder()->getAddressDelivery();
					if (!$d) {
						continue;
					}

					$cod = $v->getOrder()->getPaymentIdent() === 'cod' ? $v->getOrder()->getPrice() : 0;

					if ($cod && strtoupper($v->getOrder()->getCurrencyCode()) === 'CZK') {
						$cod = (int) round($cod);
					}

					$name = $d->getFirstName() . ' ' . $d->getLastName();
					if ($d->getCompany()) {
						$name .= ' (' . $d->getCompany() . ')';
					}

					$name = substr(trim($name), 0, 32);

					fwrite($file, implode(';', [$name, $d->getStreet(), $d->getCity(), $d->getPostal(),
							strtoupper($d->getCountry() ? $d->getCountry()->getId() : 'CZ'), 40054, $d->getPhone(), $d->getEmail(),
							$cod > 0 ? 'CZK' : '', $cod > 0 ? '1' : '', $cod > 0 ? $cod : '0',
							$v->getOrder()->getId()]) . PHP_EOL);

					$v->export('csv');
					$this->em->persist($v);
					$result['ok']++;
				}
				fclose($file);
				$result['file'] = $tmpDir . '/dpdPrivate.csv';
			} else if ($exportCsvType === 2) {
				$file = fopen($tmpDir . '/dpd.csv', 'w');
				if ($file === false) {
					$result['error']++;

					return $result;
				}
				fwrite($file, implode(';', [
						'Name', 'Address', 'City', 'Zip', 'Country', 'DPD Service', 'Phone country code', 'Phone',
						'Email', 'Currency',
						'Is COD', 'COD price',
						'Order', 'Parcel shop id', 'Parcel shop name', 'Parcel shop street', 'Parcel shop house number',
						'Parcel shop post code', 'Parcel shop city', 'Parcel shop phone',
						'Parcel shop country']) . PHP_EOL);
				foreach ($orders as $v) {
					$service = $v->getService();
					if (!in_array($service, [
						MojedpdOrder::SERVICE_DPD_CLASSIC,
						MojedpdOrder::SERVICE_DPD_PICKUP,
						MojedpdOrder::SERVICE_DPD_PRIVATE,
					], true)) {
						continue;
					}

					if (!$v->getParcelId()) {
						continue;
					}

					$serviceId     = Helpers::getServiceIdByServiceName($service);
					$isDpdPickup   = $service === MojedpdOrder::SERVICE_DPD_PICKUP;
					$psId          = $v->getParcelId();
					$psName        = $v->parcelName ?? '';
					$psStreet      = $v->parcelAddress ?? '';
					$psHouseNumber = '';
					$psPostCode    = '';
					$psCity        = '';
					$psPhone       = '';
					$psCountry     = '';
					if ($isDpdPickup && ($pickupData = $this->branches->getBranchById($v->getParcelId()))) {
						$psId          = $pickupData->id;
						$psName        = $pickupData->company;
						$psStreet      = $pickupData->street;
						$psHouseNumber = $pickupData->house_number;
						$psPostCode    = $pickupData->postcode;
						$psCity        = $pickupData->city;
						$psPhone       = $pickupData->phone;
						$psCountry     = substr((string) $pickupData->id, 0, 2);
					}
					$d = $v->getOrder()->getAddressDelivery();
					if (!$d) {
						continue;
					}

					$cod     = $v->getOrder()->getPaymentIdent() === 'cod' ? $v->getOrder()->getPrice() : 0;
					$country = strtoupper($d->getCountry() ? $d->getCountry()->getId() : 'CZ');

					if ($cod && strtoupper($v->getOrder()->getCurrencyCode()) === 'CZK') {
						$cod = (int) round($cod);
					}

					$phoneUtil        = PhoneNumberUtil::getInstance();
					$phoneParsed      = $phoneUtil->parse((string) $d->getPhone(), $country);
					$phone            = $phoneParsed->getNationalNumber();
					$phoneCountryCode = $phoneParsed->getCountryCode();

					$name = $d->getFirstName() . ' ' . $d->getLastName();
					if ($d->getCompany()) {
						$name .= ' (' . $d->getCompany() . ')';
					}

					$name = substr(trim($name), 0, 32);

					fwrite($file, implode(';', [
							$name, $d->getStreet(), $d->getCity(), $d->getPostal(),
							$country,
							$serviceId, $phoneCountryCode, $phone, $d->getEmail(), $cod > 0 ? 'CZK' : '',
							$cod > 0 ? '1' : '',
							$cod > 0 ? $cod : '0', $v->getOrder()->getId(), $psId,
							$psName, $psStreet, $psHouseNumber, $psPostCode, $psCity, $psPhone, $psCountry,
						]) . PHP_EOL);

					$v->export('csv');
					$this->em->persist($v);
					$result['ok']++;
				}
				fclose($file);
				$result['file'] = $tmpDir . '/dpd.csv';
			} else {
				$file = fopen($tmpDir . '/dpd.csv', 'w');
				if ($file === false) {
					$result['error']++;

					return $result;
				}

				fwrite($file, implode(';', [
						'Name', 'Address', 'City', 'Zip', 'Country', 'DPD Service', 'Phone country code', 'Phone',
						'Email',
						'Currency', 'COD price', 'Order', 'Parcel shop id', 'Payment type', 'Privat phone',
						'Privat email', 'PickUp phone',
					]) . PHP_EOL);

				foreach ($orders as $v) {
					$service = $v->getService();
					if (!in_array($service, [
						MojedpdOrder::SERVICE_DPD_CLASSIC,
						MojedpdOrder::SERVICE_DPD_PICKUP,
						MojedpdOrder::SERVICE_DPD_PRIVATE,
					], true)) {
						continue;
					}

					$serviceId = Helpers::getServiceIdByServiceName($service);
					$psId      = $v->getParcelId() ?? '';
					$d         = $v->getOrder()->getAddressDelivery();
					if (!$d) {
						continue;
					}

					$v->getOrder()->getPaymentIdent();

					$cod     = $v->getOrder()->getPaymentIdent() === 'cod' ? $v->getOrder()->getPrice() : 0;
					$country = strtoupper($d->getCountry() ? $d->getCountry()->getId() : 'CZ');

					if ($cod && strtoupper($v->getOrder()->getCurrencyCode()) === 'CZK') {
						$cod = (int) round($cod);
					}

					$phoneUtil        = PhoneNumberUtil::getInstance();
					$phoneParsed      = $phoneUtil->parse((string) $d->getPhone(), $country);
					$phone            = $phoneParsed->getNationalNumber();
					$phoneCountryCode = $phoneParsed->getCountryCode();

					$name = $d->getFirstName() . ' ' . $d->getLastName();
					if ($d->getCompany()) {
						$name .= ' (' . $d->getCompany() . ')';
					}
					fwrite($file, implode(';', [
							trim($name), $d->getStreet(), $d->getCity(), $d->getPostal(), $country,
							$serviceId, $phoneCountryCode, $phone, $d->getEmail(),
							$v->getOrder()->getCurrencyCode(), $cod > 0 ? $cod : '0',
							$v->getOrder()->getId(), $psId, 2, $phone, $d->getEmail(), $phone,
						]) . PHP_EOL);

					$v->export('csv');
					$this->em->persist($v);
					$result['ok']++;
				}

				fclose($file);

				$result['file'] = $tmpDir . '/dpd.csv';
			}

			$this->em->flush();
			$this->em->commit();
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}
			Debugger::log($e->getMessage(), 'mojedpdCsv');
			$result['ok'] = 0;
		}

		return $result;
	}

}
