<?php declare(strict_types = 1);

namespace Mojedpd\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Mojedpd\Model\Entities\MojedpdOrder;

/**
 * @method MojedpdOrder|null getReference($id)
 * @method MojedpdOrder[] getAll()
 * @method MojedpdOrder|null get($id)
 */
class OrdersExported extends BaseEntityService
{
	protected $entityClass = MojedpdOrder::class;

	/**
	 * Vrati seznam objednavek, ktere jeste nebyly exportovany do XML. Možno vyfiltrovat podle id objednávek
	 *
	 * @return MojedpdOrder[]
	 */
	public function getOrders(array $filtrIds = [], bool $notExported = true): array
	{
		$arr = [];
		//objednavky ktere nejsou v seznamu zpracovanych objednavek
		$ordersQuery = $this->getEr()->createQueryBuilder('dpdo')
			->select('o, dpdo, oad, oai, os, s')
			->join('dpdo.order', 'o')
			->leftJoin('o.addressDelivery', 'oad')
			->leftJoin('o.addressInvoice', 'oai')
			->leftJoin('o.spedition', 'os')
			->leftJoin('os.spedition', 's')
			->orderBy('o.id')->groupBy('o.id');

		if ($notExported) {
			$ordersQuery->where('dpdo.exported IS NULL');
		}

		if ($filtrIds !== []) {
			$ordersQuery->andWhere('dpdo.order IN (:ids)')->setParameter('ids', $filtrIds);
		}

		foreach ((array) $ordersQuery->getQuery()->getResult() as $row) {
			/** @var MojedpdOrder $row */
			$arr[$row->getOrder()->getId()] = $row;
		}

		return $arr;
	}

	/**
	 * @return MojedpdOrder[]
	 */
	public function getOrdersNotCompleted(): array
	{
		$data = [];

		foreach ((array) $this->getEr()->createQueryBuilder('dpdo')
			->where('dpdo.lastStatus NOT IN (:status) OR dpdo.lastStatus IS NULL')
			->andWhere('dpdo.shipmentId IS NOT NULL')
			->setParameters(new ArrayCollection([new Parameter('status', [
				MojedpdOrder::statusCompleted,
				MojedpdOrder::statusNotFound,
				MojedpdOrder::statusCanceled,
				MojedpdOrder::statusReturned,
			])]))->getQuery()
			->getResult() as $row) {
			/** @var MojedpdOrder $row */
			$data[$row->getOrder()->getId()] = $row;
		}

		return $data;
	}

	/**
	 * @param int[]|string[] $orderIds
	 */
	public function markAsCompleted(array $orderIds): void
	{
		$this->em->getConnection()->executeStatement("UPDATE mojedpd__order SET last_status = '" . MojedpdOrder::statusCompleted . "' 
			WHERE order_id IN (" . implode(',', $orderIds) . ")");
	}

	public function findIdByPackageNumber(string $packageNumber): array
	{
		$ids = [];

		foreach ($this->em->getConnection()->fetchAllAssociative("SELECT o.order_id 
				FROM mojedpd__order o
				INNER JOIN mojedpd__order_parcel op ON o.order_id = op.order_id
				WHERE op.parcel_number LIKE :number", [
			'number' => "%$packageNumber%",
		]) as $row) {
			/** @var array $row */
			$ids[] = (int) $row['order_id'];
		}

		return $ids;
	}
}
