<?php declare(strict_types = 1);

namespace Mojedpd\AdminModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Event\Event;
use EshopOrders\AdminModule\Model\Dao\OrderExpedition;
use EshopOrders\AdminModule\Model\Orders;
use Mojedpd\Model\Entities\MojedpdOrder;
use Mojedpd\Model\MojeDpdConfig;
use Mojedpd\Model\OrderApiService;
use Mojedpd\Model\OrdersExported;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ExpeditionSubscriber implements EventSubscriberInterface
{
	protected OrdersExported  $ordersExported;
	protected Translator      $translator;
	protected OrderApiService $apiService;

	public function __construct(
		OrdersExported  $ordersExported,
		Translator      $translator,
		OrderApiService $apiService
	)
	{
		$this->ordersExported = $ordersExported;
		$this->translator     = $translator;
		$this->apiService     = $apiService;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			Orders::class . '::processOrdersForExpedition' => 'processOrdersForExpedition',
			Orders::class . '::sendExpeditionData'         => 'sendExpeditionData',
			Orders::class . '::generateLabel'              => 'generateLabel',
		];
	}

	public function processOrdersForExpedition(Event $event): void
	{
		if (MojeDpdConfig::load('exported') !== 'api') {
			return;
		}

		$orderExpeditions = [];
		$ids              = $this->filterOrders($event->data['data']);

		if ($ids) {
			foreach ($this->ordersExported->getOrders($ids) as $id => $expedition) {
				$orderExpeditions[$id] = new OrderExpedition($expedition->getOrder(), $this->translator->translate('mojedpd.nameExpedition'), $expedition);
			}

			$event->data['expedition'] += $orderExpeditions;
		}
	}

	public function sendExpeditionData(Event $event): void
	{
		if (MojeDpdConfig::load('exported') !== 'api') {
			return;
		}

		$ids      = $this->filterOrders($event->data['data']);
		$quantity = $event->data['quantity'] ?? 1;

		if ($ids) {
			$orders = $this->ordersExported->getOrders($ids);

			if ($orders) {
				$result = $this->apiService->generateOrderOverApi($orders, $quantity);

				$event->data['okCount']    += $result['ok'];
				$event->data['errorCount'] += $result['error'];
			}
		}
	}

	public function generateLabel(Event $event): void
	{
		if (MojeDpdConfig::load('exported') !== 'api') {
			return;
		}

		$ids = $this->filterOrders($event->data['data']);

		if ($ids) {
			$orders = $this->ordersExported->getOrders($ids, false);

			if ($orders) {
				$result = $this->apiService->generateLabelPdf($orders);

				if ($result['error'] === 0) {
					foreach ($result['files'] as $file) {
						$event->data['files'][] = $file;
					}
					$event->data['okCount'] += $result['ok'];
				} else {
					$event->data['errorCount'] += $result['error'];
				}
			}
		}
	}

	protected function filterOrders(array $data): array
	{
		$ids = [];
		foreach ($data as $v) {
			if (MojedpdOrder::filterIsDpd((string) $v['service'])) {
				$ids[] = $v['id'];
			}
		}

		return $ids;
	}
}
