<?php declare(strict_types = 1);

namespace Mojedpd\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\GridFilterEvent;
use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\Criteria;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\AdminModule\Components\Order\OrderSpeditionForm;
use EshopOrders\AdminModule\Components\Order\OrderStatusesGrid;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\Model\Event\OrderStatusEvent;
use Exception;
use Mojedpd\Model\Entities\MojedpdOrder;
use Mojedpd\Model\OrderApiService;
use Mojedpd\Model\OrdersExported;
use Nette\Utils\Html;
use Nette\Utils\Strings;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderSubscriber implements EventSubscriberInterface
{
	protected OrdersExported         $ordersExported;
	protected OrderApiService        $orderApiService;
	protected EntityManagerDecorator $em;

	/** @var MojedpdOrder[]|null */
	protected static ?array $allOrders = null;

	public function __construct(
		OrdersExported         $ordersExported,
		OrderApiService        $orderApiService,
		EntityManagerDecorator $em
	)
	{
		$this->ordersExported  = $ordersExported;
		$this->orderApiService = $orderApiService;
		$this->em              = $em;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrdersGrid::class . '::columnSpeditionRender' => 'columnSpeditionRender',
			OrdersGrid::class . '::filterPackageNumber'   => 'gridFilterPackageNumber',
			OrderSpeditionForm::class . '::formSuccess'   => 'orderSpeditionFormSuccess',
			Order::class . '::changeStatus'               => ['orderChangeStatus', 0],
			OrderStatusesGrid::class . '::beforeDelete'   => 'orderStatusDelete',
		];
	}

	public function orderSpeditionFormSuccess(FormSuccessEvent $event): void
	{
		/** @var OrderSpedition $orderSpedition */
		$orderSpedition = $event->custom['entity'];

		if (!$orderSpedition->getOrder()) {
			return;
		}

		/** @var OrderSpeditionForm $control */
		$control      = $event->control;
		$orderId      = $orderSpedition->getOrder()->getId();
		$values       = $event->values;
		$mojedpdOrder = $this->ordersExported->getOrders([$orderId], false)[$orderId] ?? null;

		/** @var Spedition|null $spedition */
		$spedition = null;

		$isMojedpd = false;
		foreach ($control->getSpeditions() as $sped) {
			if (Strings::startsWith($sped->getIdent(), 'dpd') && $sped->getId() == $values->spedition) {
				$isMojedpd = true;
				$spedition = $sped;
				break;
			}
		}

		if (!$spedition) {
			return;
		}

		try {
			if ($isMojedpd) {
				if (!$mojedpdOrder) {
					$mojedpdOrder = new MojedpdOrder($spedition->getIdent(), $orderSpedition->getOrder());

					$this->em->persist($mojedpdOrder);
				}
			}
		} catch (Exception $exception) {
			Debugger::log($exception, 'orderSpeditionForm');

			if ($event->presenter) {
				$event->presenter->flashMessageDanger('mojedpd.orderForm.speditionSaveError');
			}
		}
	}

	public function orderChangeStatus(OrderStatusEvent $event): void
	{
		if (!MojedpdOrder::filterIsDpd((string) $event->order->getSpeditionIdent())) {
			return;
		}

		$isCanceled = false;
		/** @var Collection|array $statuses */
		$statuses = $event->order->getOrderStatuses();
		if ($statuses) {
			if ($statuses instanceof Collection) {
				$statuses = $statuses->toArray();
			}

			foreach ($statuses as $orderStatus) {
				/** @var OrderStatus $orderStatus */
				if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_CANCELED && !$orderStatus->isDeleted()) {
					$isCanceled = true;
				}
			}
		}

		if ($event->status === OrderStatus::STATUS_CANCELED && !$isCanceled) {
			$this->orderApiService->cancelOrder($event->order);
		}
	}

	public function columnSpeditionRender(Event $event): void
	{
		$data    = $event->data;
		$exports = null;

		/** @var Order $order */
		$order = $data['order'];

		/** @var Html $html */
		$html = $data['html'];

		$dpdList = [];
		foreach ($data['ordersBySpedition'] as $k => $v) {
			if (MojedpdOrder::filterIsDpd($k)) {
				$dpdList += $v;
			}
		}

		if ($dpdList) {
			if (self::$allOrders === null) {
				self::$allOrders = $this->ordersExported->getOrders(array_keys($dpdList), false);
			}

			if (isset(self::$allOrders[$order->getId()])) {
				$exports = [self::$allOrders[$order->getId()]];
			}
		} else if (MojedpdOrder::filterIsDpd((string) $order->getSpeditionIdent())) {
			$exports = $this->ordersExported->getOrders([$order->getId()], false);
		}

		if ($exports) {
			foreach ($exports as $export) {
				$h     = $html->addHtml(Html::el('div'));
				$urls  = $export->getTrackingUrls();
				$count = count($urls);
				$i     = 1;
				foreach ($urls as $number => $url) {
					$h->addHtml(Html::el('a', ['target' => '_blank'])
						->setAttribute('href', $url)
						->setText($number)
					);
					if ($i !== $count) {
						$h->addText(', ');
					}
					$i++;
				}
			}
		}
	}

	public function gridFilterPackageNumber(GridFilterEvent $event): void
	{
		$ids = $this->ordersExported->findIdByPackageNumber($event->value);

		if ($ids) {
			$event->criteria->orWhere(Criteria::expr()->in('o.id', $ids));
		}
	}

	public function orderStatusDelete(\EshopOrders\AdminModule\Model\Event\OrderStatusEvent $event): void
	{
		$orderStatus = $event->orderStatus;
		$order       = $orderStatus->getOrder();

		if ($orderStatus->getStatus()->getId() !== OrderStatus::STATUS_SPEDITION || ($order->getSpeditionIdent() && !MojedpdOrder::filterIsDpd($order->getSpeditionIdent()))) {
			return;
		}

		$orderId      = $order->getId();
		$mojeDpdOrder = $this->ordersExported->getOrders([$orderId], false)[$orderId] ?? null;

		if (!$mojeDpdOrder) {
			return;
		}

		$mojeDpdOrder->resetExport();
		$this->em->persist($mojeDpdOrder);
		$this->em->flush($mojeDpdOrder);
	}
}
