<?php declare(strict_types = 1);

namespace Mojedpd\Model\Entities;

use Core\Model\Helpers\Arrays;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\Order;
use Mojedpd\Model\MojeDpdConfig;
use Nette\Utils\DateTime;

/**
 * @ORM\Table(name="mojedpd__order")
 * @ORM\Entity
 */
class MojedpdOrder
{

	public const SERVICE_DPD_PRIVATE    = 'dpdPrivate';
	public const SERVICE_DPD_PICKUP     = 'dpdPickup';
	public const SERVICE_DPD_CLASSIC    = 'dpdClassic';
	public const SERVICE_ID_DPD_PRIVATE = 40054;
	public const SERVICE_ID_DPD_PICKUP  = 50101;
	public const SERVICE_ID_DPD_CLASSIC = 1;

	public const statusCompleted = 'completed';
	public const statusNotFound  = 'notFound';
	public const statusCanceled  = 'canceled';
	public const statusReturned  = 'returned';

	/**
	 * @ORM\Id
	 * @ORM\OneToOne(targetEntity="\EshopOrders\Model\Entities\Order")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected Order $order;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	protected ?string $parcelId = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $parcelName = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $parcelAddress = null;

	/**
	 * @var DateTimeInterface|null
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	protected ?DateTimeInterface $exported = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	private ?string $exportType = null;

	/**
	 * @ORM\Column(type="string", nullable=false)
	 */
	protected string $service;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $shipmentId = null;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $lastStatus = null;

	/**
	 * @var Collection<int, MojedpdParcel>
	 * @ORM\OneToMany(targetEntity="MojedpdParcel", mappedBy="dpdOrder", cascade={"all"}, orphanRemoval=true, indexBy="id")
	 */
	public Collection $associatedNumberPackages;

	public function __construct(string $service, Order $order, ?string $parcelId = null)
	{
		$this->service                  = $service;
		$this->order                    = $order;
		$this->parcelId                 = $parcelId;
		$this->associatedNumberPackages = new ArrayCollection();
	}

	public function getOrder(): Order { return $this->order; }

	public function getParcelId(): ?string { return $this->parcelId; }

	public function export(string $type, ?DateTime $datetime = null): self
	{
		$this->exportType = $type;
		$this->exported   = $datetime ?? new DateTime;

		return $this;
	}

	public function getExportType(): ?string { return $this->exportType; }

	public function getExported(): ?DateTimeInterface { return $this->exported; }

	public function getService(): string { return $this->service; }

	public static function filterIsDpd(string $speditionIdent): bool
	{
		return Arrays::contains(['dpdPickup', 'dpdPrivate', 'dpdClassic', 'dpdEu'], $speditionIdent);
	}

	public function getTrackingUrl(string $number): string
	{
		return sprintf(MojeDpdConfig::load('trackingUrl', ''), $number);
	}

	/**
	 * @return string[]
	 */
	public function getTrackingUrls(): array
	{
		$result = [];

		foreach ($this->associatedNumberPackages as $package) {
			$result[$package->parcelNumber] = $this->getTrackingUrl((string) $package->parcelNumber);
		}

		return $result;
	}

	public function resetExport(): void
	{
		$this->shipmentId = null;
		$this->lastStatus = null;
		$this->exported   = null;

		$this->associatedNumberPackages->clear();
	}
}
