<?php declare(strict_types = 1);

namespace Mojedpd\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Strings;
use EshopOrders\AdminModule\Model\Statuses;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Event\OrderPackageNumbers;
use EshopOrders\Model\ExpeditionLogger;
use EshopOrders\Model\Helpers\OrderHelper;
use Mojedpd\Model\Entities\MojedpdOrder;
use Mojedpd\Model\Entities\MojedpdParcel;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;
use Tracy\Debugger;

class OrderApiService
{
	protected array                  $mojedpd;
	protected EntityManagerDecorator $em;
	protected EventDispatcher        $eventDispatcher;
	protected Statuses               $statuses;
	protected ExpeditionLogger       $expeditionLogger;

	public static array $serviceCodes = [
		'classic'   => ['001'],
		'private'   => ['001', '013'],
		'pickup'    => ['001', '013', '200'],
		'expressEU' => ['030'],
	];

	public function __construct(
		array                  $mojedpd,
		EntityManagerDecorator $em,
		EventDispatcher        $eventDispatcher,
		Statuses               $statuses,
		ExpeditionLogger       $expeditionLogger
	)
	{
		$this->mojedpd          = $mojedpd;
		$this->em               = $em;
		$this->eventDispatcher  = $eventDispatcher;
		$this->statuses         = $statuses;
		$this->expeditionLogger = $expeditionLogger;
	}

	/**
	 * @param MojedpdOrder[] $orders
	 */
	public function generateOrderOverApi(array $orders, int $quantity = 1): array
	{
		$result = [
			'ok'    => 0,
			'error' => 0,
		];


		if (!MojeDpdConfig::load('token')) {
			return $result;
		}

		foreach ($orders as $dpdOrder) {
			$order = $dpdOrder->getOrder();

			try {
				$addrDeli = $order->getAddressDelivery();

				if (!$addrDeli) {
					continue;
				}

				$tmp = Strings::getPhoneObject(
					(string) $addrDeli->getPhone(),
					strtoupper($addrDeli->getCountry() ? $addrDeli->getCountry()->getId() : 'CZ')
				);

				$phonePrefix = $tmp ? $tmp->getCountryCode() : '';
				$phone       = $tmp ? $tmp->getNationalNumber() : '';

				$speditionId = $order->getSpeditionIdent();
				$serviceName = 'private';
				if ($speditionId) {
					switch ($speditionId) {
						case 'dpdPrivate':
							$serviceName = 'private';
							break;
						case 'dpdPickup':
							$serviceName = 'pickup';
							break;
						case 'dpdClassic':
							$serviceName = 'classic';
							break;
						case 'dpdEu':
							$serviceName = 'expressEU';
							break;
						default:
							$serviceName = 'private';
					}
				} else {
					if ($order->getSpedition() && $order->getSpedition()->getSpedition() && $order->getSpedition()->getSpedition()->isPickup) {
						$serviceName = 'pickup';
					} else {
						if (Strings::upper($addrDeli->getCountry() ? $addrDeli->getCountry()->getId() : 'CZ') === 'CZ') {
							$serviceName = 'private';
						} else {
							$serviceName = 'expressEU';
						}
					}
				}
				$elementCodes = self::$serviceCodes[$serviceName];

				$data = [
					'customerId' => (string) MojeDpdConfig::load('customerId'),
					'shipments'  => [],
				];

				$parcels = [];
				for ($i = 0; $i < $quantity; $i++) {
					$parcels[] = [
						'weight' => 1,
					];
				}

				$shipment = [
					'numOrder'        => 1,
					'parcels'         => $parcels,
					'senderAddressId' => (string) MojeDpdConfig::load('senderAddressId'),
					'receiver'        => [
						'name'                => Strings::truncate($addrDeli->getName(), 35, ''),
						'city'                => $addrDeli->getCity(),
						'contactEmail'        => $addrDeli->getEmail(),
						'contactMobile'       => $phone,
						'contactMobilePrefix' => $phonePrefix,
						'contactName'         => Strings::truncate($addrDeli->getName(), 35, ''),
						'contactPhone'        => $phone,
						'contactPhonePrefix'  => $phonePrefix,
						'countryCode'         => Strings::upper($addrDeli->getCountry() ? $addrDeli->getCountry()->getId() : 'CZ'),
						'countryName'         => $addrDeli->getCountry() && $addrDeli->getCountry()->getText() ? $addrDeli->getCountry()->getText()->name : 'CZ',
						'street'              => Strings::truncate((string) $addrDeli->getStreet(), 35, ''),
						'zipCode'             => str_replace('-', '', (string) $addrDeli->getPostal()),
					],
					'service'         => [
						'additionalService'       => [
							'predicts' => [],
						],
						'mainServiceElementCodes' => $elementCodes,
					],
					'reference1'      => (string) $order->getId(),
					'saveMode'        => 'parcel number',
				];

				if ($phone) {
					$shipment['service']['additionalService']['predicts'][] = [
						'destination' => $phonePrefix . $phone,
						'type'        => 'SMS',
					];
				}

				if ($addrDeli->getEmail()) {
					$shipment['service']['additionalService']['predicts'][] = [
						'destination' => $addrDeli->getEmail(),
						'type'        => 'email',
					];
				}

				if ($addrDeli->getCompany()) {
					$shipment['receiver']['companyName'] = Strings::truncate($addrDeli->getCompany(), 35, '');
				}

				if ($serviceName === 'pickup' && $dpdOrder->getParcelId()) {
					$shipment['service']['additionalService']['pudoId'] = (string) $dpdOrder->getParcelId();
				}

				if ($order->getPaymentIdent() === 'cod') {
					$curr = $order->getCurrencyCode();
					if (!$curr) {
						$result['error']++;
						continue;
					}

					$decimals = $curr === 'CZK' || !$order->currency
						? 0
						: $order->currency->decimals;

					$cod = round($order->getPrice(true), $decimals);

					if ($curr === 'EUR') {
						$cod = OrderHelper::roundSkCod($cod);
					}

					$shipment['service']['additionalService']['cod'] = [
						'amount'      => $cod,
						'currency'    => $curr,
						'reference'   => (string) $order->getId(),
						'split'       => 'First parcel',
						'paymentType' => 'Credit Card',
					];
				}

				$data['shipments'][] = $shipment;

				$curl = curl_init();
				curl_setopt_array($curl, [
					CURLOPT_URL            => MojeDpdConfig::load('apiBaseUrl') . '/shipments',
					CURLOPT_RETURNTRANSFER => true,
					CURLOPT_ENCODING       => '',
					CURLOPT_MAXREDIRS      => 10,
					CURLOPT_TIMEOUT        => 0,
					CURLOPT_FOLLOWLOCATION => true,
					CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
					CURLOPT_CUSTOMREQUEST  => 'POST',
					CURLOPT_POSTFIELDS     => Json::encode($data),
					CURLOPT_HTTPHEADER     => [
						'Content-Type: application/json',
						'Authorization: Bearer ' . MojeDpdConfig::load('token'),
					],
				]);

				$response = curl_exec($curl);
				$httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
				$error    = curl_error($curl);
				curl_close($curl);

				if ($httpCode === 200) {
					$responseData = Json::decode((string) $response);
					$event        = new OrderPackageNumbers($order);

					foreach ($responseData->shipmentResults as $shipmentResult) {
						if ($shipmentResult->errors) {
							foreach ($shipmentResult->errors as $error) {
								$this->expeditionLogger->logError('mojeDpd', $error->errorCode . ' - ' . $error->errorContent, (int) $order->getId());
							}

							throw new \Exception((string) $response);
						}

						$shipmentId           = $shipmentResult->shipmentId;
						$dpdOrder->shipmentId = (string) $shipmentId;

						foreach ($dpdOrder->associatedNumberPackages as $packageId => $package) {
							$dpdOrder->associatedNumberPackages->remove($packageId);
							$this->em->remove($package);
						}

						foreach ($shipmentResult->parcelResults as $parcelResult) {
							$package               = new MojedpdParcel($dpdOrder);
							$package->parcelId     = (string) $parcelResult->parcelId;
							$package->parcelNumber = $parcelResult->parcelNumber ? (string) $parcelResult->parcelNumber : null;
							$this->em->persist($package);
							$dpdOrder->associatedNumberPackages->add($package);

							$event->packageNumbers[] = $package->parcelNumber;
							$event->trackingUrls[]   = $dpdOrder->getTrackingUrl((string) $package->parcelNumber);
						}
					}

					$dpdOrder->export('api');
					$this->em->persist($dpdOrder);
					$this->em->flush();

					$result['ok']++;

					$this->eventDispatcher->dispatch($event, 'eshopOrders.packageNumberAssigned');
				} else {
					throw new \Exception((string) $response);
				}
			} catch (\Exception $e) {
				Debugger::log($e->getMessage(), 'mojedpdApi');
				Debugger::log(json_encode($shipment), 'mojedpdApi');
				$result['error']++;
				bdump($e->getMessage());
			}
		}

		return $result;
	}

	/**
	 * @param MojedpdOrder[] $orders
	 *
	 * @return array
	 */
	public function generateLabelPdf(array $orders): array
	{
		$result = [
			'ok'    => 0,
			'error' => 0,
			'files' => [],
		];

		if (!MojeDpdConfig::load('token')) {
			return $result;
		}

		foreach ($orders as $order) {
			foreach ($order->associatedNumberPackages as $package) {
				$data = [
					'shipmentIdList' => (int) $order->shipmentId,
					'labelSize'      => MojeDpdConfig::load('labelSize'),
					'printFormat'    => 'pdf',
					'parcelNumber'   => $package->parcelNumber,
				];

				$curl = curl_init();
				curl_setopt_array($curl, [
					CURLOPT_URL            => MojeDpdConfig::load('apiBaseUrl') . '/labels?' . http_build_query($data),
					CURLOPT_RETURNTRANSFER => true,
					CURLOPT_ENCODING       => '',
					CURLOPT_MAXREDIRS      => 10,
					CURLOPT_TIMEOUT        => 0,
					CURLOPT_FOLLOWLOCATION => true,
					CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
					CURLOPT_CUSTOMREQUEST  => 'GET',
					CURLOPT_HTTPHEADER     => [
						'Content-Type: application/json',
						'Authorization: Bearer ' . MojeDpdConfig::load('token'),
					],
				]);

				$response = curl_exec($curl);
				$httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
				$error    = curl_error($curl);
				curl_close($curl);

				if ($httpCode === 200) {
					$responseData = Json::decode((string) $response);
					FileSystem::createDir(TMP_DIR . '/dpdLabels');
					if (file_put_contents(TMP_DIR . '/dpdLabels/stitky.pdf', base64_decode(explode(',', $responseData->pdfFile, 2)[1]))) {
						$result['files'][] = TMP_DIR . '/dpdLabels/stitky.pdf';
						$result['ok']++;
					} else {
						$result['error']++;
					}
				} else {
					Debugger::log($response, 'mojedpdApi');
					$result['error']++;
				}
			}
		}

		return $result;
	}

	public function cancelOrder(Order $order): void
	{
		$speditionId = (string) $order->getSpeditionIdent();

		if (!MojedpdOrder::filterIsDpd($speditionId) || !MojeDpdConfig::load('token')) {
			return;
		}

		/** @var MojedpdOrder|null $dpdOrder */
		$dpdOrder = $this->em->getRepository(MojedpdOrder::class)->findOneBy(['order' => $order->getId()]);
		if (!$dpdOrder || $dpdOrder->lastStatus === MojedpdOrder::statusCanceled || !$dpdOrder->shipmentId) {
			return;
		}

		$data = [
			'customerId'     => (string) MojeDpdConfig::load('customerId'),
			'shipmentIdList' => [
				(int) $dpdOrder->shipmentId,
			],
		];

		$curl = curl_init();
		curl_setopt_array($curl, [
			CURLOPT_URL            => MojeDpdConfig::load('apiBaseUrl') . '/shipments/cancellation',
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING       => '',
			CURLOPT_MAXREDIRS      => 10,
			CURLOPT_TIMEOUT        => 0,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST  => 'PUT',
			CURLOPT_POSTFIELDS     => Json::encode($data),
			CURLOPT_HTTPHEADER     => [
				'Content-Type: application/json',
				'Authorization: Bearer ' . MojeDpdConfig::load('token'),
			],
		]);

		$response = curl_exec($curl);
		$httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		$error    = curl_error($curl);
		curl_close($curl);

		if ($httpCode === 200) {
			$this->em->getConnection()->executeStatement("UPDATE mojedpd__order SET last_status = '" . MojedpdOrder::statusCanceled . "' WHERE order_id = " . $dpdOrder->getOrder()->getId());
		} else {
			Debugger::log($response, 'mojedpdApi');
		}
	}

	public function checkCompleted(MojedpdOrder $dpdOrder): bool
	{
		$curl = curl_init();

		$parcels = [];
		foreach ($this->em->getConnection()->fetchAllAssociative("SELECT parcel_number FROM mojedpd__order_parcel WHERE order_id = ?", [
			$dpdOrder->getOrder()->getId(),
		]) as $row) {
			if ($row['parcel_number']) {
				$parcels[] = (string) $row['parcel_number'];
			}
		}

		if (empty($parcels)) {
			return false;
		}

		$data = [
			'language'      => 'CZ',
			'parcelNumbers' => $parcels,
		];

		curl_setopt_array($curl, [
			CURLOPT_URL            => MojeDpdConfig::load('trackingApiBaseUrl') . '/tracking/v2/parcels',
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING       => '',
			CURLOPT_MAXREDIRS      => 10,
			CURLOPT_TIMEOUT        => 0,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST  => 'POST',
			CURLOPT_POSTFIELDS     => Json::encode($data),
			CURLOPT_HTTPHEADER     => [
				'Content-Type: application/json',
				'apiKey: ' . MojeDpdConfig::load('trackingApiKey'),
			],
		]);

		$response = curl_exec($curl);
		$httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
		$error    = curl_error($curl);
		curl_close($curl);

		try {
			if ($httpCode === 200) {
				$responseJson = Json::decode((string) $response, Json::FORCE_ARRAY);

				foreach ($responseJson as $v) {
					foreach (array_reverse($v['parcelEvents'] ?? []) as $e) {
						if ($e['statusId'] === 'SPR' || $e['statusLabel'] === 'Status parcel - Return') {
							$this->em->getConnection()->executeStatement("UPDATE mojedpd__order SET last_status = '" . MojedpdOrder::statusReturned . "' WHERE order_id = " . $dpdOrder->getOrder()->getId());

							return false;
						}

						if ($e['statusFamilyLabel'] === 'DELIVERED' || $e['statusId'] === 'DEY' || $e['statusId'] === 'DODEY') {
							$r = $this->statuses->changeStatus([$dpdOrder->getOrder()->getId()], OrderStatus::STATUS_FINISHED);

							if ($r) {
								$this->em->getConnection()->executeStatement("UPDATE mojedpd__order SET last_status = '" . MojedpdOrder::statusCompleted . "' WHERE order_id = " . $dpdOrder->getOrder()
										->getId());
							}

							return true;
						}
					}
				}
			} else {
				$this->em->getConnection()->executeStatement("UPDATE mojedpd__order SET last_status = '" . MojedpdOrder::statusNotFound . "' WHERE order_id = " . $dpdOrder->getOrder()->getId());
				Debugger::log('Http code error: ' . $httpCode, 'mojedpdApi');
				Debugger::log($response, 'mojedpdApi');
			}
		} catch (\Exception $e) {
			Debugger::log('Error: ' . $response, 'mojedpdApi');
			Debugger::log($error, 'mojedpdApi');
			Debugger::log($e);
		}

		return false;
	}
}
