<?php declare(strict_types = 1);

namespace Mojedpd\Model\Subscribers;

use Core\Model\Countries;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use EshopOrders\FrontModule\Model\CartFacade;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\FrontModule\Model\Event\OrderFormEvent;
use EshopOrders\FrontModule\Model\Event\SaveOrderFormDataEvent;
use EshopOrders\FrontModule\Model\Event\SetOrderFormDataEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\PaymentSpeditions;
use Mojedpd\Model\Branches;
use Mojedpd\Model\Entities\MojedpdOrder;
use Mojedpd\Model\Exceptions\PickupNotFound;
use Nette\Application\LinkGenerator;
use Nette\Http\Request;
use Nette\Localization\Translator;
use Nette\Utils\Json;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderFormSubscriber implements EventSubscriberInterface
{

	protected EntityManagerDecorator $em;
	protected Translator             $translator;
	protected Branches               $branches;
	protected Countries              $countriesService;
	protected PaymentSpeditions      $paymentSpeditions;
	protected LinkGenerator          $linkGenerator;
	protected Request                $httpRequest;
	protected CartFacade             $cartFacade;

	public function __construct(
		EntityManagerDecorator $em,
		Translator             $translator,
		Branches               $branches,
		Countries              $countriesService,
		PaymentSpeditions      $paymentSpeditions,
		LinkGenerator          $linkGenerator,
		Request                $request,
		CartFacade             $cartFacade
	)
	{
		$this->em                = $em;
		$this->translator        = $translator;
		$this->branches          = $branches;
		$this->countriesService  = $countriesService;
		$this->paymentSpeditions = $paymentSpeditions;
		$this->linkGenerator     = $linkGenerator;
		$this->httpRequest       = $request;
		$this->cartFacade        = $cartFacade;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.createOrderForm'        => ['createOrderForm', 100],
			'eshopOrders.setOrderFormData'       => ['setOrderFormData', 100],
			'eshopOrders.orderFormValidate'      => ['orderFormValidate', 100],
			'eshopOrders.orderBeforeSave'        => ['orderBeforeSave', 100],
			'eshopCheckout.orderBeforeSave'      => ['orderCheckoutBeforeSave', 100],
			'eshopOrders.saveOrderFormDataStep2' => ['saveOrderFormDataStep2', 100],
			'eshopOrders.saveOrderFormDataStep3' => ['saveOrderFormDataStep3', 100],
		];
	}

	public function createOrderForm(OrderFormEvent $event): void
	{
		$form    = $event->form;
		$methods = $this->getMojedpdSpeditionIds();

		$cart        = $this->cartFacade->getCart();
		$templates   = $form->getCustomData('speditionServiceTemplates') ?: [];
		$templates[] = __DIR__ . '/speditionTemplate.latte';
		$form->addCustomData('speditionServiceTemplates', $templates);
		$container = $form->addContainer('mojedpdPickup');

		foreach ($methods as $row) {
			$spedition = $row->getSpedition();
			if (!$spedition->isPickup) {
				continue;
			}

			$speditionId = $spedition->getId();

			foreach ($row->getCountries() as $country) {
				$k = $country->getId() . '_' . $speditionId;
				$container->addHidden($k);

				$dataEl = $container->addHidden($k . '_data');
				$pointEl = $container->addText($k . '_p')
					->setHtmlAttribute('readonly')
					->setHtmlAttribute('data-mojedpd-choose-pickup-point')
					->setPlaceholder('mojedpdFront.orderForm.chooseAddress')
					->setHtmlAttribute('data-country', $country->getId())
					->setHtmlAttribute('data-data', $dataEl->getHtmlId())
					->setHtmlAttribute('data-target', $container->getComponent($k)->getHtmlId());

				if ($cart->hasDisabledDeliveryBoxes()) {
					$pointEl->setHtmlAttribute('data-mojedpd-disable-box', '1');
				}

				/** @phpstan-ignore-next-line */
				$event->template->mojeDpdLastKey = $k;
			}
		}
	}

	public function setOrderFormData(SetOrderFormDataEvent $event): void
	{
		$data  = $event->data;
		$form  = $event->form;
		$value = $form->getValues();

		foreach ($data['mojedpdPickup'] ?? [] as $k => $v) {
			if (isset($form->getComponent('mojedpdPickup', false)[$k])) {
				$form->getComponent('mojedpdPickup')[$k]->setDefaultValue($v);
			}
		}

		try {
			$parcel = $this->getParcelByData($data);
		} catch (PickupNotFound $ex) {
			$parcel         = null;
		}
		if ($parcel) {
			foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $k) {
				$form->getComponent($k . '2')->setValue($form->getComponent($k . '2')->isRequired() && !$value[$k] ? 'placeholder' : ($value[$k] ?: null));
			}

			$form->getComponent('useAddrDeli')->setValue(true);
			$form->getComponent('street2')->setValue(trim($parcel->street . ' ' . $parcel->house_number));
			$form->getComponent('city2')->setValue($parcel->city);
			$form->getComponent('postal2')->setValue(str_replace(' ', '', $parcel->postcode));
			$form->getComponent('country2')->setValue($data['speditionCountry']);
		}
	}

	public function orderFormValidate(OrderFormEvent $event): void
	{
		$form   = $event->form;
		$values = $form->getValues();

		$speditions = $this->getMojedpdSpeditionIds();
		$spedition  = $speditions[$values->speditions->{$values->speditionCountry}] ?? null;

		if ($spedition && $spedition->getSpedition()->isPickup) {
			$parcelId = $values->mojedpdPickup->{$values->speditionCountry . '_' . $spedition->getSpedition()->getId()};

			if (!$parcelId) {
				$form->addError($this->translator->translate('mojedpdFront.orderForm.mojedpdIdMissing'));
			}
		}
	}

	public function saveOrderFormDataStep2(SaveOrderFormDataEvent $event): void
	{
		$data    = &$event->data;
		$request = $this->httpRequest;
		$form    = &$event->form;

		$paymentSpedition = $this->getMojedpdSpeditionIds()[$data['spedition']] ?? null;
		if ($paymentSpedition) {
			$k = $data['speditionCountry'] . '_' . $data['spedition'];

			$data['mojedpdPickup'][$k]        = $request->getPost('mojedpdPickup')[$k];
			$data['mojedpdPickup'][$k . '_p'] = $request->getPost('mojedpdPickup')[$k . '_p'];

			$pickupNotFound = false;
			try {
				$parcel = $this->getParcelByData($data);
			} catch (PickupNotFound $ex) {
				$parcel         = null;
				$pickupNotFound = true;
			}
			if ($parcel) {
				$data['disableDeliveryAddressSpedition'] = 'mojeDpd';
				$data['disableDeliveryAddress']          = true;
				$data['useAddrDeli']                     = false;

				$form->getComponent('useAddrDeli')->setValue(true);

				return;
			}

			if ($pickupNotFound) {
				$form->addError($this->translator->translate('mojedpdFront.orderForm.busyPickup'));
			}
		}

		if ($data['disableDeliveryAddressSpedition'] === 'mojeDpd') {
			unset($data['disableDeliveryAddress']);
		}
	}

	public function saveOrderFormDataStep3(SaveOrderFormDataEvent $event): void
	{
		$data = &$event->data;
		$form = &$event->form;

		try {
			$parcel = $this->getParcelByData($data);
		} catch (PickupNotFound $ex) {
			$parcel = null;
		}
		if ($parcel) {
			$d = [];
			foreach (['company', 'firstName', 'lastName', 'email', 'phone'] as $k) {
				$d[$k . '2'] = $data[$k];
			}

			$countryId   = $data['speditionCountry'];
			$countryText = $this->countriesService->getAllNameColumn()[$countryId] ?? $countryId;

			$d += [
				'useAddrDeli'  => true,
				'street2'      => trim($parcel->street . ' ' . $parcel->house_number),
				'city2'        => $parcel->city,
				'postal2'      => str_replace(' ', '', $parcel->postcode),
				'country2'     => $countryId,
				'country2Text' => $countryText,
			];

			$data = array_merge($data, $d);
			$form->setValues($d);
		}
	}

	public function orderBeforeSave(OrderEvent $event): void
	{
		$values = $event->formData;
		/** @var Order|null $order */
		$order     = $event->order;
		$spedition = null;

		if (!$order) {
			return;
		}

		if (isset($values['speditions'])) {
			$dpdSpeditions = $this->getMojedpdSpeditionIds();
			$spedition     = $dpdSpeditions[$values['speditions'][$values['speditionCountry']]] ?? null;
		} else if ($order->getSpedition()) {
			$spedition = $order->getSpedition();
		}

		if ($spedition) {
			$spedition = $spedition->getSpedition();

			if ($spedition->getIdent() === 'dpdPickup') {
				$parcelId = $values['mojedpdPickup'][$values['speditionCountry'] . '_' . $spedition->getId()];
				$parcel   = $this->branches->getBranchById($parcelId);

				$entity                = new MojedpdOrder(MojedpdOrder::SERVICE_DPD_PICKUP, $order, $parcelId);

				if ($parcel) {
					$entity->parcelName    = $parcel->company;
					$entity->parcelAddress = "{$parcel->street} {$parcel->house_number}, {$parcel->postal} {$parcel->city}";
				} else {
					$dataString = $values['mojedpdPickup'][$values['speditionCountry'] . '_' . $spedition->getId() . '_data'] ?? null;

					try {
						$data = Json::decode((string) $dataString, Json::FORCE_ARRAY);
						if ($data) {
							$entity->parcelName    = $data['name'];

							$entity->parcelAddress = "{$data['address']['street']} {$data['address']['propertyNumber']}, {$data['address']['zip']} {$data['address']['city']}";
						}
					} catch(\Exception $e) {
						Debugger::log('failed parse fallback json pickup point: ' . $dataString, 'mojedpdApi');
					}
				}

				$this->em->persist($entity);
				$orderSpedition = $event->order ? $event->order->getSpedition() : null;
				if ($orderSpedition) {
					$orderSpedition->deliveryPointId = (string) $parcelId;
					$this->em->persist($orderSpedition);
				}
			} else if (in_array($spedition->getIdent(), ['dpdPrivate', 'dpdClassic'])) {
				$entity = new MojedpdOrder($spedition->getIdent(), $order);
				$this->em->persist($entity);

				$orderSpedition = $event->order ? $event->order->getSpedition() : null;
				if ($orderSpedition) {
					$orderSpedition->deliveryPointId = null;
					$this->em->persist($orderSpedition);
				}
			}
		}
	}

	public function orderCheckoutBeforeSave(OrderEvent $event): void
	{
		if (!$event->order || !$event->orderSpedition || !$event->orderSpedition->getSpedition() || $event->order->getId()) {
			return;
		}

		$dpdSpeditions = $this->getMojedpdSpeditionIds();

		$spedition = $dpdSpeditions[$event->orderSpedition->getSpeditionId()] ?? null;
		if ($spedition) {
			$spedition = $spedition->getSpedition();

			if ($spedition->getIdent() === 'dpdPickup') {
				$this->em->persist(new MojedpdOrder(MojedpdOrder::SERVICE_DPD_PICKUP, $event->order));
			} else if (in_array($spedition->getIdent(), ['dpdPrivate', 'dpdClassic'])) {
				$entity = new MojedpdOrder($spedition->getIdent(), $event->order);
				$this->em->persist($entity);
			}
		}
	}

	protected function getMojedpdSpeditionIds(): array
	{
		$result = [];
		foreach ($this->paymentSpeditions->getAllPublished() as $row) {
			if (!in_array($row->getSpedition()->getIdent(), ['dpdPickup', 'dpdPrivate', 'dpdClassic'])) {
				continue;
			}

			$result[$row->getSpedition()->getId()] = $row;
		}

		return $result;
	}

	protected function getParcelByData(array $data): ?\stdClass
	{
		$dpdSpeditions = $this->getMojedpdSpeditionIds();

		if (isset($dpdSpeditions[$data['spedition']])) {
			$spedition = $dpdSpeditions[$data['spedition']]->getSpedition();
			if ($spedition->isPickup) {
				$parcelId = $data['mojedpdPickup'][$data['speditionCountry'] . '_' . $spedition->getId()] ?? null;

				if ($parcelId) {
					$branch = $this->branches->getBranchById($parcelId);
					if (!$branch) {
						throw new PickupNotFound;
					}

					return $branch;
				}
			}
		}

		return null;
	}
}
