<?php declare(strict_types = 1);

namespace MoneyS3\CronModule\Presenters;

use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Orders;
use MoneyS3\Model\OrdersExported;
use MoneyS3\Model\OrderXmlService;
use Nette\Application\Responses\TextResponse;
use Nette\Utils\FileSystem;
use Tracy\Debugger;
use Tracy\ILogger;

class ExportPresenter extends BasePresenter
{
	protected Orders          $ordersService;
	protected OrderXmlService $orderXmlService;
	protected OrdersExported  $ordersExported;

	public function __construct(Orders $orders, OrderXmlService $orderXml, OrdersExported $ordersExported)
	{
		parent::__construct();
		$this->ordersService   = $orders;
		$this->orderXmlService = $orderXml;
		$this->ordersExported  = $ordersExported;
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionExportOrders(): void
	{
		set_time_limit(600);

		if ($this->getParameter('from')) {
			$orders = $this->ordersExported->em->getRepository(Order::class)->createQueryBuilder('o')
				->where('o.id >= :from')
				->setParameter('from', $this->getParameter('from'))
				->getQuery()->getResult();
		} else {
			$orders = $this->ordersExported->getOrdersNotExported();
		}

		foreach ($orders as $order) {
			if ($order === null) {
				continue;
			}

			if ($this->exportOrder($order)) {
				$this->ordersExported->markAsDone($order); //vytvori entitu a ulozi do DB
			} else {
				Debugger::log("Failed export order id '{$order->getId()}'", 'moneyS3Export');
			}
		}

		$this->sendResponse(new TextResponse('OK'));
	}

	public function exportOrder(?Order $order): bool
	{
		if (!$order) {
			return false;
		}

		try {
			$dir = ROOT_DIR . '/../money_import/import';
			FileSystem::createDir($dir);
			$orders_file = $dir . "/op-{$order->getId()}.xml";

			$dom = $this->orderXmlService->generateOrderXml($order);

			$ret = $dom->save($orders_file);
			if (!$ret) {
				return false;
			}
		} catch (\Exception $e) {
			Debugger::log($e, ILogger::EXCEPTION);
			LogNotifier::toDevelopers($e->getMessage());

			return false;
		}

		return true;
	}
}
