<?php declare(strict_types = 1);

namespace MoneyS3\Model;

use DOMDocument;
use DOMElement;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderItem;
use MoneyS3\Model\Entities\PaymentText;
use Nette\InvalidArgumentException;

class OrderXmlService
{
	/** @var PaymentText[] */
	private array $paymentTexts;
	protected ProductsFacade $productsService;
	protected PaymentTexts $paymentTextsService;

	public function __construct(ProductsFacade $products, PaymentTexts $paymentTextsService)
	{
		$this->productsService     = $products;
		$this->paymentTextsService = $paymentTextsService;
		$this->paymentTexts = $this->paymentTextsService->getAll();
	}

	public function generateOrderXml(Order $order): DOMDocument
	{
		if ($order->getId() === null) {
			throw new InvalidArgumentException;
		}

		$default_xml = $this->getDefaultOrderXml($order->getId());

		$dom               = new DOMDocument;
		$dom->formatOutput = true;
		$dom->encoding     = 'utf-8';

		$dom->loadXML($default_xml);

		$SeznamObjPrij = $dom->getElementsByTagName('SeznamObjPrij')->item(0);
		$orderXml      = $this->generateOrderInnerXml($dom, $order);

		if ($SeznamObjPrij === null) {
			throw new InvalidArgumentException;
		}

		$SeznamObjPrij->appendChild($orderXml);

		return $dom;
	}

	public function getDefaultOrderXml(int $order_id): string
	{
		return '<' . '?xml version="1.0" encoding="UTF-8"?' . '>' .
			'<MoneyData messageId="' . $order_id . '">' . PHP_EOL .
			'<SeznamObjPrij>' . PHP_EOL .
			'</SeznamObjPrij>' . PHP_EOL .
			'</MoneyData>';
	}

	public function generateOrderInnerXml(DOMDocument $dom, Order $order): DOMElement
	{
		$invoiceAddr  = $order->getAddressInvoice();
		$deliveryAddr = $order->getAddressDelivery();

		$orderXml = $dom->createElement('ObjPrij');

		$child = $dom->createElement('Doklad');
		$orderXml->appendChild($child);

		$child = $dom->createElement('PrimDoklad', (string) $order->getId());
		$orderXml->appendChild($child);

		if ($invoiceAddr) {
			$child = $dom->createElement('Popis', (string) ($invoiceAddr->getCompany() ?: trim($invoiceAddr->getFirstName() . ' ' . $invoiceAddr->getLastName())));
			$orderXml->appendChild($child);
		}

		$child = $dom->createElement('Poznamka', 'Typ: ' . $this->getGroup($order) . '; Poznámka: ' . (string) $order->getMessage());
		$orderXml->appendChild($child);

		$child = $dom->createElement('Vystaveno', $order->getCreatedTime()->format('Y-m-d'));
		$orderXml->appendChild($child);

		if ($invoiceAddr) {
			$deliveryXml = $this->generateDeliveryXml($dom, $order, $invoiceAddr, 'DodOdb');
			$orderXml->appendChild($deliveryXml);
		}

		if ($deliveryAddr) {
			$deliveryXml = $this->generateDeliveryXml($dom, $order, $deliveryAddr);
			$orderXml->appendChild($deliveryXml);
		}

		/** @var int $paymentId */
		$paymentId = $order->getPayment()->getPayment()->getId();
		if (array_key_exists($paymentId, $this->paymentTexts)) {
			$platPodm = $this->paymentTexts[$paymentId]->getText();
			if (trim($platPodm) === '') {
				$platPodm = $order->getPayment()->getName();
			}
		} else {
			$platPodm = $order->getPayment()->getName();
		}
		$child = $dom->createElement('PlatPodm', $platPodm);
		$orderXml->appendChild($child);
		$child = $dom->createElement('Doprava', $order->getSpedition()->getName());
		$orderXml->appendChild($child);

		$child  = $dom->createElement('eshop');
		$child2 = $dom->createElement('IN_Export', "1");
		$child->appendChild($child2);
		$child2 = $dom->createElement('IN_Poznamk', 'Typ: ' . $this->getGroup($order));
		$child->appendChild($child2);
		$orderXml->appendChild($child);

		foreach ($order->getOrderItems() as $orderItem) {
			$itemXml = $this->generateItemXml($dom, $orderItem);
			$orderXml->appendChild($itemXml);
		}

		$itemSpeditionXml = $this->generateSpeditionItemXml($dom, $order);
		$orderXml->appendChild($itemSpeditionXml);

		return $orderXml;
	}

	/**
	 * Konecny prijemce
	 */
	public function generateDeliveryXml(DOMDocument $dom, Order $order, OrderAddress $orderAddress, string $type = 'KonecPrij'): DOMElement
	{
		$deliveryXml = $dom->createElement($type);

		$name  = trim($orderAddress->getFirstName() . ' ' . $orderAddress->getLastName());
		$child = $dom->createElement('Nazev', $name);
		$deliveryXml->appendChild($child);
		if ($type == 'DodOdb') {
			$child  = $dom->createElement('Skupina');
			$child2 = $dom->createElement('Zkratka', $this->getGroup($order));
			$child->appendChild($child2);
			$child2 = $dom->createElement('Nazev', $this->getGroup($order));
			$child->appendChild($child2);
			$deliveryXml->appendChild($child);

			$child = $dom->createElement('ObchNazev', $name);
			$deliveryXml->appendChild($child);
			$child = $dom->createElement('FaktNazev', $name);
			$deliveryXml->appendChild($child);
			$child = $dom->createElement('ICO', $orderAddress->getIdNumber() ?: '');
			$deliveryXml->appendChild($child);
			$child = $dom->createElement('DIC', $orderAddress->getVatNumber() ?: '');
			$deliveryXml->appendChild($child);
		}

		//TODO GUID pokud je to registrovany zakaznik a ma sve ID napojene na Money

		if ($type === 'KonecPrij') {
			$addressXml = $this->generateAddressXml($dom, $orderAddress, 'Adresa');
			$deliveryXml->appendChild($addressXml);
		} else {
			$addressXml = $this->generateAddressXml($dom, $orderAddress, 'ObchAdresa');
			$deliveryXml->appendChild($addressXml);
			$addressXml = $this->generateAddressXml($dom, $orderAddress, 'FaktAdresa');
			$deliveryXml->appendChild($addressXml);
		}

		$phoneXml    = $dom->createElement('Tel');
		$phoneNumber = $dom->createElement('Cislo', (string)$orderAddress->getPhone());
		$phoneXml->appendChild($phoneNumber);
		$deliveryXml->appendChild($phoneXml);

		$emailXml = $dom->createElement('Email', (string)$orderAddress->getEmail());
		$deliveryXml->appendChild($emailXml);

		//TODO platceDPH a fyzickaOsoba ?

		return $deliveryXml;
	}

	/**
	 * Konecny prijemce
	 */
	public function generateAddressXml(DOMDocument $dom, OrderAddress $orderAddress, string $elementName): DOMElement
	{
		$addressXml = $dom->createElement($elementName);

		$street = $dom->createElement('Ulice', (string) $orderAddress->getStreet());
		$addressXml->appendChild($street);
		$place = $dom->createElement('Misto', (string) $orderAddress->getCity());
		$addressXml->appendChild($place);
		$zipCode = $dom->createElement('PSC', (string) $orderAddress->getPostal());
		$addressXml->appendChild($zipCode);

		if ($orderAddress->getCountry()) {
			$country = $dom->createElement('Stat', (string) $orderAddress->getCountry()->getText()->name);
			$addressXml->appendChild($country);
			$countryCode = $dom->createElement('KodStatu', (string) $orderAddress->getCountry()->getId());
			$addressXml->appendChild($countryCode);
		}

		return $addressXml;
	}

	public function generateItemXml(DOMDocument $dom, OrderItem $orderItem): DOMElement
	{
		$itemXml = $dom->createElement('Polozka');

		$name  = (string) $orderItem->getOrderItemText('cs')->getName();
		$child = $dom->createElement('Popis', $name);
		$itemXml->appendChild($child);
		$child = $dom->createElement('PocetMJ', (string) $orderItem->getQuantity());
		$itemXml->appendChild($child);
		$child = $dom->createElement('Cena', (string) $orderItem->getPrice());
		$itemXml->appendChild($child);
		$vatRate = $orderItem->getVatRate() ? $orderItem->getVatRate() : 21; //get default vat rate from db?
		$child   = $dom->createElement('SazbaDPH', (string) $vatRate);
		$itemXml->appendChild($child);
		$child = $dom->createElement('TypCeny', "1"); //0=bez DPH, 1=s DPH, 2=jen DPH, 3=jen zaklad
		$itemXml->appendChild($child);
		$child = $dom->createElement('Sleva', "0");
		$itemXml->appendChild($child);
		//$child = $dom->createElement('CenovaHlad', "ZC"); //TODO cenove hladiny
		//$itemXml->appendChild($child);
		$child = $dom->createElement('CenaPoSleve', "1");
		$itemXml->appendChild($child);

		$product = $this->productsService->getProduct((int) $orderItem->getProductId());
		if ($product) {
			// pokud je to skladova zasoba (KmKarta) tak vyzaduje dva identifikatory pro sparovani - napr GUID + Katalog
			$child  = $dom->createElement('KmKarta');
			$child2 = $dom->createElement('GUID', (string) $product->getCode2());
			$child->appendChild($child2);
			$itemXml->appendChild($child);

			$child  = $dom->createElement('Sklad');
			$child2 = $dom->createElement('GUID', MoneyS3Config::load('ordersExport.wareHouseGUID'));
			$child->appendChild($child2);
			$itemXml->appendChild($child);
		} else {
			$child  = $dom->createElement('NesklPolozka');
			$child2 = $dom->createElement('Zkrat', (string) $orderItem->getOrderItemText('cs')->getName());
			$child->appendChild($child2);
			$child2 = $dom->createElement('UzivCode', (string) $orderItem->getOrderItemText('cs')->getName());
			$child->appendChild($child2);
			$itemXml->appendChild($child);
		}

		$child = $dom->createElement('Poznamka', '');
		$itemXml->appendChild($child);

		return $itemXml;
	}

	public function generateSpeditionItemXml(DOMDocument $dom, Order $order): DOMElement
	{
		$itemSpeditionXml = $dom->createElement('Polozka');

		$name  = $order->getSpedition()->getName() . ' - ' . $order->getPayment()->getName();
		$child = $dom->createElement('Popis', $name);
		$itemSpeditionXml->appendChild($child);
		$child = $dom->createElement('PocetMJ', "1");
		$itemSpeditionXml->appendChild($child);
		$child = $dom->createElement('ZbyvaMJ', "1");
		$itemSpeditionXml->appendChild($child);

		$child = $dom->createElement('Cena', (string) $order->getPaySpedPrice());
		$itemSpeditionXml->appendChild($child);
		$child = $dom->createElement('SazbaDPH', (string) MoneyS3Config::load('speditionVatRate', '21'));
		$itemSpeditionXml->appendChild($child);
		$child = $dom->createElement('TypCeny', "1"); //0=bez DPH, 1=s DPH, 2=jen DPH, 3=jen zaklad
		$itemSpeditionXml->appendChild($child);
		$child = $dom->createElement('Sleva', "0");
		$itemSpeditionXml->appendChild($child);
		$child = $dom->createElement('CenaPoSleve', "1");
		$itemSpeditionXml->appendChild($child);

		$child  = $dom->createElement('NesklPolozka');
		$child2 = $dom->createElement('TypZarDoby', "N"); //N=není,D=ve dnech,M=v měsících,R=v rocích,S=doživotní
		$child->appendChild($child2);
		$itemSpeditionXml->appendChild($child);

		return $itemSpeditionXml;
	}

	protected function getGroup(Order $order): string
	{
		return $order->getCustomer() && $order->getCustomer()->getGroupCustomers()
			? $order->getCustomer()->getGroupCustomers()->short
			: 'MOC';
	}

}
