<?php declare(strict_types = 1);

namespace MultihubDropShip\Console;

use Doctrine;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use Exception;
use MultihubDropShip\Model\Clients;
use MultihubDropShip\Model\Sync\SyncOrder;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Tracy\Debugger;

class DownloadOrders extends AbstractCommand
{
	/** @var SyncOrder|null @inject */
	public ?SyncOrder $syncOrders = null;

	/** @var Clients|null @inject */
	public ?Clients $clients = null;

	protected function configure(): void
	{
		$this->setName('multihub-drop-ship:download-orders')
			->addOption('clientId', null, InputOption::VALUE_REQUIRED)
			->setDescription('Download new orders');
	}

	protected function executeSchemaCommand(InputInterface $input, OutputInterface $output, Doctrine\ORM\Tools\SchemaTool $schemaTool, array $metadatas, SymfonyStyle $ui): int
	{
		$output->writeln('<fg=white;bg=#007bff>Multihub - Download orders</>');
		try {
			if (!$this->clients || !$this->syncOrders) {
				throw new Exception('Service not found');
			}

			$client = $this->clients->getClient((int) $input->getOption('clientId'));
			if ($client) {
				$this->syncOrders->downloadNew($client);
				$output->writeLn('<info>[OK] - orders downloaded</info>');

				try {
					$this->syncOrders->downloadCancelled($client);
				} catch (Exception $e) {
					Debugger::log($e, 'setcancelled');
				}
			} else {
				throw new Exception('Client not found');
			}
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}
}
