<?php declare(strict_types = 1);

namespace MultihubDropShip\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use Doctrine\ORM\Query\Expr\Join;
use MultihubDropShip\Model\Dao\Client;

class Clients
{
	/** @var array<int, Client>|null */
	protected ?array $cClients = null;

	protected EntityManagerDecorator $em;
	protected Sites                  $sites;

	public function __construct(
		EntityManagerDecorator $em,
		Sites                  $sites
	)
	{
		$this->em    = $em;
		$this->sites = $sites;
	}

	protected function loadClients(): void
	{
		if ($this->cClients === null) {
			$this->cClients = [];

			foreach ($this->em->getRepository(\MultihubDropShip\Model\Entities\Client::class)
				         ->createQueryBuilder('c')
				         ->select('c.id, c.name, c.apiKey, c.secretToken, c.isTest, c.sendAll, c.addCanAddToCartProds, GROUP_CONCAT(IDENTITY(ac.category)) as allowedCategories, c.imageBaseUrl, IDENTITY(c.seller) as seller, IDENTITY(c.sellerKaufland) as sellerKaufland, c.siteIdent, c.lang')
				         ->leftJoin('c.allowedCategories', 'ac', Join::WITH, 'c.id = ac.client')
				         ->where('c.isActive = 1')
				         ->groupBy('c.id')
				         ->getQuery()->getArrayResult() as $row) {
				if (!$row['id']) {
					continue;
				}

				$isTest = (bool) $row['isTest'];

				$client = new Client(
					(int) $row['id'],
					(string) $row['name'],
					(string) $row['apiKey'],
					$row['secretToken'],
					(string) $row['imageBaseUrl'],
					MultihubDropShipConfig::load($isTest ? 'testEndPoint' : 'productionEndPoint'),
				);

				$client->sendAll              = (bool) $row['sendAll'];
				$client->updateProductV       = MultihubDropShipConfig::load('updateProductV');
				$client->updateCategoryV      = MultihubDropShipConfig::load('updateCategoryV');
				$client->allowedCategoriesIds = $row['allowedCategories']
					? array_map('intval', explode(',', (string) $row['allowedCategories']))
					: [];
				$client->isTest               = $isTest;
				$client->addCanAddToCartProds = (bool) $row['addCanAddToCartProds'];

				if ($row['lang']) {
					$client->lang = $row['lang'];
				}

				if ($row['siteIdent']) {
					$client->siteIdent = $row['siteIdent'];
				} else {
					$client->siteIdent = $this->sites->getCurrentSite()->getIdent();
				}

				if ($row['seller']) {
					$client->sellerId = $row['seller'];
				}

				if ($row['sellerKaufland']) {
					$client->sellerIdKaufland = $row['sellerKaufland'];
				}

				$this->cClients[$client->getId()] = $client;
			}
		}
	}

	public function getClients(): array
	{
		if ($this->cClients === null) {
			$this->loadClients();
		}

		return $this->cClients ?? [];
	}

	public function getClient(?int $id): ?Client { return $this->getClients()[$id] ?? null; }
}
