<?php declare(strict_types = 1);

namespace MultihubDropShip\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Strings;
use Core\Model\Images\ImagePipe;
use Core\Model\Lang\Langs;
use Currency\Model\Config;
use Currency\Model\Currencies;
use Currency\Model\Exchange;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Entities\CategoryTexts;
use EshopCatalog\Model\Entities\DynamicFeatureProduct;
use EshopCatalog\Model\Entities\Feature;
use EshopCatalog\Model\Entities\FeatureCategory;
use EshopCatalog\Model\Entities\FeatureProduct;
use EshopCatalog\Model\Entities\FeatureTexts;
use EshopCatalog\Model\Entities\FeatureValueTexts;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopCatalog\Model\Navigation\Home;
use Gallery\FrontModule\Model\Dao\Image;
use MultihubDropShip\Model\Dao\Client;
use MultihubDropShip\Model\Dao\Product;
use MultihubDropShip\Model\Dao\ProductCategory;
use MultihubDropShip\Model\Dao\ProductCategoryImageList;
use MultihubDropShip\Model\Dao\ProductCategoryText;
use MultihubDropShip\Model\Dao\ProductFeature;
use MultihubDropShip\Model\Dao\ProductFeatureText;
use MultihubDropShip\Model\Dao\ProductImage;
use MultihubDropShip\Model\Dao\ProductImagesList;
use MultihubDropShip\Model\Dao\ProductPrice;
use MultihubDropShip\Model\Dao\ProductPriceGroup;
use MultihubDropShip\Model\Dao\ProductText;
use Navigations\FrontModule\Model\LinksHelper;
use Navigations\Model\Navigations;
use Tracy\Debugger;

class ProductBuilder
{
	protected EntityManagerDecorator $em;
	protected ProductsFacade         $productsFacade;
	protected Langs                  $langs;
	protected Currencies             $currencies;
	protected EventDispatcher        $eventDispatcher;
	protected Exchange               $exchange;
	protected LinksHelper            $linksHelper;
	protected Navigations            $navigations;
	protected Categories             $categories;
	protected ImagePipe              $imagePipe;

	public function __construct(
		EntityManagerDecorator $em,
		ProductsFacade         $productsFacade,
		Langs                  $langs,
		Currencies             $currencies,
		EventDispatcher        $eventDispatcher,
		Exchange               $exchange,
		LinksHelper            $linksHelper,
		Navigations            $navigations,
		Categories             $categories,
		ImagePipe              $imagePipe
	)
	{
		$this->em              = $em;
		$this->productsFacade  = $productsFacade;
		$this->langs           = $langs;
		$this->currencies      = $currencies;
		$this->eventDispatcher = $eventDispatcher;
		$this->exchange        = $exchange;
		$this->linksHelper     = $linksHelper;
		$this->navigations     = $navigations;
		$this->categories      = $categories;
		$this->imagePipe       = $imagePipe;
	}

	public function buildProducts(Client $client, array $ids): array
	{
		/** @var Product[] $result */
		$result = [];
		$this->productsFacade->clearTemp();

		$usedCurrencies = [];
		$defaultPrices  = [];

		$availabilities          = [];
		$productsPrices          = [];
		$changeCountryByCurrency = Config::load('changeCountry');

		foreach (array_chunk($ids, 600) as $chunk) {
			foreach ($this->productsFacade->productsService->getEr()->createQueryBuilder('p')
				         ->select('p.id, p.quantity')
				         ->where('p.id IN (' . implode(',', $chunk) . ')')
				         ->getQuery()->getArrayResult() as $row) {
				$availabilities[$row['id']] = $row;
			}
		}

		$event = new Event(['products' => &$availabilities]);
		$this->eventDispatcher->dispatch($event, __CLASS__ . '::beforeSendAvailabilitiesProcess');

		// Zaklad z productsFacade dao
		ProductQuery::$ignoreProductPublish = true;
		ProductsFacade::$ignoreIsActive     = true;

		$productNav = array_values($this->navigations->getPublishedByComponent('eshopCatalog.navigation.home'))[0] ?? null;

		$catRootId = $this->categories->getRootIdForSite($client->siteIdent);

		$categoryIds               = [];
		$productLinks              = [];
		$productsDefaultCategories = [];
		$productsInCategories      = [];

		foreach (array_chunk($ids, 500) as $chunk) {
			foreach ($this->productsFacade->getProducts($chunk) as $eshopProduct) {
				if (!$eshopProduct->gallery || !$eshopProduct->defaultCategory) {
					continue;
				}

				$productsDefaultCategories[$eshopProduct->id] = $eshopProduct->defaultCategory->id;

				$categoryParentPath = $eshopProduct->defaultCategory->getParentPathStringFlipped();
				$categoryName       = (string) $eshopProduct->defaultCategory->name;

				if ($categoryParentPath) {
					$categoryName = $categoryParentPath . ' > ' . $categoryName;
				}

				$categoryId = $eshopProduct->defaultCategory->getId();

				$productsInCategories[$categoryId][] = $eshopProduct->getId();

				$productCategory = new ProductCategory(
					(string) $categoryId,
					$categoryName,
				);

				if ($eshopProduct->defaultCategory->lvl !== null) {
					$productCategory->level = $eshopProduct->defaultCategory->lvl;
				}

				if ($eshopProduct->defaultCategory->image) {
					$productCategory->image = str_replace(' ', '%20', $client->imageBaseUrl . $eshopProduct->defaultCategory->image);

					$small  = explode('/thumbs', $this->imagePipe->request($eshopProduct->defaultCategory->image, '100x100', 'fit', false, false, 'webp'), 2)[1];
					$medium = explode('/thumbs', $this->imagePipe->request($eshopProduct->defaultCategory->image, '470x266', 'fit', false, false, 'webp'), 2)[1];

					$productCategory->imagesLists[] = new ProductCategoryImageList('small', str_replace(' ', '%20', $client->imageBaseUrl . '/thumbs' . $small));
					$productCategory->imagesLists[] = new ProductCategoryImageList('medium', str_replace(' ', '%20', $client->imageBaseUrl . '/thumbs' . $medium));
				}

				$categoryIds[$categoryId] = $categoryId;

				$productAvailability = $availabilities[$eshopProduct->id];
				$quantity            = (int) $productAvailability['quantity'];

				if (
					$quantity <= 0 && (
						$productAvailability['unlimitedQuantity']
						|| ($eshopProduct->availability && $eshopProduct->availability->canAddToCart() && $client->addCanAddToCartProds)
					)
				) {
					$quantity = 50;
				}

				$product = new Product(
					(string) $eshopProduct->id,
					$quantity,
					$eshopProduct->isActive && $eshopProduct->canAddToCart ? 'active' : 'disabled',
					$eshopProduct->isOversize ? 'big' : 'small',
					$productCategory
				);

				$product->categoryHierarchy = array_merge([$categoryId], array_keys($eshopProduct->defaultCategory->getParentPath()));

				$product->otherCategoriesHierarchy = [];
				foreach ($eshopProduct->categories as $v) {
					$vCat = $this->categories->get($v);
					if ($vCat) {
						$product->otherCategoriesHierarchy[] = array_merge([$v], array_keys($vCat->getParentPath()));
					}
				}

				if ($productNav) {
					foreach ($this->linksHelper->getOtherLangLinks($productNav, [
						'id'     => $eshopProduct->id,
						'action' => Home::ACTION_PRODUCT,
					], Home::DEFAULT_PRESENTER) as $linkLang => $link) {
						$productLinks[$eshopProduct->id][$linkLang] = $link['link'];
					}
				}

				$product->code1     = $eshopProduct->code1;
				$product->code2     = $eshopProduct->code2;
				$product->ean       = $eshopProduct->getEan();
				$product->width     = $eshopProduct->width;
				$product->height    = $eshopProduct->height;
				$product->depth     = $eshopProduct->depth;
				$product->weight    = $eshopProduct->weight;
				$product->condition = $eshopProduct->condition;

				if ($eshopProduct->variantOf) {
					$product->groupId            = (string) $eshopProduct->variantId;
					$product->variantGroupName   = $eshopProduct->getVariantName();
					$product->variantFeatureDiff = $eshopProduct->getVariantDiffFeatures();

					if ($eshopProduct->variantImage) {
						$product->variantImage = new ProductImage(str_replace(' ', '%20', $client->imageBaseUrl . $eshopProduct->variantImage->getFilePath()), 0);
					}
				}

				if ($eshopProduct->getManufacturer()) {
					$product->brand = $eshopProduct->getManufacturer()->name;
				}

				$hasCover    = false;
				$cover       = null;
				$eshopImages = [];
				foreach ($eshopProduct->gallery->getImages() as $eshopImage) {
					$position = $eshopImage->position;

					if ($eshopImage->isCover) {
						$cover    = $eshopImage;
						$position = -100;
					}

					$eshopImages[$position] = $eshopImage;
				}
				ksort($eshopImages);

				if ($cover) {
					$small  = explode('/thumbs', $this->imagePipe->request($cover->getFilePath(), '100x100', 'fit', false, false, 'webp'), 2)[1];
					$medium = explode('/thumbs', $this->imagePipe->request($cover->getFilePath(), '470x266', 'fit', false, false, 'webp'), 2)[1];

					$smallList           = new ProductImagesList('small');
					$smallList->images[] = new ProductImage(str_replace(' ', '%20', $client->imageBaseUrl . '/thumbs' . $small), 0);

					$product->imagesLists[] = $smallList;

					$mediumList           = new ProductImagesList('medium');
					$mediumList->images[] = new ProductImage(str_replace(' ', '%20', $client->imageBaseUrl . '/thumbs' . $medium), 0);

					$product->imagesLists[] = $mediumList;
				}

				foreach (array_values($eshopImages) as $imagePosition => $eshopImage) {
					/** @var Image $eshopImage */
					if (!$eshopImage->getFilePath()) {
						continue;
					}

					$image         = new ProductImage(str_replace(' ', '%20', $client->imageBaseUrl . $eshopImage->getFilePath()), $imagePosition + 1);
					$image->isMain = (bool) $eshopImage->isCover;

					if ($eshopImage->isCover) {
						$hasCover = true;
					}

					$product->images[] = $image;
				}

				if (!$hasCover && isset($product->images[0])) {
					$product->images[0]->isMain = true;
				}

				$result[$product->remoteId] = $product;
			}
		}

		if (empty($result)) {
			return [];
		}

		// Category texts
		$categoryTexts = [];
		foreach (array_chunk($categoryIds, 250) as $chunk) {
			foreach ($this->em->getRepository(CategoryTexts::class)
				         ->createQueryBuilder('ct')
				         ->select('IDENTITY(ct.id) as category, ct.lang, ct.name, ct.nameH1, ct.shortDescription, ct.description, ct.image')
				         ->where('ct.id IN (:ids)')
				         ->setParameters([
					         'ids' => $chunk,
				         ])->getQuery()->getArrayResult() as $row) {
				$cat = $this->categories->getCategories($catRootId, $row['lang'])[$row['category']] ?? null;

				$text                   = new ProductCategoryText($row['lang'], (string) $row['name']);
				$text->url              = $cat->link ?? null;
				$text->nameH1           = $row['nameH1'];
				$text->shortDescription = $row['shortDescription'];
				$text->description      = $row['description'];
				$text->image            = str_replace(' ', '%20', $client->imageBaseUrl . $row['image']);

				$categoryTexts[$row['category']][$row['lang']] = $text;
			}
		}

		foreach (array_chunk(array_intersect($ids, array_keys($result)), 250) as $chunk) {
			foreach ($chunk as $id) {
				$result[$id]->category->texts = array_values($categoryTexts[$result[$id]->category->remoteId]);
			}

			// Texts
			foreach ($this->em->getRepository(ProductTexts::class)
				         ->createQueryBuilder('pt')
				         ->select('IDENTITY(pt.id) as product, pt.lang, pt.name, pt.shortDescription, pt.description')
				         ->where('pt.id IN (:ids)')
				         ->setParameters([
					         'ids' => $chunk,
				         ])->getQuery()->getArrayResult() as $row) {
				if (!$row['name']) {
					continue;
				}

				$productText                   = new ProductText($row['lang'], $row['name']);
				$productText->shortDescription = $row['shortDescription'];
				$productText->description      = $row['description'];
				$productText->url              = $productLinks[$row['product']][$row['lang']] ?? null;

				$tmp = $this->categories->getCategories($catRootId, $row['lang'])[$productsDefaultCategories[$row['product']]] ?? null;
				if ($tmp) {
					$productText->gpsrWarningText = $tmp->getSafetyWarningText();
				}

				$result[$row['product']]->texts[] = $productText;
			}

			// Features
			$featureProducts         = [];
			$featuresIds             = [];
			$featuresValuesIds       = [];
			$featuresTexts           = [];
			$featuresValuesTexts     = [];
			$dynamicFeaturesProducts = [];
			$dynamicFeaturesIds      = [];
			$dynamicFeaturesUnits    = [];

			foreach ($this->em->getRepository(FeatureCategory::class)
				         ->createQueryBuilder('fc')
				         ->select('IDENTITY(fc.category) as category, IDENTITY(fc.featureValue) as featureValueId, IDENTITY(fv.feature) as featureId')
				         ->innerJoin('fc.featureValue', 'fv')
				         ->where('fc.category IN (:cats)')
				         ->setParameters([
					         'cats' => array_keys($categoryIds),
				         ])->getQuery()->getArrayResult() as $row) {
				foreach ($productsInCategories[$row['category']] ?? [] as $prodId) {
					$featureProducts[$prodId][$row['featureValueId']] = [
						'featureId' => $row['featureId'],
						'valueId'   => $row['featureValueId'],
					];
				}

				$featuresValuesIds[$row['featureValueId']] = $row['featureValueId'];
				$featuresIds[$row['featureId']]            = $row['featureId'];
			}

			foreach ($this->em->getRepository(FeatureProduct::class)
				         ->createQueryBuilder('fp')
				         ->select('IDENTITY(fp.product) as product, IDENTITY(fp.featureValue) as featureValueId, IDENTITY(fv.feature) as featureId')
				         ->innerJoin('fp.featureValue', 'fv')
				         ->where('fp.product IN (:ids)')
				         ->setParameters([
					         'ids' => $chunk,
				         ])->getQuery()->getArrayResult() as $row) {
				$featureProducts[$row['product']][$row['featureValueId']] = [
					'featureId' => $row['featureId'],
					'valueId'   => $row['featureValueId'],
				];
				$featuresValuesIds[$row['featureValueId']]                = $row['featureValueId'];
				$featuresIds[$row['featureId']]                           = $row['featureId'];
			}

			foreach ($this->em->getRepository(DynamicFeatureProduct::class)
				         ->createQueryBuilder('dfp')
				         ->select('IDENTITY(dfp.product) as product, IDENTITY(dfp.feature) as featureId, dfp.value')
				         ->where('dfp.product IN (:ids)')
				         ->setParameters([
					         'ids' => $chunk,
				         ])->getQuery()->getArrayResult() as $row) {
				$dynamicFeaturesProducts[$row['product']][] = [
					'featureId' => $row['featureId'],
					'value'     => $row['value'],
				];
				$featuresIds[$row['featureId']]             = $row['featureId'];
				$dynamicFeaturesIds[$row['featureId']]      = $row['featureId'];
			}

			if (!empty($dynamicFeaturesIds)) {
				foreach ($this->em->getRepository(Feature::class)
					         ->createQueryBuilder('f')
					         ->select('f.id, f.unit')
					         ->where('f.id IN (:ids)')
					         ->setParameters([
						         'ids' => $dynamicFeaturesIds,
					         ])->getQuery()->getArrayResult() as $row) {
					$dynamicFeaturesUnits[$row['id']] = $row['unit'];
				}
			}

			if (!empty($featureProducts)) {
				foreach (array_chunk($featuresIds, 250) as $featuresIdsChunk) {
					foreach ($this->em->getRepository(FeatureTexts::class)
						         ->createQueryBuilder('ft')
						         ->select('IDENTITY(ft.id) as id, ft.lang, ft.name')
						         ->where('ft.id IN (:ids)')
						         ->setParameters([
							         'ids' => $featuresIdsChunk,
						         ])->getQuery()->getArrayResult() as $row) {
						$featuresTexts[$row['id']][(string) $row['lang']] = $row['name'];
					}
				}

				foreach (array_chunk($featuresValuesIds, 250) as $featuresValuesIdsChunk) {
					foreach ($this->em->getRepository(FeatureValueTexts::class)
						         ->createQueryBuilder('fvt')
						         ->select('IDENTITY(fvt.id) as id, fvt.lang, fvt.name')
						         ->where('fvt.id IN (:ids)')
						         ->setParameters([
							         'ids' => $featuresValuesIdsChunk,
						         ])->getQuery()->getArrayResult() as $row) {
						$featuresValuesTexts[$row['id']][$row['lang']] = $row['name'];
					}
				}

				foreach ($featureProducts as $productId => $rows) {
					foreach ($rows as $row) {
						$productFeature = new ProductFeature((string) $row['featureId'], (string) $row['valueId']);

						foreach ($featuresTexts[$row['featureId']] as $lang => $featureName) {
							$featureValue = $featuresValuesTexts[$row['valueId']][$lang] ?? null;

							if (!$featureValue) {
								continue;
							}

							$productFeatureText = new ProductFeatureText($lang, $featureName, $featureValue);

							$productFeature->texts[] = $productFeatureText;
						}

						$result[$productId]->features[] = $productFeature;
					}
				}

				foreach ($dynamicFeaturesProducts as $productId => $rows) {
					foreach ($rows as $row) {
						$productFeature = new ProductFeature('d_' . $row['featureId'], Strings::webalize((string) $row['value']));

						foreach ($featuresTexts[$row['featureId']] as $lang => $featureName) {
							$productFeatureText = new ProductFeatureText(
								$lang,
								$featureName,
								trim($row['value'] . ' ' . ($dynamicFeaturesUnits[$row['featureId']] ?: ''))
							);

							$productFeature->texts[] = $productFeatureText;
						}

						$result[$productId]->features[] = $productFeature;
					}
				}
			}

			// Ceny
			$defaultCurrency = $this->currencies->getDefaultCode();
			$baseVats        = [];
			$countriesVats   = [];

			// Zakladni
			foreach ($this->em->getRepository(\EshopCatalog\Model\Entities\Product::class)
				         ->createQueryBuilder('p')
				         ->select('p.id, p.price, pVr.rate as vatRate')
				         ->leftJoin('p.vatRate', 'pVr')
				         ->where('p.id IN (:ids)')
				         ->setParameters([
					         'ids' => $chunk,
				         ])->getQuery()->getArrayResult() as $row) {
				$productPrice = new ProductPrice(round((float) $row['price'], 2), Strings::upper($defaultCurrency), 'CZ');

				$productPrice->vatRate = (int) $row['vatRate'] ?: 21;

				$baseVats[$row['id']]                                      = (int) $row['vatRate'] ?: 21;
				$result[$row['id']]->prices[]                              = $productPrice;
				$defaultPrices[$row['id']]                                 = round((float) $row['price'], 2);
				$usedCurrencies[$row['id']]['CZ'][$productPrice->currency] = $productPrice->currency;
			}

			// Podle zeme
			$pricesToConvert = [];
			foreach ($this->em->getRepository(\EshopCatalog\Model\Entities\ProductPrice::class)
				         ->createQueryBuilder('pp')
				         ->select('IDENTITY(pp.product) as product, IDENTITY(pp.country) as country, pp.currency, pp.price, ppVr.rate as vatRate')
				         ->leftJoin('pp.vatRate', 'ppVr')
				         ->where('pp.product IN (:ids)')
				         ->setParameters([
					         'ids' => $chunk,
				         ])->getQuery()->getArrayResult() as $row) {
				if (!$row['vatRate']) {
					continue;
				}

				$productId = (int) $row['product'];
				$vatRate   = (int) $row['vatRate'];
				$country   = Strings::upper((string) $row['country']) ?: 'CZ';

				$curr = Strings::upper($row['currency'] ?: $defaultCurrency);

				if ($row['price'] && $row['price'] > 0) {
					$price = (float) $row['price'];
				} else {
					$price = $this->exchange->changeBySite('allegro', $defaultPrices[$productId], $curr, $defaultCurrency);
				}

				$productPrice = new ProductPrice(round($price, 2), $curr, $country);

				$productPrice->vatRate = $vatRate;

				$result[$productId]->prices[]                                  = $productPrice;
				$usedCurrencies[$productId][$country][$productPrice->currency] = $productPrice->currency;

				$countriesVats[$productId][$country] = $vatRate;

				$productsPrices[$productId][$country . '_' . $curr] = [
					'product'  => $productId,
					'price'    => $price,
					'currency' => $curr,
					'country'  => $country,
				];
			}

			foreach ($chunk as $prodId) {
				try {
					foreach (MultihubDropShipConfig::load('additionalPrices')[$client->siteIdent] ?? [] as $country => $row) {
						$vatRate           = $countriesVats[$row['product']][$country] ?? (int) $row['defaultVat'] ?: 21;
						$currency          = (string) $row['currency'];
						$currencySiteIdent = $row['currencySiteIdent'] ?? $client->siteIdent;

						if (isset($usedCurrencies[$prodId][$country][$currency])) {
							continue;
						}

						$price                  = null;
						$preloadedPricesCountry = Strings::upper($changeCountryByCurrency[$currency] ?: $country);
						$preloadedTmp           = $productsPrices[$prodId][$preloadedPricesCountry . '_' . $currency] ?? null;
						if ($preloadedTmp) {
							if ($preloadedTmp['price']) {
								$price = $preloadedTmp['price'];
							}
						}

						if (!$price) {
							$price = $this->exchange->changeBySite($currencySiteIdent, $defaultPrices[$prodId], $currency, $defaultCurrency);
						}

						$productPrice          = new ProductPrice(round($price, 2), $currency, $country);
						$productPrice->vatRate = $vatRate;

						$result[$prodId]->prices[]                                  = $productPrice;
						$usedCurrencies[$prodId][$country][$productPrice->currency] = $productPrice->currency;

						$countriesVats[$prodId][$country] = $vatRate;
					}
				} catch (\Exception $e) {
					Debugger::log($e, Debugger::ERROR);
				}
			}

			$dCurrency = Strings::upper($defaultCurrency);

			// Skupiny zakazniku
			foreach ($this->em->getRepository(\EshopCatalog\Model\Entities\ProductPriceLevel::class)
				         ->createQueryBuilder('ppl')
				         ->select('IDENTITY(ppl.productId) as product, IDENTITY(ppl.groupId) as groupId, ppl.price')
				         ->where('ppl.productId IN (:ids)')
				         ->andWhere('ppl.price > 0')
				         ->setParameters([
					         'ids' => $chunk,
				         ])->getQuery()->getArrayResult() as $row) {
				$groupId   = (int) $row['groupId'];
				$productId = (int) $row['product'];
				$price     = (float) $row['price'];

				$productPrice          = new ProductPriceGroup($price, $dCurrency, 'CZ', $groupId);
				$productPrice->vatRate = $baseVats[$productId];

				$result[$productId]->priceLevels[$groupId . '_' . $dCurrency . '_CZ'] = $productPrice;
				$defaultLevelPrices[$productId][$groupId]                             = round($price, 2);
			}

			foreach ($this->em->getRepository(\EshopCatalog\Model\Entities\ProductPriceLevelCountry::class)
				         ->createQueryBuilder('pplc')
				         ->select('IDENTITY(pplc.product) as product, IDENTITY(pplc.group) as groupId, pplc.price, IDENTITY(pplc.country) as country, pplc.currency')
				         ->where('pplc.product IN (:ids)')
				         ->andWhere('pplc.price > 0')
				         ->setParameters([
					         'ids' => $chunk,
				         ])->getQuery()->getArrayResult() as $row) {
				$country   = Strings::upper($row['country']);
				$groupId   = (int) $row['groupId'];
				$productId = (int) $row['product'];
				$currency  = Strings::upper($row['currency']);
				$price     = (float) $row['price'];

				$key = $groupId . '_' . $currency . '_' . $country;

				if (isset($result[$productId]->priceLevels[$key])) {
					$result[$productId]->priceLevels[$key]->price = $price;
				} else {
					$productPrice = new ProductPriceGroup(round($price, 2), $currency, $country, $groupId);

					$productPrice->vatRate = $countriesVats[$productId][$country];

					$result[$productId]->priceLevels[$key] = $productPrice;
				}
			}

			foreach ($chunk as $prodId) {
				foreach (MultihubDropShipConfig::load('additionalPrices')[$client->siteIdent] ?? [] as $country => $row) {
					$currencySiteIdent = $row['currencySiteIdent'] ?? $client->siteIdent;

					foreach ($usedCurrencies[$prodId][$country] ?? [] as $currency) {

						foreach ($defaultLevelPrices[$prodId] ?? [] as $groupId => $defaultPrice) {
							$key = $groupId . '_' . $currency . '_' . $country;
							if (isset($result[$prodId]->priceLevels[$key])) {
								continue;
							}

							$price = $this->exchange->changeBySite($currencySiteIdent, $defaultPrice, $currency, $defaultCurrency);

							$productPrice          = new ProductPriceGroup(round($price, 2), $currency, $country, $groupId);
							$productPrice->vatRate = $countriesVats[$prodId][$country] ?? 21;

							$result[$prodId]->priceLevels[$key] = $productPrice;
						}
					}
				}
			}

			$this->em->clear();
		}

		foreach ($result as $k => $v) {
			$v->priceLevels = array_values($v->priceLevels);
		}

		//		$event = new ProductBuilderEvent($client, $result, $eshopProducts);
		//		$this->eventDispatcher->dispatch($event, self::class . '::buildProducts');

		return $result;
	}
}
