<?php declare(strict_types = 1);

namespace MultihubDropShip\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use EshopOrders\AdminModule\Model\Statuses;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Event\OrderPackageNumbers;
use MultihubDropShip\Model\Api\OrderApi;
use MultihubDropShip\Model\Clients;
use MultihubDropShip\Model\Entities\DropShipOrder;
use MultihubDropShip\Model\MultihubDropShipLogger;
use MultihubDropShip\Model\OrdersExported;
use MultihubDropShip\Model\Sync\SyncOrder;
use MultihubDropShip\Model\Sync\SyncShipment;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected OrdersExported         $ordersExported;
	protected SyncOrder              $syncOrder;
	protected Clients                $clients;
	protected Statuses               $statuses;
	protected OrderApi               $orderApi;
	protected SyncShipment           $syncShipment;

	public function __construct(
		EntityManagerDecorator $em,
		OrdersExported         $ordersExported,
		SyncOrder              $syncOrder,
		Clients                $clients,
		Statuses               $statuses,
		OrderApi               $orderApi,
		SyncShipment           $syncShipment
	)
	{
		$this->em             = $em;
		$this->ordersExported = $ordersExported;
		$this->syncOrder      = $syncOrder;
		$this->clients        = $clients;
		$this->statuses       = $statuses;
		$this->orderApi       = $orderApi;
		$this->syncShipment   = $syncShipment;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderOnSuccess'             => ['orderOnSuccess', 80],
			'eshopOrders.admin.orderOnSuccess'       => 'orderOnSuccess',
			'eshopOrders.orders.checkCompleted'      => 'checkCompleted',
			'eshopOrders.packageNumberAssigned'      => 'packageNumberAssigned',
			'MultihubDropShip.orders.checkCompleted' => 'checkCompleted',
		];
	}

	public function orderOnSuccess(OrderEvent $event): void
	{
		$order = $event->order;

		if (!$order) {
			$order = $event->formData['fromOrder'] ?? null;
		}

		if (!$order) {
			return;
		}

		foreach ($order->getOrderItems() as $orderItem) {
			if (!$orderItem->getProduct()) {
				continue;
			}

			ProductSubscriber::$update[] = $orderItem->getProductId();
		}
	}

	public function checkCompleted(Event $event): void
	{
		$orders = $this->ordersExported->getOrdersNotCompleted();

		foreach ($orders as $id => $dropShipOrder) {
			try {
				if (!$dropShipOrder->client->getId()) {
					continue;
				}

				$client = $this->clients->getClient($dropShipOrder->client->getId());
				if (!$client) {
					continue;
				}

				$detail = $this->syncOrder->getDetail($client, $dropShipOrder->dropShipId);
				if (!$detail) {
					throw new \Exception('Check completed - Detail ' . $id . ' not found - ' . $client->getName() . ' - dropship id: ' . $dropShipOrder->dropShipId);
				}

				if ($detail['status'] === DropShipOrder::StatusCancelled) {
					$this->statuses->changeStatus([$dropShipOrder->order->getId()], OrderStatus::STATUS_CANCELED);
				} else if ($detail['status'] === DropShipOrder::StatusCompleted) {
					$this->statuses->changeStatus([$dropShipOrder->order->getId()], OrderStatus::STATUS_FINISHED);
				}
			} catch (\Exception $e) {
				MultihubDropShipLogger::log($e);
				LogNotifier::toDevelopers($e->getMessage(), 'Multihub DropShip');
			}
		}
	}

	public function packageNumberAssigned(OrderPackageNumbers $event): void
	{
		if (!$event->packageNumbers) {
			return;
		}

		/** @var DropShipOrder|null $order */
		$order = $this->em->getRepository(DropShipOrder::class)->findOneBy(['order' => $event->order->getId()]);

		if ($order) {
			$client = $this->clients->getClient($order->client->getId());
			if (!$client) {
				return;
			}

			$r = $this->orderApi->addTrackingCode($client, $order->dropShipId, $event->packageNumbers[0]);
			if ($r) {
				$this->syncShipment->setShipped($client, $order->dropShipId);
				$order->setExported();
				$order->trackingNumber = $event->packageNumbers[0];
				$order->trackingUrl    = $event->trackingUrls[0] ?? '';

				$this->em->persist($order);
				$this->em->flush();
			}
		}
	}
}
