<?php declare(strict_types = 1);

namespace MultihubDropShip\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\Event;
use Core\Model\Event\GridFilterEvent;
use Core\Model\FileRequests;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use Doctrine\Common\Collections\Criteria;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Orders;
use MultihubDropShip\Model\Clients;
use MultihubDropShip\Model\Entities\DropShipOrder;
use MultihubDropShip\Model\OrdersExported;
use MultihubDropShip\Model\Sync\SyncShipment;
use Nette\Application\LinkGenerator;
use Nette\Localization\Translator;
use Nette\Utils\FileSystem;
use Nette\Utils\Html;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderSubscriber implements EventSubscriberInterface
{
	/** @var DropShipOrder[]|null */
	protected static ?array $allOrders   = null;
	protected static ?array $allPackages = null;

	public function __construct(
		protected EntityManagerDecorator $em,
		protected OrdersExported         $ordersExported,
		protected Translator             $translator,
		protected LinkGenerator          $linkGenerator,
		protected FileRequests           $fileRequests,
		protected Orders                 $orders,
		protected SyncShipment           $syncShipment,
		protected Clients                $clients,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			OrderForm::class . '::onAttach'               => 'orderFormAttached',
			OrdersGrid::class . '::columnIdRender'        => 'columnIdRender',
			OrdersGrid::class . '::columnSpeditionRender' => 'columnSpeditionRender',
			OrdersGrid::class . '::filterPackageNumber'   => 'gridFilterPackageNumber',
			OrdersGrid::class . '::filterId'              => 'gridFilterId',
			OrdersGrid::class . '::onAttach'              => ['ordersGridAttached', 255],
		];
	}

	public function orderFormAttached(ControlEvent $event): void
	{
		/** @var OrderForm $control */
		$control = $event->control;
		$order   = $control->order;

		if (!$order || !$order->getId()) {
			return;
		}

		$spedition = $order->getSpedition() ? $order->getSpedition()->getSpedition() : null;

		if ($spedition && $spedition->getIdent() === 'allegro') {
			$dropShipOrder = $this->ordersExported->getByIds([$order->getId()])[$order->getId()] ?? null;
			if (!$dropShipOrder) {
				return;
			}

			$trackingUrl = [
				$dropShipOrder->trackingUrl,
			];

			$control->template->dropShipExport              = $dropShipOrder;
			$control->template->trackingUrls                = $trackingUrl;
			$control->template->speditionExportInfoTemplate = __DIR__ . '/exportInfoTemplate.latte';
		}
	}

	public function columnSpeditionRender(Event $event): void
	{
		$data           = $event->data;
		$speditionIdent = 'allegro';

		/** @var Order $order */
		$order = $data['order'];

		/** @var Html $html */
		$html = $data['html'];

		$dropShipOrder = null;
		if ($data['ordersBySpedition'][$speditionIdent] ?? null) {
			$dropShipOrder = $this->getOrders(array_keys($data['ordersBySpedition'][$speditionIdent] ?? []))[$order->getId()] ?? null;
		}

		if ($dropShipOrder) {
			$html->setText($html->getText());

			$html->addHtml(Html::el('div')
				->addHtml(Html::el('a', ['target' => '_blank'])
					->setAttribute('href', $dropShipOrder->trackingUrl)
					->setText((string) $dropShipOrder->trackingNumber)
				)
			);
		}
	}

	public function columnIdRender(Event $event): void
	{
		$data           = $event->data;
		$speditionIdent = 'allegro';
		$export         = null;

		/** @var Order $order */
		$order = $data['order'];

		/** @var Html $html */
		$html = $data['html'];

		if ($data['ordersBySpedition'][$speditionIdent]) {
			$export = $this->getOrders(array_keys($data['ordersBySpedition'][$speditionIdent] ?? []))[$order->getId()] ?? null;
		} else if ($order->getSpeditionIdent() === $speditionIdent) {
			$export = $this->ordersExported->getByIds([$order->getId()])[$order->getId()];
		}

		/** @var DropShipOrder|null $export */
		if ($export) {
			$html->addHtml(Html::el('div')
				->setText(Strings::truncate($export->dropShipId, 8)));
		}
	}

	public function gridFilterPackageNumber(GridFilterEvent $event): void
	{
		/** @var string $value */
		$value = $event->value;
		$ids   = $this->ordersExported->findIdByPackageNumber((string) $value);

		if ($ids !== []) {
			$event->criteria->orWhere(Criteria::expr()->in('o.id', $ids));
		}
	}

	public function gridFilterId(GridFilterEvent $event): void
	{
		/** @var string $value */
		$value = $event->value;
		$ids   = $this->ordersExported->findIdByDropShipId((string) $value);

		if ($ids !== []) {
			$event->criteria->orWhere(Criteria::expr()->in('o.id', $ids));
		}
	}

	/**
	 * @return DropShipOrder[]|null
	 */
	protected function getOrders(array $ids): ?array
	{
		if (self::$allOrders === null) {
			self::$allOrders = $this->ordersExported->getByIds($ids);
		}

		return self::$allOrders;
	}

	public function ordersGridAttached(ControlEvent $event): void
	{
		/** @var OrdersGrid $control */
		$control = $event->control;

		$control->addExportCallback('multihubAllegroLabel', 'multihubDropShip.ordersGrid.multihubAllegroLabel', function(array $ids) use ($control) {
			$allegroIds = [];
			foreach ($ids as $id) {
				$order = $this->orders->getFullOrder((int) $id);

				if ($order && Arrays::contains(['allegro', 'allegroPickup'], $order->getSpeditionIdent())) {
					$allegroIds[] = (int) $id;
				}
			}

			$orders = $allegroIds === [] ? [] : $this->ordersExported->getExported($allegroIds);
			if ($orders === []) {
				$control->presenter->flashMessageSuccess($this->translator->translate('multihubDropShip.export.noOrdersPrint'));
				$control->presenter->redrawControl('flashes');

				return;
			}

			set_time_limit(120);
			foreach ($orders as $order) {
				$client = $this->clients->getClient($order->client->getId());
				if (!$client) {
					continue;
				}

				$result = $this->syncShipment->getLabel($client, $order->dropShipId);

				if ($result) {
					FileSystem::createDir(TMP_DIR . '/dropShipLabels');
					$file = TMP_DIR . '/dropShipLabels/stitek_' . $order->order->getId() . '.pdf';

					if (file_put_contents($file, base64_decode($result))) {
						$arr                                         = [
							'name' => 'Allegro labels',
							'url'  => $control->presenter->link(':Core:Admin:FileResponse:', base64_encode(serialize([
								'file'     => $file,
								'filename' => basename($file),
							]))),
						];
						$control->presenter->payload->fileRequests[] = $arr;
					}
				}
			}

			$control->presenter->flashMessageSuccess($this->translator->translate('mojedpd.export.labelsSuccess'));
			$control->presenter->redrawControl('flashes');
		});
	}
}
